/**
 * OW2 Specifications
 * Copyright (C) 2010 Bull S.A.S.
 * Contact: easybeans@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: ClassTransformer.java 5417 2010-03-24 09:04:14Z benoitf $
 * --------------------------------------------------------------------------
 */

package javax.persistence.spi;

import java.lang.instrument.IllegalClassFormatException;
import java.security.ProtectionDomain;

/**
 * A persistence provider supplies an instance of this interface to the
 * PersistenceUnitInfo.addTransformer method.<br>
 * The supplied transformer instance will get called to transform entity class
 * files when they are loaded or redefined. The transformation occurs before the
 * class is defined by the JVM.
 * @see <a href="http://jcp.org/en/jsr/detail?id=317">JPA 2.0 specification</a>
 * @author Florent Benoit
 * @since JPA 1.0 version.
 */

public interface ClassTransformer {
    /**
     * Invoked when a class is being loaded or redefined. The implementation of
     * this method may transform the supplied class file and return a new
     * replacement class file.
     * @param loader The defining loader of the class to be transformed, may be
     *        null if the bootstrap loader
     * @param className The name of the class in the internal form of fully
     *        qualified class and interface names
     * @param classBeingRedefined If this is a redefine, the class being
     *        redefined, otherwise null
     * @param protectionDomain The protection domain of the class being defined
     *        or redefined
     * @param classfileBuffer The input byte buffer in class file format - must
     *        not be modified
     * @return A well-formed class file buffer (the result of the transform), or
     *         null if no transform is performed
     * @throws IllegalClassFormatException If the input does not represent a
     *         well-formed class file
     */
    byte[] transform(ClassLoader loader, String className, Class<?> classBeingRedefined, ProtectionDomain protectionDomain,
            byte[] classfileBuffer) throws IllegalClassFormatException;
}
