/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2009 EADS DEFENCE AND SECURITY SYSTEMS
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.content;

import java.io.File;
import java.net.URI;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.logging.LogFactory;
import org.ow2.weblab.core.extended.exception.WebLabCheckedException;
import org.ow2.weblab.core.extended.exception.WebLabUncheckedException;
import org.ow2.weblab.core.extended.ontologies.WebLab;
import org.ow2.weblab.core.extended.properties.PropertiesLoader;
import org.ow2.weblab.core.extended.uri.WebLabRI;
import org.ow2.weblab.core.model.Resource;
import org.ow2.weblab.core.model.processing.WProcessingAnnotator;
import org.ow2.weblab.rdf.Value;

/**
 */
public abstract class FolderContentManager {

	protected String propertyKey = "folderpath";

	protected String propertyFileName = "content.properties";

	protected int bufferSize = 10000;

	protected File folder;

	/**
	 * The implementation of {@link ResourceHelper} to be used.
	 */
	public static final String SIMPLE_RESOURCE_RDF_HELPER = "org.weblab_project.core.helper.impl.JenaSingleResourceHelper";

	/**
	 * Constructor
	 * 
	 * @param folderPath
	 *            The path to the content folder.
	 * @throws WebLabUncheckedException
	 */
	protected FolderContentManager(final String folderPath)
			throws WebLabUncheckedException {
		super();
		this.folder = new File(folderPath);
		if (!FolderContentManager.checkFolder(this.folder)) {
			throw new WebLabUncheckedException("Folder '"
					+ this.folder.getAbsolutePath()
					+ "' is not a valid folder path");
		}
		LogFactory.getLog(this.getClass()).debug(
				"Content provider uses folder : "
						+ this.folder.getAbsolutePath());
	}

	/**
	 * Checks if a <code>folder</code> is right configured.
	 * 
	 * @param folder
	 *            The file to check existence.
	 * @return false if <code>folder</code> is not configured.
	 */
	protected static boolean checkFolder(final File folder) {
		if (folder.exists()) {
			if (!folder.isDirectory()) {
				return false;
			}
			return true;
		}
		return folder.mkdirs();
	}

	/**
	 * Uses to get a file from a WebLabRI from a content manager.
	 * 
	 * @param uri
	 *            a valid WLRI.
	 * @return the file corresponding to the WLRI.
	 */
	protected File getFileFromWLRi(final WebLabRI uri) {
		LogFactory.getLog(this.getClass()).debug(
				"File path : " + this.folder.getAbsolutePath() + "/"
						+ uri.toString().hashCode());
		final File newFile = new File(this.folder.getAbsolutePath() + "/"
				+ uri.toString().hashCode());
		if (!newFile.getParentFile().exists()
				&& !newFile.getParentFile().mkdirs()) {
			LogFactory.getLog(this.getClass()).warn(
					"Unable to create file. It may throw exception later.");
		}
		return newFile;
	}

	/**
	 * Returns the native file corresponding to the WLRI.
	 * 
	 * @param uri
	 *            WLRI which identify the native file.
	 * @return a native file.
	 */
	public File getFileFromWLRi(final String uri) {
		return this.getFileFromWLRi(new WebLabRI(uri));
	}

	/**
	 * Extract the {@link WebLab#HAS_NATIVE_CONTENT} object of <code>res</code>
	 * to get the file in the <code>FolderContentManager</code>.
	 * 
	 * @param res
	 *            The resource to extract native content.
	 * @return The file representing the native content of the resource.
	 * @throws WebLabCheckedException
	 */
	public File getNativeFileFromResource(final Resource res)
			throws WebLabCheckedException {
		WProcessingAnnotator wpa = new WProcessingAnnotator(res);
		return this.getFileFromResourceAndPredicate(res, wpa
				.readNativeContent(), "");
	}

	/**
	 * Extract the {@link WebLab#HAS_NORMALISED_CONTENT} object of
	 * <code>res</code> to get the file in the <code>FolderContentManager</code>
	 * .
	 * 
	 * @param res
	 *            The resource to extract normalised content.
	 * @return The file representing the normalised content of the resource.
	 * @throws WebLabCheckedException
	 */
	public File getNormalisedFileFromResource(final Resource res)
			throws WebLabCheckedException {
		WProcessingAnnotator wpa = new WProcessingAnnotator(res);
		return this.getFileFromResourceAndPredicate(res, wpa
				.readNormalisedContent(), WebLab.HAS_NORMALISED_CONTENT);
	}

	/**
	 * Extract the <code>pred</code> object of <code>res</code> to get the file
	 * in the <code>FolderContentManager</code>.
	 * 
	 * @param res
	 *            The resource to extract content.
	 * @param pred
	 *            The predicate used to find content URI in res annotations.
	 * @return The file representing the content of the resource.
	 * @throws WebLabCheckedException
	 */
	protected File getFileFromResourceAndPredicate(final Resource res,
			final Value<URI> values, final String pred)
			throws WebLabCheckedException {
		File file;
		if (values == null || values.size() <= 0) {
			throw new WebLabCheckedException("No statement having '" + pred
					+ "' as predicate found on resource '" + res.getUri()
					+ "'.");
		} else if (values.size() == 1) {
			try {
				file = this.getFileFromWLRi(values.toString());
			} catch (final Exception e) {
				throw new WebLabCheckedException(
						"Unable to retrieve file from predicate '" + pred
								+ "' and resource '" + res.getUri() + "'.", e);
			}
		} else {
			try {
				file = this.getFileFromWLRi(values.toString());
			} catch (final Exception e) {
				throw new WebLabCheckedException("Unable to retrieve file "
						+ "from predicate '" + pred + "' and resource '"
						+ res.getUri() + "'.", e);
			}
			LogFactory.getLog(this.getClass()).warn(
					"Two statements having '" + pred + "' as predicate "
							+ "found on resource '" + res.getUri()
							+ "'. The fist one was used.");
			LogFactory.getLog(this.getClass()).debug(values.toString());
		}
		if (!file.exists() || !file.isFile() || !file.canRead()) {
			throw new WebLabCheckedException("Unable to retrieve file "
					+ "from predicate '" + pred + "' and resource '"
					+ res.getUri() + "'; File '" + file.getPath()
					+ "' does not exist, is not a file or is not accessible.");
		}
		return file;
	}

	/**
	 * Uses this to automatically get a folder path. Load it from a property
	 * file.
	 * 
	 * @param propertyPath
	 *            path to the property file.
	 * @param propertyValue
	 *            name of the key in the property file.
	 * @param defaultValue
	 *            value returned if unable to get one.
	 * @return the path to the file repository folder.
	 */
	protected static String getFolderValue(final String propertyPath,
			final String propertyValue, final String defaultValue) {
		final String path;
		Map<String, String> map;
		try {
			map = PropertiesLoader.loadProperties(propertyPath);
		} catch (WebLabUncheckedException wlue) {
			map = new HashMap<String, String>(0);
		}
		if (map.containsKey(propertyValue)) {
			path = map.get(propertyValue);
		} else {
			LogFactory.getLog(FolderContentManager.class).warn(
					"Unable to load '" + propertyValue + "' from file '"
							+ propertyPath + "'.");
			path = defaultValue;
		}
		LogFactory.getLog(FolderContentManager.class).debug(
				"Loaded : '" + path + "' as content folder.");
		return path;
	}

	/**
	 * @return The folder.
	 */
	public File getFolder() {
		return this.folder;
	}

	/**
	 * @param folder
	 *            The folder to set.
	 */
	public void setFolder(final File folder) {
		this.folder = folder;
	}

}
