package org.ow2.weblab.content;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.ow2.weblab.core.extended.exception.WebLabCheckedException;
import org.ow2.weblab.core.extended.exception.WebLabUncheckedException;
import org.ow2.weblab.core.extended.properties.PropertiesLoader;
import org.ow2.weblab.core.model.Resource;
import org.ow2.weblab.core.model.processing.WProcessingAnnotator;
import org.ow2.weblab.rdf.Value;

public abstract class ContentManager {
	public static final String CONTENT_MANAGER_PROPERTIES_FILE = "contentManager.properties";
	public static final String CONTENT_MANAGER_IMPLEMENTATION = "content.manager.implementation";
	public static int BUFFER_SIZE = 1024;

	protected static Log logger = LogFactory.getLog(ContentManager.class);
	protected static ContentManager instance;
	private static Boolean loaded = false;

	/**
	 * Abstract method to implement : the class is only expected to save the
	 * content and return the URI of the save content.
	 * 
	 * @param content
	 * @param res
	 * @return
	 * @throws WebLabCheckedException
	 */
	protected abstract URI saveContent(final InputStream content, final Resource res) throws WebLabCheckedException;

	/**
	 * Abstract method to implement : the class is only expected to save the
	 * content and return the URI of the save content.
	 * 
	 * @param res
	 * @param destUri
	 * @return
	 * @throws WebLabCheckedException
	 */
	protected abstract File readContent(final Resource res, final URI destUri) throws WebLabCheckedException;

	public URI saveNativeContent(final InputStream content, final Resource res) throws WebLabCheckedException {
		URI destURI = saveContent(content, res);

		WProcessingAnnotator wpa = new WProcessingAnnotator(res);
		wpa.writeNativeContent(destURI);
		return destURI;
	}

	public URI saveNormalisedContent(final InputStream content, final Resource res) throws WebLabCheckedException {
		URI destURI = saveContent(content, res);
		WProcessingAnnotator wpa = new WProcessingAnnotator(res);
		wpa.writeNormalisedContent(destURI);
		return destURI;
	}

	public File readNativeContent(final Resource res) throws WebLabCheckedException {
		WProcessingAnnotator wpa = new WProcessingAnnotator(res);
		Value<URI> values = wpa.readNativeContent();
		if (values == null || values.size() == 0)
			throw new WebLabCheckedException("There is no native content defined on this resource [" + res.getUri() + "]");
		if (values.size() > 1)
			throw new WebLabCheckedException("There is multiple native content defined on this resource [" + res.getUri() + "]:" + values);

		URI uri = values.getValues().get(0);
		return readContent(res, uri);
	}

	public File readNormalisedContent(final Resource res) throws WebLabCheckedException {
		WProcessingAnnotator wpa = new WProcessingAnnotator(res);
		Value<URI> values = wpa.readNormalisedContent();
		if (values == null || values.size() == 0)
			throw new WebLabCheckedException("There is no native content defined on this resource [" + res.getUri() + "]");
		if (values.size() > 1)
			throw new WebLabCheckedException("There is multiple native content defined on this resource [" + res.getUri() + "]:" + values);

		URI uri = values.getValues().get(0);
		return readContent(res, uri);
	}

	protected static String getImplementationName() {
		if (loaded)
			return instance.getClass().getName();
		return ContentManager.getPropertyValue(CONTENT_MANAGER_IMPLEMENTATION, "org.ow2.weblab.content.WebDAVContentManager");
	}

	protected static ContentManager getInstance(String implClass) {
		try {
			synchronized (loaded) {
				if (loaded == false || instance == null || instance.getClass().getName().compareTo(implClass) != 0) {
					Class theClass = Class.forName(implClass);
					instance = (ContentManager) theClass.newInstance();
					loaded = true;
				}
			}
			return instance;
		} catch (ClassNotFoundException e) {
			throw new WebLabUncheckedException("Cannot found the class [" + implClass + "] to instanciate the content manager", e);
		} catch (InstantiationException e) {
			throw new WebLabUncheckedException("Cannot instanciate [" + implClass + "] as content manager");
		} catch (IllegalAccessException e) {
			throw new WebLabUncheckedException("Error while constructing a [" + implClass + "] as content manager");
		}
	}

	public static ContentManager getInstance() {
		return getInstance(getImplementationName());
	}

	/**
	 * Uses this to automatically get a folder path. Load it from a property
	 * file.
	 * 
	 * @param propertyPath
	 *            path to the property file.
	 * @param propertyValue
	 *            name of the key in the property file.
	 * @param defaultValue
	 *            value returned if unable to get one.
	 * @return the path to the file repository folder.
	 */
	protected static String getPropertyValue(final String propertyValue, final String defaultValue) {
		final String value;
		Map<String, String> map;
		try {
			map = PropertiesLoader.loadProperties(CONTENT_MANAGER_PROPERTIES_FILE);
		} catch (WebLabUncheckedException wlue) {
			map = new HashMap<String, String>(0);
		}
		if (map.containsKey(propertyValue)) {
			value = map.get(propertyValue);
		} else {
			logger.warn("Unable to load '" + propertyValue + "' from file '" + CONTENT_MANAGER_PROPERTIES_FILE + "'.");
			value = defaultValue;
		}
		logger.debug("Loaded : '" + value + "' as [" + propertyValue + "].");
		return value;
	}

	/**
	 * Read input from input stream and write it to output stream until there is
	 * no more input from input stream.
	 * 
	 * @param is
	 *            input stream the input stream to read from.
	 * @param os
	 *            output stream the output stream to write to.
	 * @param buf
	 *            the byte array to use as a buffer
	 */
	protected static void writeStream(InputStream is, OutputStream os) throws IOException {
		byte[] buf = new byte[BUFFER_SIZE];
		int numRead;
		while ((numRead = is.read(buf)) >= 0) {
			os.write(buf, 0, numRead);
		}
		is.close();
		os.close();
	}

	/**
	 * @param file
	 *            Input file
	 * @param newFile
	 *            Output file
	 * @throws WebLabCheckedException
	 *             If an IOException occurs.
	 */
	protected static void copyFile(final File file, final File newFile) throws WebLabCheckedException {
		byte[] tab = new byte[BUFFER_SIZE];
		try {
			BufferedInputStream bis = new BufferedInputStream(new FileInputStream(file));
			try {
				BufferedOutputStream bos = new BufferedOutputStream(new FileOutputStream(newFile, true));
				try {
					int readed = bis.read(tab);
					while (readed != -1) {
						bos.write(tab, 0, readed);
						readed = bis.read(tab);
					}
				} finally {
					try {
						bos.close();
					} catch (final IOException ioe) {
						logger.warn("Unable to close stream.", ioe);
					}
				}
			} finally {
				try {
					bis.close();
				} catch (final IOException ioe) {
					logger.warn("Unable to close stream.", ioe);
				}
			}
		} catch (final IOException ioe) {
			throw new WebLabCheckedException("Unable to copy file.", ioe);
		}
	}

}
