/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2011 CASSIDIAN an EADS Company
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.content.api;

import java.io.File;
import java.io.InputStream;
import java.net.URI;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.ow2.weblab.core.extended.exception.WebLabCheckedException;
import org.ow2.weblab.core.extended.exception.WebLabUncheckedException;
import org.ow2.weblab.core.model.Resource;
import org.ow2.weblab.core.model.processing.WProcessingAnnotator;
import org.ow2.weblab.rdf.Value;

public class ContentManager {

	// ////////////////////////////////
	// static part
	// ////////////////////////////////

	public static final String CONTENT_MANAGER_PROPERTIES_FILE = "contentManager.properties";
	public static final String READER_IMPLEMENTATION = "reader.implementation";
	public static final String READER_DEFAULT_IMPLEMENTATION = "org.ow2.weblab.content.impl.FileContentManager";
	public static final String WRITER_IMPLEMENTATION = "writer.implementation";
	public static final String WRITER_DEFAULT_IMPLEMENTATION = "org.ow2.weblab.content.impl.FileContentManager";

	protected static Boolean loaded = Boolean.FALSE;
	protected static ContentManager instance;

	protected static ContentWriter writer;
	protected static ContentReader reader;

	/**
	 * Default starting method to get the ContentManager instance.
	 * 
	 * @return the ContentManager instance
	 */
	public static ContentManager getInstance() {
		try {
			synchronized (ContentManager.loaded) {
				if ((!ContentManager.loaded.booleanValue()) || (ContentManager.instance == null)) {
					ContentManager.instance = new ContentManager();
					ContentManager.instance.loadReaderImpl();
					ContentManager.instance.loadWriterImpl();
					ContentManager.loaded = Boolean.TRUE;
				}
			}
			return ContentManager.instance;
		} catch (final ClassNotFoundException cnfe) {
			throw new WebLabUncheckedException("Cannot found one of the implementation.", cnfe);
		} catch (final InstantiationException ie) {
			throw new WebLabUncheckedException("Cannot instanciate one of the implementation.", ie);
		} catch (final IllegalAccessException iae) {
			throw new WebLabUncheckedException("Error while constructing one of the implementation.", iae);
		}
	}

	// ////////////////////////////////
	// instance part
	// ////////////////////////////////
	protected final Log logger;

	/**
	 * The default constructor that initialises the logger.
	 */
	private ContentManager() {
		this.logger = LogFactory.getLog(this.getClass());
	}

	private void loadWriterImpl() throws InstantiationException, IllegalAccessException, ClassNotFoundException {
		String writerImplName = ContentUtil.getPropertyValue(CONTENT_MANAGER_PROPERTIES_FILE, WRITER_IMPLEMENTATION, WRITER_DEFAULT_IMPLEMENTATION);

		final Class<?> theClass = Class.forName(writerImplName);
		ContentManager.writer = (ContentWriter) theClass.newInstance();
		logger.info(writerImplName + " loaded as content writer.");
	}

	private void loadReaderImpl() throws InstantiationException, IllegalAccessException, ClassNotFoundException {
		String readerImplName = ContentUtil.getPropertyValue(CONTENT_MANAGER_PROPERTIES_FILE, READER_IMPLEMENTATION, READER_DEFAULT_IMPLEMENTATION);

		final Class<?> theClass = Class.forName(readerImplName);
		ContentManager.reader = (ContentReader) theClass.newInstance();
		logger.info(readerImplName + " loaded as content reader.");
	}

	/**
	 * Write the input stream to the content manager and annotate the WebLab
	 * resource to link it to the native content
	 * 
	 * @param res
	 *            standard WebLab Resource
	 * @param content
	 *            an InputStream which could be consumed to get the raw content
	 * @return the URI of the content (for information)
	 * @throws WebLabCheckedException
	 */
	public URI writeNativeContent(final InputStream content, final Resource res) throws WebLabCheckedException {
		final URI destURI = writer.writeContent(content);

		final WProcessingAnnotator wpa = new WProcessingAnnotator(res);
		wpa.writeNativeContent(destURI);
		return destURI;
	}

	/**
	 * Write the input stream to the content manager and annotate the WebLab
	 * resource to link it to the normalised content
	 * 
	 * @param res
	 *            standard WebLab Resource
	 * @param content
	 *            an InputStream which could be consumed to get the raw content
	 * @return the URI of the content (for information)
	 * @throws WebLabCheckedException
	 */
	public URI writeNormalisedContent(final InputStream content, final Resource res) throws WebLabCheckedException {
		final URI destURI = writer.writeContent(content);
		final WProcessingAnnotator wpa = new WProcessingAnnotator(res);
		wpa.writeNormalisedContent(destURI);
		return destURI;
	}

	/**
	 * Read annotation on the input resource and get access to its native
	 * content through a JAVA file. Only read access are guaranted on the
	 * returned File
	 * 
	 * @param a
	 *            standard WebLab Resource
	 * @return a java File object with read rights
	 * @throws WebLabCheckedException
	 */
	public File readNativeContent(final Resource res) throws WebLabCheckedException {
		final WProcessingAnnotator wpa = new WProcessingAnnotator(res);
		final Value<URI> values = wpa.readNativeContent();
		if ((values == null) || (values.size() == 0)) {
			throw new WebLabCheckedException("There is no native content defined on this resource [" + res.getUri() + "]");
		}
		if (values.size() > 1) {
			throw new WebLabCheckedException("There is multiple native content defined on this resource [" + res.getUri() + "]:" + values);
		}

		final URI uri = values.getValues().get(0);
		return reader.readContent(uri);
	}

	/**
	 * Read annotation on the input resource and get access to its normalised
	 * content through a JAVA file. Only read access are guaranted on the
	 * returned File
	 * 
	 * @param a
	 *            standard WebLab Resource
	 * @return a java File object with read rights
	 * @throws WebLabCheckedException
	 */
	public File readNormalisedContent(final Resource res) throws WebLabCheckedException {
		final WProcessingAnnotator wpa = new WProcessingAnnotator(res);
		final Value<URI> values = wpa.readNormalisedContent();
		if ((values == null) || (values.size() == 0)) {
			throw new WebLabCheckedException("There is no native content defined on this resource [" + res.getUri() + "]");
		}
		if (values.size() > 1) {
			throw new WebLabCheckedException("There is multiple native content defined on this resource [" + res.getUri() + "]:" + values);
		}

		final URI uri = values.getValues().get(0);
		return reader.readContent(uri);
	}

}
