package org.ow2.weblab.content.api;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.ow2.weblab.core.extended.exception.WebLabCheckedException;
import org.ow2.weblab.core.extended.exception.WebLabUncheckedException;
import org.ow2.weblab.core.extended.properties.PropertiesLoader;

public class ContentUtil {

	private static Log logger = LogFactory.getLog(ContentUtil.class);
	private static int BUFFER_SIZE = 1024;
	
	/**
	 * Uses this to automatically get a folder path. Load it from a property
	 * file.
	 * 
	 * @param propertyPath
	 *            path to the property file.
	 * @param propertyValue
	 *            name of the key in the property file.
	 * @param defaultValue
	 *            value returned if unable to get one.
	 * @return the path to the file repository folder.
	 */
	public static String getPropertyValue(final String propertyPath, final String propertyValue, final String defaultValue) {
		final String value;
		Map<String, String> map;
		try {
			map = PropertiesLoader.loadProperties(propertyPath);
		} catch (final WebLabUncheckedException wlue) {
			map = new HashMap<String, String>(0);
		}
		if (map.containsKey(propertyValue)) {
			value = map.get(propertyValue);
		} else {
			logger.warn("Unable to load '" + propertyValue + "' from file '" + propertyPath + "'. Using default value ["+defaultValue+"].");
			value = defaultValue;
		}
		logger.debug("Loaded : '" + value + "' as [" + propertyValue + "].");
		return value;
	}

	/**
	 * Read input from input stream and write it to output stream until there is
	 * no more input from input stream.
	 * 
	 * @param is
	 *            input stream the input stream to read from.
	 * @param os
	 *            output stream the output stream to write to.
	 * @param buf
	 *            the byte array to use as a buffer
	 */
	public static void writeStream(final InputStream is, final OutputStream os) throws IOException {
		final byte[] buf = new byte[BUFFER_SIZE];
		int numRead;
		while ((numRead = is.read(buf)) >= 0) {
			os.write(buf, 0, numRead);
		}
		is.close();
		os.close();
	}

	/**
	 * @param file
	 *            Input file
	 * @param newFile
	 *            Output file
	 * @throws WebLabCheckedException
	 *             If an IOException occurs.
	 */
	public static void copyFile(final File file, final File newFile) throws WebLabCheckedException {
		final byte[] tab = new byte[BUFFER_SIZE];
		try {
			final BufferedInputStream bis = new BufferedInputStream(new FileInputStream(file));
			try {
				final BufferedOutputStream bos = new BufferedOutputStream(new FileOutputStream(newFile, true));
				try {
					int readed = bis.read(tab);
					while (readed != -1) {
						bos.write(tab, 0, readed);
						readed = bis.read(tab);
					}
				} finally {
					try {
						bos.close();
					} catch (final IOException ioe) {
						logger.warn("Unable to close stream.", ioe);
					}
				}
			} finally {
				try {
					bis.close();
				} catch (final IOException ioe) {
					logger.warn("Unable to close stream.", ioe);
				}
			}
		} catch (final IOException ioe) {
			throw new WebLabCheckedException("Unable to copy file.", ioe);
		}
	}

	public static boolean equals(final File one, final File another) throws WebLabCheckedException {
		if ((one == null) || (another == null)) {
			return false;
		}
		if (one.length() != another.length()) {
			return false;
		}
		if (ContentUtil.SHA1(one).compareTo(ContentUtil.SHA1(another)) != 0) {
			return false;
		}
		return true;
	}

	public static boolean equals(final byte[] one, final byte[] another) throws WebLabCheckedException {
		if ((one == null) || (another == null)) {
			return false;
		}
		if (one.length != another.length) {
			return false;
		}
		if (ContentUtil.SHA1(one).compareTo(ContentUtil.SHA1(another)) != 0) {
			return false;
		}
		return true;
	}

	public static String SHA1(final File in) throws WebLabCheckedException {
		try {
			final MessageDigest md = MessageDigest.getInstance("SHA-1");
			final FileInputStream fis = new FileInputStream(in);
			byte[] sha1hash = new byte[40];
			final byte[] dataBytes = new byte[1024];
			int nread = 0;
			while ((nread = fis.read(dataBytes)) != -1) {
				md.update(dataBytes, 0, nread);
			}
			sha1hash = md.digest();

			final String hash = ContentUtil.convertToHex(sha1hash);

			return hash;
		} catch (final NoSuchAlgorithmException e) {
			throw new WebLabCheckedException("Cannot load SHA1 algorithm.", e);
		} catch (final FileNotFoundException e) {
			throw new WebLabCheckedException("Cannot load file [" + in + "] for SHA1 computation.", e);
		} catch (final IOException e) {
			throw new WebLabCheckedException("Cannot access content of file [" + in + "] for SHA1 computation.", e);
		}
	}

	public static String SHA1(final byte[] dataBytes) throws WebLabCheckedException {
		try {
			final MessageDigest md = MessageDigest.getInstance("SHA-1");
			byte[] sha1hash = new byte[40];
			md.update(dataBytes, 0, dataBytes.length);
			sha1hash = md.digest();
			final String hash = ContentUtil.convertToHex(sha1hash);
			return hash;
		} catch (final NoSuchAlgorithmException e) {
			throw new WebLabCheckedException("Cannot load SHA1 algorithm.", e);
		}
	}

	public static String convertToHex(final byte[] data) {
		final StringBuffer buf = new StringBuffer();
		for (final byte element : data) {
			int halfbyte = (element >>> 4) & 0x0F;
			int two_halfs = 0;
			do {
				if ((0 <= halfbyte) && (halfbyte <= 9)) {
					buf.append((char) ('0' + halfbyte));
				} else {
					buf.append((char) ('a' + (halfbyte - 10)));
				}
				halfbyte = element & 0x0F;
			} while (two_halfs++ < 1);
		}
		return buf.toString();
	}

}
