/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2011 CASSIDIAN an EADS Company
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.content.impl;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URL;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.ow2.weblab.content.api.ContentManager;
import org.ow2.weblab.content.api.ContentReader;
import org.ow2.weblab.content.api.ContentWriter;
import org.ow2.weblab.core.extended.exception.WebLabCheckedException;
import org.ow2.weblab.core.extended.exception.WebLabUncheckedException;



/**
 * Default implementation for both ContentReader and ContentWriter
 */
public class FileContentManager implements ContentReader, ContentWriter {


	public static final String FILE_CONTENT_MANAGER_PROPERTIES_FILE = "fileContentManager.properties";

	public static final String NETWORK_SCHEMES = "http:ftp:https";

	public static final String FILE_SCHEME = "file";


	public static final String FOLDER_CONTENT_PATH = "file.path";


	public static final String FOLDER_CONTENT_DEFAULT_PATH = "data/content";


	public static final String PREFIX = "weblab.";


	public static final String SUFFIX = ".content";


	private String contentFolderPath;


	private File contentFolder;


	private final Log logger;


	public FileContentManager() {
		this.logger = LogFactory.getLog(this.getClass());
		this.setContentFolderPath(ContentManager.getPropertyValue(FileContentManager.FILE_CONTENT_MANAGER_PROPERTIES_FILE,
				FileContentManager.FOLDER_CONTENT_PATH, FileContentManager.FOLDER_CONTENT_DEFAULT_PATH));
	}


	@Override
	public URI writeContent(final InputStream content) throws WebLabCheckedException {
		try {
			final File contentFile = File.createTempFile(FileContentManager.PREFIX, FileContentManager.SUFFIX, this.contentFolder);
			FileUtils.copyInputStreamToFile(content, contentFile);
			return contentFile.toURI();
		} catch (final IOException ioe) {
			throw new WebLabCheckedException("Cannot create file in content folder [" + this.contentFolder + "].", ioe);
		} finally {
			IOUtils.closeQuietly(content);
		}
	}


	@Override
	public URI writeExposedContent(final InputStream content) throws WebLabCheckedException {
		// TODO Make a specific implementation for exposed content
		this.logger.warn("Not yet implemented !!!");
		return this.writeContent(content);
	}


	@Override
	public File readContent(final URI destUri) throws WebLabCheckedException {
		//URI does not have scheme specified
		if (destUri.isAbsolute() == false) {
			throw new WebLabCheckedException("Invalid URI, scheme is null in "+destUri);
		}
		
		final String scheme = destUri.getScheme();
		URI finalURI = destUri;
		final File toRead;
		if (FileContentManager.NETWORK_SCHEMES.contains(scheme)) {
			// Read the stream to a temporary file in the content Folder
			try {
				URL url = destUri.toURL();
				finalURI = writeContent(url.openStream());
				toRead = new File(finalURI);
			} catch (MalformedURLException e) {
				logger.error(e,e);
				throw new WebLabCheckedException("Network resource URL is malformed : "+destUri,e);
			} catch (IOException e) {
				logger.error(e,e);
				throw new WebLabCheckedException("Network resource URL can not be accessed : "+destUri,e);
			}
		}else if (!FileContentManager.FILE_SCHEME.equalsIgnoreCase(scheme)) {
			throw new WebLabCheckedException("Invalid URI scheme [" + scheme + "], only [" + FileContentManager.FILE_SCHEME + "] is valid.");
		} else {
			//URI is fully parseable and thus begins with a leading /
			if (finalURI.isOpaque() == false) {
				toRead = new File(finalURI);
			} else {
				//In the following case, we have to parse the URI by hand
				//When opaque, the URI is always relative
				//remove the scheme from uri and build the correct relative path
				String path_l = finalURI.getPath();
				path_l = finalURI.toASCIIString().replaceFirst(finalURI.getScheme() +":", "");
				
				
				//when leading slash does not appear, we should search in the defined path
				logger.debug("Relative Path used for reading : " + path_l);
				toRead = new File(contentFolderPath, path_l);
			}
		}
		
		if (!toRead.exists()) {
			throw new WebLabCheckedException("File [" + toRead.getPath() + "] does not exists.");
		}
		//Should be tested after existence. In the opposite, we miss the existence problem
		//as isFile => exists AND normal file
		if (!toRead.isFile()) {
			throw new WebLabCheckedException("File [" + toRead.getPath() + "] is not a file.");
		}
		if (!toRead.canRead()) {
			throw new WebLabCheckedException("File [" + toRead.getPath() + "] is not readable.");
		}

		return toRead;
	}


	public String getContentFolderPath() {
		return this.contentFolderPath;
	}


	protected void setContentFolderPath(final String contentFolderPath) {
		final File folder = new File(contentFolderPath);
		try {
			FileUtils.forceMkdir(folder);
		} catch (final IOException ioe) {
			throw new WebLabUncheckedException("Unable to create Content folder [" + folder + "].", ioe);
		}
		if (!folder.exists()) {
			throw new WebLabUncheckedException("Content folder [" + folder + "] does not exists or is not readable.");
		}
		if (!folder.canWrite()) {
			throw new WebLabUncheckedException("Cannot write in content folder [" + folder + "].");
		}
		if (folder.isFile()) {
			throw new WebLabUncheckedException("Content folder [" + folder + "] is a File instead of a directory.");
		}
		this.contentFolderPath = contentFolderPath;
		this.logger.info(FileContentManager.FOLDER_CONTENT_PATH + "=" + this.contentFolderPath);
		this.contentFolder = folder;
	}

	protected File getContentFolder() {
		return contentFolder;
	}
	
	protected Log getLogger() {
		return logger;
	}
}
