/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2011 CASSIDIAN an EADS Company
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.content.api;

import java.io.File;
import java.io.InputStream;
import java.net.URI;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.ow2.weblab.core.extended.exception.WebLabCheckedException;
import org.ow2.weblab.core.extended.exception.WebLabUncheckedException;
import org.ow2.weblab.core.extended.properties.PropertiesLoader;
import org.ow2.weblab.core.model.Resource;
import org.ow2.weblab.core.model.processing.WProcessingAnnotator;
import org.ow2.weblab.rdf.Value;

public class ContentManager {


	// ////////////////////////////////
	// static part
	// ////////////////////////////////

	public static final String CONTENT_MANAGER_PROPERTIES_FILE = "contentManager.properties";


	public static final String READER_IMPLEMENTATION = "reader.implementation";


	public static final String READER_DEFAULT_IMPLEMENTATION = "org.ow2.weblab.content.impl.SecureFileContentManager";


	public static final String WRITER_IMPLEMENTATION = "writer.implementation";


	public static final String WRITER_DEFAULT_IMPLEMENTATION = "org.ow2.weblab.content.impl.SecureFileContentManager";


	protected static ContentManager instance;


	/**
	 * Default starting method to get the ContentManager instance.
	 * 
	 * @return the ContentManager instance
	 */
	public static synchronized ContentManager getInstance() {
		try {
			if (ContentManager.instance == null) {
				ContentManager.instance = new ContentManager();
			}
			return ContentManager.instance;
		} catch (final ClassNotFoundException cnfe) {
			throw new WebLabUncheckedException("Cannot found one of the implementation.", cnfe);
		} catch (final InstantiationException ie) {
			throw new WebLabUncheckedException("Cannot instanciate one of the implementation.", ie);
		} catch (final IllegalAccessException iae) {
			throw new WebLabUncheckedException("Error while constructing one of the implementation.", iae);
		}
	}


	// ////////////////////////////////
	// instance part
	// ////////////////////////////////
	protected final Log logger;


	final protected ContentWriter writer;


	final protected ContentReader reader;


	/**
	 * The default constructor that initialises the logger.
	 * 
	 * @throws ClassNotFoundException
	 * @throws IllegalAccessException
	 * @throws InstantiationException
	 */
	private ContentManager() throws InstantiationException, IllegalAccessException, ClassNotFoundException {
		this.logger = LogFactory.getLog(this.getClass());
		this.reader = ContentManager.loadReaderImpl();
		this.logger.info(this.reader.getClass().getCanonicalName() + " loaded as content reader.");
		this.writer = ContentManager.loadWriterImpl();
		this.logger.info(this.writer.getClass().getCanonicalName() + " loaded as content writer.");
	}


	private static ContentWriter loadWriterImpl() throws InstantiationException, IllegalAccessException, ClassNotFoundException {
		final String writerImplName = ContentManager.getPropertyValue(ContentManager.CONTENT_MANAGER_PROPERTIES_FILE, ContentManager.WRITER_IMPLEMENTATION,
				ContentManager.WRITER_DEFAULT_IMPLEMENTATION);
		final Class<?> theClass = Class.forName(writerImplName);
		return (ContentWriter) theClass.newInstance();
	}


	private static ContentReader loadReaderImpl() throws InstantiationException, IllegalAccessException, ClassNotFoundException {
		final String readerImplName = ContentManager.getPropertyValue(ContentManager.CONTENT_MANAGER_PROPERTIES_FILE, ContentManager.READER_IMPLEMENTATION,
				ContentManager.READER_DEFAULT_IMPLEMENTATION);
		final Class<?> theClass = Class.forName(readerImplName);
		return (ContentReader) theClass.newInstance();
	}


	/**
	 * Uses this to automatically get a folder path. Load it from a property file.
	 * 
	 * @param propertyPath
	 *            path to the property file.
	 * @param propertyKey
	 *            name of the key in the property file.
	 * @param defaultValue
	 *            value returned if unable to get one.
	 * @return the path to the file repository folder.
	 */
	public static String getPropertyValue(final String propertyPath, final String propertyKey, final String defaultValue) {
		final Map<String, String> map;
		try {
			map = PropertiesLoader.loadProperties(propertyPath);
		} catch (final WebLabUncheckedException wlue) {
			/*
			 * It's not clean to catch a runtime exception and provide a default value; but its needed so that users do not have to provide a property file when
			 * using the default implementation.
			 */
			LogFactory.getLog(ContentManager.class).debug(
					"No property file found. Return default value [" + defaultValue + "] for property [" + propertyKey + "].", wlue);
			return defaultValue;
		}
		final String value;
		if (map.containsKey(propertyKey)) {
			value = map.get(propertyKey);
			LogFactory.getLog(ContentManager.class).debug("Using [" + value + "] for property [" + propertyKey + "].");
		} else {
			LogFactory.getLog(ContentManager.class).debug("No value found for property [" + propertyKey + "]. Return default value [" + defaultValue + "].");
			value = defaultValue;
		}
		map.clear();
		return value;
	}


	/**
	 * Write the input stream to the content manager and annotate the WebLab resource to link it to the native content
	 * 
	 * @param res
	 *            standard WebLab Resource
	 * @param content
	 *            an InputStream which could be consumed to get the raw content
	 * @return the URI of the content (for information)
	 * @throws WebLabCheckedException
	 */
	public URI writeNativeContent(final InputStream content, final Resource res) throws WebLabCheckedException {
		final URI destURI = this.writer.writeContent(content);
		final WProcessingAnnotator wpa = new WProcessingAnnotator(res);
		wpa.writeNativeContent(destURI);
		return destURI;
	}


	/**
	 * Write the input stream to the content manager and annotate the WebLab resource to link it to the normalised content
	 * 
	 * @param res
	 *            standard WebLab Resource
	 * @param content
	 *            an InputStream which could be consumed to get the raw content
	 * @return the URI of the content (for information)
	 * @throws WebLabCheckedException
	 */
	public URI writeNormalisedContent(final InputStream content, final Resource res) throws WebLabCheckedException {
		final URI destURI = this.writer.writeContent(content);
		final WProcessingAnnotator wpa = new WProcessingAnnotator(res);
		wpa.writeNormalisedContent(destURI);
		return destURI;
	}


	/**
	 * Read annotation on the input resource and get access to its native content through a JAVA file. Only read access are guaranteed on the returned File
	 * 
	 * @param a
	 *            standard WebLab Resource
	 * @return a java File object with read rights
	 * @throws WebLabCheckedException
	 */
	public File readNativeContent(final Resource res) throws WebLabCheckedException {
		final WProcessingAnnotator wpa = new WProcessingAnnotator(res);
		final Value<URI> values = wpa.readNativeContent();
		if ((values == null) || (values.size() == 0)) {
			throw new WebLabCheckedException("There is no native content defined on this resource [" + res.getUri() + "]");
		}
		if (values.size() > 1) {
			throw new WebLabCheckedException("There is multiple native content defined on this resource [" + res.getUri() + "]:" + values);
		}

		final URI uri = values.getValues().get(0);
		return this.reader.readContent(uri);
	}


	/**
	 * Read annotation on the input resource and get access to its normalised content through a JAVA file. Only read access are guaranteed on the returned File
	 * 
	 * @param a
	 *            standard WebLab Resource
	 * @return a java File object with read rights
	 * @throws WebLabCheckedException
	 */
	public File readNormalisedContent(final Resource res) throws WebLabCheckedException {
		final WProcessingAnnotator wpa = new WProcessingAnnotator(res);
		final Value<URI> values = wpa.readNormalisedContent();
		if ((values == null) || (!values.hasValue())) {
			throw new WebLabCheckedException("There is no normalised content defined on this resource [" + res.getUri() + "]");
		}
		if (values.size() > 1) {
			throw new WebLabCheckedException("There is multiple normalised content defined on this resource [" + res.getUri() + "]:" + values);
		}

		final URI uri = values.firstTypedValue();
		return this.reader.readContent(uri);
	}


	/**
	 * Accessor to the internally used content reader.
	 * Reserved to advanced user, those who need to read other content than WebLab normalised and native content.
	 * 
	 * @return the internally used content writer.
	 */
	public ContentReader getReader() {
		return this.reader;
	}


	/**
	 * Accessor to the internally used content writer.
	 * Reserved to advanced user, those who need to write other content than WebLab normalised and native content.
	 * 
	 * @return the internally used content writer.
	 */
	public ContentWriter getWriter() {
		return this.writer;
	}

}
