/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2009 EADS DEFENCE AND SECURITY SYSTEMS
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.content;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.httpclient.Credentials;
import org.apache.commons.httpclient.HostConfiguration;
import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpConnectionManager;
import org.apache.commons.httpclient.HttpException;
import org.apache.commons.httpclient.MultiThreadedHttpConnectionManager;
import org.apache.commons.httpclient.UsernamePasswordCredentials;
import org.apache.commons.httpclient.auth.AuthScope;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.httpclient.methods.InputStreamRequestEntity;
import org.apache.commons.httpclient.params.HttpConnectionManagerParams;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.jackrabbit.webdav.client.methods.PutMethod;
import org.ow2.weblab.core.extended.exception.WebLabCheckedException;
import org.ow2.weblab.core.extended.exception.WebLabUncheckedException;
import org.ow2.weblab.core.extended.ontologies.WebLab;
import org.ow2.weblab.core.extended.properties.PropertiesLoader;
import org.ow2.weblab.core.extended.uri.WebLabRI;
import org.ow2.weblab.core.model.Resource;
import org.ow2.weblab.core.model.processing.WProcessingAnnotator;
import org.ow2.weblab.rdf.Value;

/***
 * 
 * @date
 * @deprecated
 */
public class FolderContentManager {
	public static final String CONTENT_PROPERTIES = "contentManager.properties";
	public static final String CONTENT_PATH = "content.path";
	public static final String WEBDAV_HOST = "webdav.host";
	public static final String WEBDAV_PASSWORD = "webdav.password";
	public static final String WEBDAV_USERNAME = "webdav.username";

	public static int BUFFER_SIZE = 1024;

	public static Log logger = LogFactory.getLog(FolderContentManager.class);

	protected String propertyKey = "folderpath";

	protected String propertyFileName = "content.properties";

	protected File folder;

	protected String webdavHost;
	protected String webdavUserName;
	protected String webdavPassword;
	protected HttpClient client;
	/**
	 * The implementation of {@link ResourceHelper} to be used.
	 */
	public static final String SIMPLE_RESOURCE_RDF_HELPER = "org.weblab_project.core.helper.impl.JenaSingleResourceHelper";

	protected FolderContentManager() throws WebLabUncheckedException {
		super();

		this.folder = new File(FolderContentManager.getPropertyValue(CONTENT_PROPERTIES, CONTENT_PATH, ""));
		this.webdavHost = FolderContentManager.getPropertyValue(CONTENT_PROPERTIES, WEBDAV_HOST, "");
		if (!webdavHost.endsWith("/"))
			webdavHost += '/';
		this.webdavUserName = FolderContentManager.getPropertyValue(CONTENT_PROPERTIES, WEBDAV_USERNAME, "");
		this.webdavPassword = FolderContentManager.getPropertyValue(CONTENT_PROPERTIES, WEBDAV_PASSWORD, "");

		if (webdavHost.length() > 1)
			logger.debug("Webdav is configured for: " + webdavUserName + "@" + webdavHost);
	}

	/**
	 * Constructor
	 * 
	 * @param folderPath
	 *            The path to the content folder.
	 * @throws WebLabUncheckedException
	 */
	protected FolderContentManager(final String folderPath) throws WebLabUncheckedException {
		this();
		this.folder = new File(folderPath);
		if (!FolderContentManager.checkFolder(this.folder))
			throw new WebLabUncheckedException("Folder '" + this.folder.getAbsolutePath() + "' is not a valid folder path");

		logger.debug("Content provider uses folder : " + this.folder.getAbsolutePath());

	}

	private void initWebDAVClient() {
		HostConfiguration hostConfig = new HostConfiguration();
		hostConfig.setHost(webdavHost);

		HttpConnectionManager connectionManager = new MultiThreadedHttpConnectionManager();
		HttpConnectionManagerParams params = new HttpConnectionManagerParams();
		int maxHostConnections = 20;

		params.setMaxConnectionsPerHost(hostConfig, maxHostConnections);
		connectionManager.setParams(params);

		client = new HttpClient(connectionManager);
		Credentials creds = new UsernamePasswordCredentials("userId", "pw");
		client.getState().setCredentials(AuthScope.ANY, creds);
		client.setHostConfiguration(hostConfig);
	}

	/**
	 * Checks if a <code>folder</code> is right configured.
	 * 
	 * @param folder
	 *            The file to check existence.
	 * @return false if <code>folder</code> is not configured.
	 */
	protected static boolean checkFolder(final File folder) {
		if (folder.exists()) {
			if (!folder.isDirectory()) {
				return false;
			}
			return true;
		}
		return folder.mkdirs();
	}

	public void saveNativeContent(final File content, final Resource res) throws WebLabCheckedException {
		saveNativeContentOnWebDAV(content, res);
	}
	
	public void saveNormalisedContent(final File content, final Resource res) throws WebLabCheckedException {
		saveNativeContentOnWebDAV(content, res);
	}
	
	/**
	 * Add new file to the WebDAV content manager
	 * 
	 * @param content
	 * @param res
	 * @throws WebLabCheckedException
	 */
	public void saveNativeContentOnWebDAV(final File content, final Resource res) throws WebLabCheckedException {
		logger.info("Saving content on WebDAV host [" + webdavHost + "]");

		if (client == null)
			initWebDAVClient();

		try {
			URI contentURI = new URI(webdavHost + content.toString().hashCode());
			PutMethod put = new PutMethod("http://localhost:8080/repository/default/test.pdf");
			put.setRequestEntity(new InputStreamRequestEntity(new FileInputStream(content)));
			client.executeMethod(put);
			WProcessingAnnotator annot = new WProcessingAnnotator(res);
			annot.writeNativeContent(contentURI);
		} catch (URISyntaxException e) {

		} catch (FileNotFoundException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		} catch (HttpException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		} catch (IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
	}

	/**
	 * Returns the native file corresponding to the WLRI.
	 * 
	 * @param uri
	 *            WLRI which identify the native file.
	 * @return a native file.
	 */
	public File getFileFromWLRi(final String uri) {
		return this.getFileFromWLRi(new WebLabRI(uri));
	}

	/**
	 * Uses to get a file from a WebLabRI from a content manager.
	 * 
	 * @param uri
	 *            a valid WLRI.
	 * @return the file corresponding to the WLRI.
	 */
	protected File getFileFromWLRi(final WebLabRI uri) {
		logger.debug("File path : " + this.folder.getAbsolutePath() + "/" + uri.toString().hashCode());
		final File newFile = new File(this.folder.getAbsolutePath() + "/" + uri.toString().hashCode());
		if (!newFile.getParentFile().exists() && !newFile.getParentFile().mkdirs()) {
			logger.warn("Unable to create file. It may throw exception later.");
		}
		return newFile;
	}

	/**
	 * Extract the {@link WebLab#HAS_NATIVE_CONTENT} object of <code>res</code>
	 * to get the file in the <code>FolderContentManager</code>.
	 * 
	 * @param res
	 *            The resource to extract native content.
	 * @return The file representing the native content of the resource.
	 * @throws WebLabCheckedException
	 */
	public File getNativeFileFromResource(final Resource res) throws WebLabCheckedException {
		WProcessingAnnotator wpa = new WProcessingAnnotator(res);
		return this.getFileFromResourceAndPredicate(res, wpa.readNativeContent(), "");
	}

	/**
	 * Extract the {@link WebLab#HAS_NORMALISED_CONTENT} object of
	 * <code>res</code> to get the file in the <code>FolderContentManager</code>
	 * .
	 * 
	 * @param res
	 *            The resource to extract normalised content.
	 * @return The file representing the normalised content of the resource.
	 * @throws WebLabCheckedException
	 */
	public File getNormalisedFileFromResource(final Resource res) throws WebLabCheckedException {
		WProcessingAnnotator wpa = new WProcessingAnnotator(res);
		return this.getFileFromResourceAndPredicate(res, wpa.readNormalisedContent(), WebLab.HAS_NORMALISED_CONTENT);
	}

	/**
	 * Extract the <code>pred</code> object of <code>res</code> to get the file
	 * in the <code>FolderContentManager</code>.
	 * 
	 * @param res
	 *            The resource to extract content.
	 * @param pred
	 *            The predicate used to find content URI in res annotations.
	 * @return The file representing the content of the resource.
	 * @throws WebLabCheckedException
	 */
	protected File getFileFromResourceAndPredicate(final Resource res, final Value<URI> values, final String pred) throws WebLabCheckedException {
		File file = null;
		if (values == null || values.size() <= 0) {
			throw new WebLabCheckedException("No statement having '" + pred + "' as predicate found on resource '" + res.getUri() + "'.");
		} else if (values.size() == 1) {

			// TODO test URI and detect scheme
			URI contentURI = values.firstTypedValue();
			String sheme = contentURI.getScheme();

			switch (URIScheme.valueOf(sheme)) {
			// FIXME the weblab scheme
			case weblab:
				// use old content access model : guess file name from content
				// URI
				file = getLocalFileFromURI(contentURI);
				break;

			case file:
				file = getLocalFile(contentURI);
				break;

			case ftp:
				// get file on FTP server
				throw new WebLabUncheckedException("Access to fTP content is not yet implemented.");
				// break;

			case http:
				// get file from http
				file = getFileFromHttp(contentURI);
				break;
			}

		} else {
			try {
				file = this.getFileFromWLRi(values.toString());
			} catch (final Exception e) {
				throw new WebLabCheckedException("Unable to retrieve file " + "from predicate '" + pred + "' and resource '" + res.getUri() + "'.", e);
			}
			logger.warn("Multiple statements having '" + pred + "' as predicate " + "found on resource '" + res.getUri() + "'. The fist one was used.");
			logger.debug(values.toString());
		}
		if (file != null && (!file.exists() || !file.isFile() || !file.canRead())) {
			throw new WebLabCheckedException("Unable to retrieve file " + "from predicate '" + pred + "' and resource '" + res.getUri() + "'; File '"
					+ file.getPath() + "' does not exist, is not a file or is not accessible.");
		}
		return file;
	}

	private File getFileFromHttp(URI contentURI) {
		if (webdavHost.contains(contentURI.getHost()))
			return getFileFromWebDav(contentURI);

		throw new WebLabUncheckedException("Access to simple HTTP content is not yet implemented.");
		// FIXME need to make simple GET HTTP in case of not webDAV
		// return null;
	}

	private File getFileFromWebDav(URI contentURI) {
		logger.info("Getting content from WebDAV host [" + webdavHost + "]");

		if (client == null)
			initWebDAVClient();

		File out = null;
		OutputStream stream = null;
		try {
			out = File.createTempFile("webDAV", "", folder);
			GetMethod get = new GetMethod(contentURI.toString());
			client.executeMethod(get);
			InputStream inStream = get.getResponseBodyAsStream();
			OutputStream outStream = new BufferedOutputStream(new FileOutputStream(out));
			writeStream(inStream, outStream);
			inStream.close();
			outStream.close();
		} catch (HttpException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		} catch (IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		} finally {
			if (stream != null)
				try {
					stream.close();
				} catch (IOException e) {
					// TODO Auto-generated catch block
					e.printStackTrace();
				}
		}
		return out;
	}

	private File getLocalFile(URI contentURI) throws WebLabCheckedException {
		File file = new File(contentURI);
		if (!file.exists())
			throw new WebLabCheckedException("The content file [" + contentURI + "] does not exists.");
		if (!file.canRead())
			throw new WebLabCheckedException("The content file [" + contentURI + "] is not readable.");
		return file;
	}

	/**
	 * @deprecated
	 * @param contentURI
	 * @return
	 * @throws WebLabCheckedException
	 */
	private File getLocalFileFromURI(URI contentURI) throws WebLabCheckedException {
		try {
			return this.getFileFromWLRi(contentURI.toString());
		} catch (final Exception e) {
			throw new WebLabCheckedException("Unable to retrieve file localtion from content URI [" + contentURI + "].", e);
		}
	}

	/**
	 * Uses this to automatically get a folder path. Load it from a property
	 * file.
	 * 
	 * @param propertyPath
	 *            path to the property file.
	 * @param propertyValue
	 *            name of the key in the property file.
	 * @param defaultValue
	 *            value returned if unable to get one.
	 * @return the path to the file repository folder.
	 */
	protected static String getPropertyValue(final String propertyPath, final String propertyValue, final String defaultValue) {
		final String value;
		Map<String, String> map;
		try {
			map = PropertiesLoader.loadProperties(propertyPath);
		} catch (WebLabUncheckedException wlue) {
			map = new HashMap<String, String>(0);
		}
		if (map.containsKey(propertyValue)) {
			value = map.get(propertyValue);
		} else {
			LogFactory.getLog(FolderContentManager.class).warn("Unable to load '" + propertyValue + "' from file '" + propertyPath + "'.");
			value = defaultValue;
		}
		LogFactory.getLog(FolderContentManager.class).debug("Loaded : '" + value + "' as [" + propertyValue + "].");
		return value;
	}

	/**
	 * @return The folder.
	 */
	public File getFolder() {
		return this.folder;
	}

	/**
	 * @param folder
	 *            The folder to set.
	 */
	public void setFolder(final File folder) {
		this.folder = folder;
	}

	/**
	 * Read input from input stream and write it to output stream until there is
	 * no more input from input stream.
	 * 
	 * @param is
	 *            input stream the input stream to read from.
	 * @param os
	 *            output stream the output stream to write to.
	 * @param buf
	 *            the byte array to use as a buffer
	 */
	public static void writeStream(InputStream is, OutputStream os) throws IOException {
		byte[] buf = new byte[BUFFER_SIZE];
		int numRead;
		while ((numRead = is.read(buf)) >= 0) {
			os.write(buf, 0, numRead);
		}
	}

	/**
	 * @param file
	 *            Input file
	 * @param newFile
	 *            Output file
	 * @throws WebLabCheckedException
	 *             If an IOException occurs.
	 */
	public static void copyFile(final File file, final File newFile) throws WebLabCheckedException {
		byte[] tab = new byte[BUFFER_SIZE];
		try {
			BufferedInputStream bis = new BufferedInputStream(new FileInputStream(file));
			try {
				BufferedOutputStream bos = new BufferedOutputStream(new FileOutputStream(newFile, true));
				try {
					int readed = bis.read(tab);
					while (readed != -1) {
						bos.write(tab, 0, readed);
						readed = bis.read(tab);
					}
				} finally {
					try {
						bos.close();
					} catch (final IOException ioe) {
						logger.warn("Unable to close stream.", ioe);
					}
				}
			} finally {
				try {
					bis.close();
				} catch (final IOException ioe) {
					logger.warn("Unable to close stream.", ioe);
				}
			}
		} catch (final IOException ioe) {
			throw new WebLabCheckedException("Unable to copy file.", ioe);
		}
	}

}
