package org.ow2.weblab.content;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.net.URISyntaxException;

import org.apache.commons.httpclient.Credentials;
import org.apache.commons.httpclient.HostConfiguration;
import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpConnectionManager;
import org.apache.commons.httpclient.HttpException;
import org.apache.commons.httpclient.MultiThreadedHttpConnectionManager;
import org.apache.commons.httpclient.UsernamePasswordCredentials;
import org.apache.commons.httpclient.auth.AuthScope;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.httpclient.methods.InputStreamRequestEntity;
import org.apache.commons.httpclient.params.HttpConnectionManagerParams;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.jackrabbit.webdav.DavException;
import org.apache.jackrabbit.webdav.client.methods.PutMethod;
import org.ow2.weblab.core.extended.exception.WebLabCheckedException;
import org.ow2.weblab.core.extended.exception.WebLabUncheckedException;
import org.ow2.weblab.core.model.Resource;

public class WebDAVContentManager extends ContentManager {

	public static Log logger = LogFactory.getLog(WebDAVContentManager.class);
	public static final String WEBDAV_TMP_FOLDER_PATH = "webdav.tmpFolder";
	public static final String WEBDAV_HOST = "webdav.host";
	public static final String WEBDAV_TIMEOUT = "webdav.timeout";
	public static final String WEBDAV_MAXHOSTCONNECTIONS = "webdav.maxHostConnections";
	public static final String WEBDAV_PASSWORD = "webdav.password";
	public static final String WEBDAV_USERNAME = "webdav.username";

	private String webdavTmpFolder;
	private String webdavHost;
	private int webdavTimeOut;
	private int webdavMaxHostConnections;

	private String webdavUserName;
	private String webdavPassword;

	private File tmpFolder;

	protected WebDAVContentManager() {
		setWebdavTmpFolder(ContentManager.getPropertyValue(WEBDAV_TMP_FOLDER_PATH, "/tmp"));
		logger.info(WEBDAV_TMP_FOLDER_PATH + "=" + this.webdavTmpFolder);

		this.webdavHost = ContentManager.getPropertyValue(WEBDAV_HOST, null);
		if (!webdavHost.endsWith("/"))
			webdavHost += '/';
		logger.info(WEBDAV_HOST + "=" + this.webdavHost);

		this.webdavTimeOut = Integer.parseInt(ContentManager.getPropertyValue(WEBDAV_TIMEOUT, "" + 30000));
		this.webdavMaxHostConnections = Integer.parseInt(ContentManager.getPropertyValue(WEBDAV_MAXHOSTCONNECTIONS, "" + 20));

		this.webdavUserName = ContentManager.getPropertyValue(WEBDAV_PASSWORD, null);
		this.webdavPassword = ContentManager.getPropertyValue(WEBDAV_USERNAME, null);
		logger.info(WEBDAV_USERNAME + "=" + this.webdavUserName + " and " + WEBDAV_PASSWORD + " is set.");
	}

	private HttpClient initWebDAVClient() throws WebLabCheckedException {
		if (webdavTmpFolder == null)
			throw new WebLabCheckedException("Cannot initiate WebDAV host: tmpFolder value is null");

		HttpClient clnt;
		if (webdavHost == null)
			throw new WebLabCheckedException("Cannot initiate WebDAV host: host value is null");

		HostConfiguration hostConfig = new HostConfiguration();
		hostConfig.setHost(webdavHost);

		HttpConnectionManager connectionManager = new MultiThreadedHttpConnectionManager();
		HttpConnectionManagerParams params = new HttpConnectionManagerParams();

		params.setMaxConnectionsPerHost(hostConfig, webdavMaxHostConnections);
		params.setSoTimeout(webdavTimeOut);
		connectionManager.setParams(params);

		clnt = new HttpClient(connectionManager);

		if (webdavUserName == null || webdavPassword == null)
			throw new WebLabCheckedException("Cannot initiate WebDAV host: user and/or password value is null");

		Credentials creds = new UsernamePasswordCredentials(webdavUserName, webdavPassword);
		clnt.getState().setCredentials(AuthScope.ANY, creds);
		clnt.setHostConfiguration(hostConfig);

		return clnt;
	}

	@Override
	protected URI saveContent(InputStream content, Resource res) throws WebLabCheckedException {
		logger.debug("Saving content on WebDAV host [" + webdavHost + "]");
		HttpClient client = null;
		synchronized (this) {
			if (client == null)
				client = initWebDAVClient();
		}

		try {
			URI destURI = new URI(this.webdavHost + res.hashCode());
			PutMethod put = new PutMethod(destURI.toString());
			put.setRequestEntity(new InputStreamRequestEntity(content));
			
			synchronized (this) {
				client.executeMethod(put);
			}

			put.checkSuccess();

			return destURI;
		} catch (FileNotFoundException e) {
			throw new WebLabCheckedException("Cannot save content: file [" + content + "] is not found.", e);
		} catch (HttpException e) {
			throw new WebLabCheckedException("Cannot save content: PUT method failure for resource [" + res.getUri() + "] on WebDAV server [" + webdavHost
					+ "]", e);
		} catch (IOException e) {
			throw new WebLabCheckedException("Cannot save content: I/O error while writing content for resource [" + res.getUri() + "] on WebDAV server ["
					+ webdavHost + "]", e);
		} catch (URISyntaxException e) {
			throw new WebLabCheckedException("Cannot save content: [" + this.webdavHost + res.hashCode() + "] is an invalid URI.", e);
		} catch (DavException e) {
			throw new WebLabCheckedException("WebDAV error while saving [" + this.webdavHost + res.hashCode() + "].", e);
		}
	}

	@Override
	protected File readContent(Resource res, URI destUri) throws WebLabCheckedException {
		if (!destUri.toString().contains(webdavHost))
			throw new WebLabCheckedException("Cannot read content [" + destUri + "]. The host is unknown.");

		logger.debug("Getting content [" + destUri + "] from WebDAV.");

		HttpClient client = null;
		synchronized (this) {
			if (client == null)
				client = initWebDAVClient();
		}

		File out = null;
		InputStream inStream = null;
		OutputStream outStream = null;
		try {
			out = File.createTempFile("webDAV", ".content", tmpFolder);
			GetMethod get = new GetMethod(destUri.toString());
			
			synchronized (this) {
				client.executeMethod(get);
			}

			if(get.getStatusCode()!=200)
				throw new WebLabCheckedException("Cannot read content: GET method failure for content [" + destUri + "] : "+get.getStatusCode()+" '"+get.getStatusText()+"'");

			inStream = get.getResponseBodyAsStream();
			outStream = new BufferedOutputStream(new FileOutputStream(out));
			writeStream(inStream, outStream);

		} catch (HttpException e) {
			throw new WebLabCheckedException("Cannot read content: GET method failure for content [" + destUri + "] on WebDAV server [" + webdavHost + "]", e);
		} catch (IOException e) {
			throw new WebLabCheckedException("Cannot read content: I/O error while copying [" + destUri + "] from WebDAV server [" + webdavHost
					+ "] to local File [" + out + "].", e);
		} finally {
			try {
				if (inStream != null)
					inStream.close();
				if (outStream != null)
					outStream.close();
			} catch (IOException e) {
				throw new WebLabCheckedException("Cannot read content: I/O error while closing stream after reading content on WebDAV.", e);
			}
		}
		return out;
	}

	public String getWebdavTmpFolder() {
		return webdavTmpFolder;
	}

	protected void setWebdavTmpFolder(String webdavTmpFolder) {
		this.webdavTmpFolder = webdavTmpFolder;
		logger.info(WEBDAV_TMP_FOLDER_PATH + "=" + this.webdavTmpFolder);
		this.tmpFolder = new File(this.webdavTmpFolder);
		if (!this.tmpFolder.exists())
			this.tmpFolder.mkdirs();
		if (!this.tmpFolder.exists())
			throw new WebLabUncheckedException("Content folder [" + this.tmpFolder + "] does not exists or is not readable.");
		if (!this.tmpFolder.canWrite())
			throw new WebLabUncheckedException("Cannot write in content folder [" + this.tmpFolder + "].");
		if (this.tmpFolder.isFile())
			throw new WebLabUncheckedException("Content folder [" + this.tmpFolder + "] is a File instead of a directory.");

	}

	public String getWebdavHost() {
		return webdavHost;
	}

	protected void setWebdavHost(String webdavHost) {
		this.webdavHost = webdavHost;
	}

	public int getWebdavTimeOut() {
		return webdavTimeOut;
	}

	protected void setWebdavTimeOut(int webdavTimeOut) {
		this.webdavTimeOut = webdavTimeOut;
	}

	public int getWebdavMaxHostConnections() {
		return webdavMaxHostConnections;
	}

	protected void setWebdavMaxHostConnections(int webdavMaxHostConnections) {
		this.webdavMaxHostConnections = webdavMaxHostConnections;
	}

	public String getWebdavUserName() {
		return webdavUserName;
	}

	protected void setWebdavUserName(String webdavUserName) {
		this.webdavUserName = webdavUserName;
	}

	public String getWebdavPassword() {
		return webdavPassword;
	}

	protected void setWebdavPassword(String webdavPassword) {
		this.webdavPassword = webdavPassword;
	}
}
