/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2009 EADS DEFENCE AND SECURITY SYSTEMS
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.crawler.filter;

import java.io.File;
import java.io.FileFilter;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;

/**
 * Use this class when you need a {@link FileFilter} on file extension (.jpg, .txt, .mpeg, ...) If reject mode, given extensions are not rejected. Else, the
 * extensions are the only accepted.
 * 
 * @author Jérémie Doucy, ymombrun
 */
public class ExtensionFilter implements FileFilter {

	/**
	 * The extensions to test.
	 */
	protected List<String> extensions;

	/**
	 * Whether extensions are the accepted ones or the rejected ones.
	 */
	protected boolean rejectMode = false;

	/**
	 * Create a file filter using extension
	 * 
	 * @param extentions
	 *            The extensions to test.
	 * @param rejectMode
	 *            Whether extensions are the accepted ones or the rejected ones.
	 */
	public ExtensionFilter(final List<String> extentions, final boolean rejectMode) {
		super();
		this.rejectMode = rejectMode;
		this.extensions = new ArrayList<String>();
		if (extentions != null) {
			this.extensions.addAll(extentions);
		}
	}

	/*
	 * (non-Javadoc)
	 * @see java.io.FileFilter#accept(java.io.File)
	 */
	public boolean accept(final File file) {
		return file.isFile() && (this.extensions.contains(ExtensionFilter.getFileExtension(file)) != this.rejectMode);
	}

	/**
	 * @param file
	 *            The file to extract extension.
	 * @return The extension of the file or the empty String if nothing is found.
	 */
	public static String getFileExtension(final File file) {
		String[] splited = file.getName().split("\\.");
		if (splited.length > 0) {
			return splited[splited.length - 1].toLowerCase(Locale.getDefault());
		}
		return "";
	}

}
