/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2012 Cassidian, an EADS company
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.components;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import org.apache.commons.logging.Log;
import org.jdom.Element;
import org.jdom.JDOMException;
import org.jdom.input.SAXBuilder;
import org.jdom.xpath.XPath;
import org.ow2.weblab.core.extended.exception.WebLabCheckedException;
import org.ow2.weblab.core.extended.jaxb.WebLabMarshaller;
import org.ow2.weblab.core.helper.impl.AdvancedSelector;
import org.ow2.weblab.core.helper.impl.IPredicateValuePairs;
import org.ow2.weblab.core.helper.impl.JenaPoKHelper;
import org.ow2.weblab.core.helper.impl.RDFSelectorFactory;
import org.ow2.weblab.core.helper.impl.Statements;
import org.ow2.weblab.core.model.Annotation;
import org.ow2.weblab.core.model.Audio;
import org.ow2.weblab.core.model.ComposedResource;
import org.ow2.weblab.core.model.Document;
import org.ow2.weblab.core.model.Image;
import org.ow2.weblab.core.model.LowLevelDescriptor;
import org.ow2.weblab.core.model.MediaUnit;
import org.ow2.weblab.core.model.PieceOfKnowledge;
import org.ow2.weblab.core.model.Query;
import org.ow2.weblab.core.model.Resource;
import org.ow2.weblab.core.model.ResultSet;
import org.ow2.weblab.core.model.Segment;
import org.ow2.weblab.core.model.Text;
import org.ow2.weblab.core.model.Video;

class WebLabMerger {
	private final Log logger;
	
	public WebLabMerger(Log logger) {
		this.logger = logger;
	}

	Resource merge(Resource initialResource,
			Resource[] resourcesToMerge) {
		List<String> initialURIs = listURIs(initialResource);
		
		List<Resource> resources = new LinkedList<Resource>();
		List<String> newURIs = new LinkedList<String>();
		for (Resource resource : resourcesToMerge) {
			List<String> URIs = listURIs(resource);
			List<String> toChangedURIs = new LinkedList<String>();
			
			for (String string : URIs) {
				if(!initialURIs.contains(string)) {
					if(newURIs.contains(string)) {
						toChangedURIs.add(string);
					}
					else {
						newURIs.add(string);
					}
				}
			}
			
			if(toChangedURIs.isEmpty()) {
				resources.add(resource);
			}
			else {
				resources.add(changeURIs(resource, toChangedURIs));
			}
		}
		
		mergeResource(initialResource, resources);
		return initialResource;
	}
	
	Resource merge(Resource initialResource,
			List<Resource> resourcesToMerge) {
		List<String> initialURIs = listURIs(initialResource);
		
		List<Resource> resources = new LinkedList<Resource>();
		List<String> newURIs = new LinkedList<String>();
		for (Resource resource : resourcesToMerge) {
			List<String> URIs = listURIs(resource);
			List<String> toChangedURIs = new LinkedList<String>();
			
			for (String string : URIs) {
				if(!initialURIs.contains(string)) {
					if(newURIs.contains(string)) {
						toChangedURIs.add(string);
					}
					else {
						newURIs.add(string);
					}
				}
			}
			
			if(toChangedURIs.isEmpty()) {
				resources.add(resource);
			}
			else {
				resources.add(changeURIs(resource, toChangedURIs));
			}
		}
		
		mergeResource(initialResource, resources);
		return initialResource;
	}
	
	private Resource changeURIs(Resource resource, List<String> toChangedURIs) {
		try {
			StringWriter sw = new StringWriter();
			WebLabMarshaller wm = new WebLabMarshaller();
			wm.marshalResource(resource, sw);
			
			String resourceAsString = sw.toString();
			
			for (String string : toChangedURIs) {
				String oldUri = "\"" + string + "\"";
				String newUri = "\"weblab://" + getNewURI() + "\"";
				resourceAsString = resourceAsString.replace(oldUri, newUri);
			}
			
			Resource result = wm.unmarshal(new StringReader(resourceAsString), Document.class);
			return result;
		} catch (WebLabCheckedException e) {
			e.printStackTrace();
		}
		
		return null;
	}

	private String getNewURI() {
		return UUID.randomUUID().toString();
	}

	private List<String> listURIs(Resource resource) {
		try {
			ByteArrayOutputStream out = new ByteArrayOutputStream();
			final WebLabMarshaller wm = new WebLabMarshaller();
			SAXBuilder saxBuilder = new SAXBuilder();
			
			wm.marshalResource(resource, out);
			org.jdom.Document dom = saxBuilder.build(new ByteArrayInputStream(out
					.toByteArray()));
			return listURIs(dom);
		} catch (WebLabCheckedException e) {
			e.printStackTrace();
		} catch (JDOMException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}
		
		return null;
	}
	
	private List<String> listURIs(org.jdom.Document dom) {
		List<String> result = new LinkedList<String>();
		
		try {
			List<?> list = XPath.selectNodes(dom, "//*[@uri]");
			for (Object object : list) {
				if(object instanceof Element) {
					Element e = (Element) object;
					result.add(e.getAttributeValue("uri"));
				}
			}
		} catch (JDOMException e) {
			e.printStackTrace();
		}
		return result;
	}

	private void mergeResource(Resource initialResource,
			List<Resource> resourcesToMerge) {
		
		List<String> doneURIs = new LinkedList<String>();
		for (Annotation initialAnnotation : initialResource.getAnnotation()) {
			List<Resource> annotationsToMerge = new LinkedList<Resource>();
			for (Resource resource : resourcesToMerge) {
				boolean notFound = true;
				for (int i = 0; i < resource.getAnnotation().size() && notFound; i++) {
					if(resource.getAnnotation().get(i).getUri().equals(initialAnnotation.getUri())) {
						notFound = false;
						annotationsToMerge.add(resource.getAnnotation().get(i));
					}
				}
				if(notFound) {
					logger.warn("Annotation remove not handle");
				}
			}
			mergeResource(initialAnnotation, annotationsToMerge);
			doneURIs.add(initialAnnotation.getUri());
		}
		for (Resource resource : resourcesToMerge) {
			for (Annotation annotation : resource.getAnnotation()) {
				if(!doneURIs.contains(annotation.getUri())){
					logger.info("Add Annotation to Resource");
					initialResource.getAnnotation().add(annotation);
				}
			}
		}
		
		for (LowLevelDescriptor initialLowLevelDescriptor : initialResource.getDescriptor()) {
			List<Resource> lowLevelDescriptorsToMerge = new LinkedList<Resource>();
			for (Resource resource : resourcesToMerge) {
				boolean notFound = true;
				for (int i = 0; i < resource.getDescriptor().size() && notFound; i++) {
					if(resource.getDescriptor().get(i).getUri().equals(initialLowLevelDescriptor.getUri())) {
						notFound = false;
						lowLevelDescriptorsToMerge.add(resource.getDescriptor().get(i));
					}
				}
				if(notFound) {
					logger.warn("LowLevelDescriptor remove not handle");
				}
			}
			mergeResource(initialLowLevelDescriptor, lowLevelDescriptorsToMerge);
			doneURIs.add(initialLowLevelDescriptor.getUri());
		}
		for (Resource resource : resourcesToMerge) {
			for (LowLevelDescriptor lowLevelDescriptor : resource.getDescriptor()) {
				if(!doneURIs.contains(lowLevelDescriptor.getUri())){
					logger.info("Add LowLevelDescriptor to Resource");
					initialResource.getDescriptor().add(lowLevelDescriptor);
				}
			}
		}
		
		if(initialResource instanceof MediaUnit) {
			MediaUnit initialMediaUnit = (MediaUnit) initialResource;
			List<MediaUnit> mediaUnitsToMerge = new LinkedList<MediaUnit>();
			for (Resource resource : resourcesToMerge) {
				if(resource instanceof MediaUnit) {
					MediaUnit mediaUnit = (MediaUnit) resource;
					mediaUnitsToMerge.add(mediaUnit);
				}
				else {
					logger.error("Resource is " + resource.getClass() + " and a " + initialResource.getClass() + "was needed");
				}
			}
			mergeMediaUnit(initialMediaUnit, mediaUnitsToMerge);
		}
		else if(initialResource instanceof ComposedResource) {
			ComposedResource initialComposedResource = (ComposedResource) initialResource;
			List<ComposedResource> composedResourcesToMerge = new LinkedList<ComposedResource>();
			for (Resource resource : resourcesToMerge) {
				if(resource instanceof ComposedResource) {
					ComposedResource composedResource = (ComposedResource) resource;
					composedResourcesToMerge.add(composedResource);
				}
				else {
					logger.error("Resource is " + resource.getClass() + " and a " + initialResource.getClass() + "was needed");
				}
			}
			mergeComposedResource(initialComposedResource, composedResourcesToMerge);
		}
		else if(initialResource instanceof LowLevelDescriptor) {
			LowLevelDescriptor initialLowLevelDescriptor = (LowLevelDescriptor) initialResource;
			List<LowLevelDescriptor> lowLevelDescriptorsToMerge = new LinkedList<LowLevelDescriptor>();
			for (Resource resource : resourcesToMerge) {
				if(resource instanceof LowLevelDescriptor) {
					LowLevelDescriptor annotation = (LowLevelDescriptor) resource;
					lowLevelDescriptorsToMerge.add(annotation);
				}
				else {
					logger.error("Resource is " + resource.getClass() + " and a " + initialResource.getClass() + "was needed");
				}
			}
			mergeLowLevelDescriptor(initialLowLevelDescriptor, lowLevelDescriptorsToMerge);
		}
		else if(initialResource instanceof PieceOfKnowledge) {
			PieceOfKnowledge initialPOK = (PieceOfKnowledge) initialResource;
			List<PieceOfKnowledge> poksToMerge = new LinkedList<PieceOfKnowledge>();
			for (Resource resource : resourcesToMerge) {
				if(resource instanceof PieceOfKnowledge) {
					PieceOfKnowledge pok = (PieceOfKnowledge) resource;
					poksToMerge.add(pok);
				}
				else {
					logger.error("Resource is " + resource.getClass() + " and a " + initialResource.getClass() + "was needed");
				}
			}
			mergePieceOfKnowledge(initialPOK, poksToMerge);
		}
		else if(initialResource instanceof Query) {
			Query initialQuery = (Query) initialResource;
			List<Query> querysToMerge = new LinkedList<Query>();
			for (Resource resource : resourcesToMerge) {
				if(resource instanceof Query) {
					Query query = (Query) resource;
					querysToMerge.add(query);
				}
				else {
					logger.error("Resource is " + resource.getClass() + " and a " + initialResource.getClass() + "was needed");
				}
			}
			mergeQuery(initialQuery, querysToMerge);
		}
		else if(initialResource instanceof ResultSet) {
			ResultSet initialResultSet = (ResultSet) initialResource;
			List<ResultSet> resultSetsToMerge = new LinkedList<ResultSet>();
			for (Resource resource : resourcesToMerge) {
				if(resource instanceof ResultSet) {
					ResultSet resultSet = (ResultSet) resource;
					resultSetsToMerge.add(resultSet);
				}
				else {
					logger.error("Resource is " + resource.getClass() + " and a " + initialResource.getClass() + "was needed");
				}
			}
			mergeResultSet(initialResultSet, resultSetsToMerge);
		}
		else {
			logger.warn("Resource type : " + initialResource.getClass() + "not completely handle");
		}
	}

	private void mergeResultSet(ResultSet initialResultSet,
			List<ResultSet> resultSetsToMerge) {
		logger.warn("ResultSet not handle yet (has it to be?...)");
	}

	private void mergeQuery(Query initialQuery, List<Query> querysToMerge) {
		logger.warn("Query not handle yet (has it to be?...)");
	}

	private void mergePieceOfKnowledge(PieceOfKnowledge initialPOK,
			List<PieceOfKnowledge> poksToMerge) {
		mergeData(initialPOK, poksToMerge);
		
		if(initialPOK instanceof Annotation) {
			Annotation initialAnnotation = (Annotation) initialPOK;
			List<Annotation> annotationsToMerge = new LinkedList<Annotation>();
			for (PieceOfKnowledge pok : poksToMerge) {
				if(pok instanceof Annotation) {
					Annotation annotation = (Annotation) pok;
					annotationsToMerge.add(annotation);
				}
				else {
					logger.error("PieceOfKnowledge is " + pok.getClass() + " and a " + initialPOK.getClass() + "was needed");
				}
			}
			mergeAnnotation(initialAnnotation, annotationsToMerge);
		}
	}
	
	private void mergeAnnotation(Annotation initialAnnotation,
			List<Annotation> annotationsToMerge) {
		logger.warn("Annotation merge not handle for now, nothing to do here");
	}

	private void mergeComposedResource(
			ComposedResource initialComposedResource,
			List<ComposedResource> composedResourcesToMerge) {
		
		List<String> doneURIs = new LinkedList<String>();
		for (Resource initialResource : initialComposedResource.getResource()) {
			List<Resource> resourcesToMerge = new LinkedList<Resource>();
			for (ComposedResource composedResource : composedResourcesToMerge) {
				boolean notFound = true;
				for (int i = 0; i < composedResource.getResource().size() && notFound; i++) {
					if(composedResource.getResource().get(i).getUri().equals(initialResource.getUri())) {
						notFound = false;
						resourcesToMerge.add(composedResource.getResource().get(i));
					}
				}
				if(notFound) {
					logger.warn("Resource remove not handle");
				}
			}
			mergeResource(initialResource, resourcesToMerge);
			doneURIs.add(initialResource.getUri());
		}
		for (ComposedResource composedResource : composedResourcesToMerge) {
			for (Resource resource : composedResource.getResource()) {
				if(!doneURIs.contains(resource.getUri())){
					logger.info("Add Resource to ComposedResource");
					initialComposedResource.getResource().add(resource);
				}
			}
		}
	}

	private void mergeMediaUnit(MediaUnit initialMediaUnit,
			List<MediaUnit> mediaUnitsToMerge) {
		List<String> doneURIs = new LinkedList<String>();
		for (Segment initialSegment : initialMediaUnit.getSegment()) {
			List<Segment> segmentsToMerge = new LinkedList<Segment>();
			for (MediaUnit mediaUnit : mediaUnitsToMerge) {
				boolean notFound = true;
				for (int i = 0; i < mediaUnit.getSegment().size() && notFound; i++) {
					if(mediaUnit.getSegment().get(i).getUri().equals(initialSegment.getUri())) {
						notFound = false;
						segmentsToMerge.add(mediaUnit.getSegment().get(i));
					}
				}
				if(notFound) {
					logger.warn("Segment remove not handle");
				}
			}
			mergeSegment(initialSegment, segmentsToMerge);
			doneURIs.add(initialSegment.getUri());
		}
		for (MediaUnit mediaUnit : mediaUnitsToMerge) {
			for (Segment segment : mediaUnit.getSegment()) {
				if(!doneURIs.contains(segment.getUri())){
					logger.info("Add Segment to MediaUnit");
					initialMediaUnit.getSegment().add(segment);
				}
			}
		}
		
		if(initialMediaUnit instanceof Document) {
			Document initialDocument = (Document) initialMediaUnit;
			List<Document> documentsToMerge = new LinkedList<Document>();
			for (MediaUnit mediaUnit : mediaUnitsToMerge) {
				if(mediaUnit instanceof Document) {
					documentsToMerge.add((Document) mediaUnit);
				}
				else {
					logger.error("MediaUnit is " + mediaUnit.getClass() + " and a " + initialMediaUnit.getClass() + "was needed");
				}
			}
			
			if(!documentsToMerge.isEmpty()) {
				mergeDocument(initialDocument, documentsToMerge);
			}
		}
		else if(initialMediaUnit instanceof Audio) {
			Audio initialAudio = (Audio) initialMediaUnit;
			List<Audio> audiosToMerge = new LinkedList<Audio>();
			for (MediaUnit mediaUnit : mediaUnitsToMerge) {
				if(mediaUnit instanceof Audio) {
					audiosToMerge.add((Audio) mediaUnit);
				}
				else {
					logger.error("MediaUnit is " + mediaUnit.getClass() + " and a " + initialMediaUnit.getClass() + "was needed");
				}
			}
			
			if(!audiosToMerge.isEmpty()) {
				mergeAudio(initialAudio, audiosToMerge);
			}
		}
		else if(initialMediaUnit instanceof Image) {
			Image initialImage = (Image) initialMediaUnit;
			List<Image> imagesToMerge = new LinkedList<Image>();
			for (MediaUnit mediaUnit : mediaUnitsToMerge) {
				if(mediaUnit instanceof Image) {
					imagesToMerge.add((Image) mediaUnit);
				}
				else {
					logger.error("MediaUnit is " + mediaUnit.getClass() + " and a " + initialMediaUnit.getClass() + "was needed");
				}
			}
			
			if(!imagesToMerge.isEmpty()) {
				mergeImage(initialImage, imagesToMerge);
			}
		}
		else if(initialMediaUnit instanceof Text) {
			Text initialText = (Text) initialMediaUnit;
			List<Text> textsToMerge = new LinkedList<Text>();
			for (MediaUnit mediaUnit : mediaUnitsToMerge) {
				if(mediaUnit instanceof Text) {
					textsToMerge.add((Text) mediaUnit);
				}
				else {
					logger.error("MediaUnit is " + mediaUnit.getClass() + " and a " + initialMediaUnit.getClass() + "was needed");
				}
			}
			
			if(!textsToMerge.isEmpty()) {
				mergeText(initialText, textsToMerge);
			}
		}
		else if(initialMediaUnit instanceof Video) {
			Video initialVideo = (Video) initialMediaUnit;
			List<Video> videosToMerge = new LinkedList<Video>();
			for (MediaUnit mediaUnit : mediaUnitsToMerge) {
				if(mediaUnit instanceof Video) {
					videosToMerge.add((Video) mediaUnit);
				}
				else {
					logger.error("MediaUnit is " + mediaUnit.getClass() + " and a " + initialMediaUnit.getClass() + "was needed");
				}
			}
			
			if(!videosToMerge.isEmpty()) {
				mergeVideo(initialVideo, videosToMerge);
			}
		}
		else {
			logger.warn("MediaUnit type : " + initialMediaUnit.getClass() + "not completely handle");
		}
	}
	
	private void mergeVideo(Video initialVideo, List<Video> videosToMerge) {
		logger.warn("Video content not handle yet (has it to be?...)");
	}

	private void mergeText(Text initialText, List<Text> textsToMerge) {
		logger.warn("Text content not handle yet (has it to be?...)");
	}

	private void mergeImage(Image initialImage, List<Image> imagesToMerge) {
		logger.warn("Image content not handle yet (has it to be?...)");
	}

	private void mergeAudio(Audio initialAudio, List<Audio> audiosToMerge) {
		logger.warn("Audio content not handle yet (has it to be?...)");
	}
	
	private void mergeDocument(Document initialDocument,
			List<Document> documentsToMerge) {
		
		List<String> doneURIs = new LinkedList<String>();
		for (MediaUnit initialMediaUnit : initialDocument.getMediaUnit()) {
			List<Resource> mediaUnitsToMerge = new LinkedList<Resource>();
			for (Document document : documentsToMerge) {
				boolean notFound = true;
				for (int i = 0; i < document.getMediaUnit().size() && notFound; i++) {
					if(document.getMediaUnit().get(i).getUri().equals(initialMediaUnit.getUri())) {
						notFound = false;
						mediaUnitsToMerge.add(document.getMediaUnit().get(i));
					}
				}
				if(notFound) {
					logger.warn("MediaUnit remove not handle");
				}
			}
			mergeResource(initialMediaUnit, mediaUnitsToMerge);
			doneURIs.add(initialMediaUnit.getUri());
		}
		for (Document document : documentsToMerge) {
			for (MediaUnit mediaUnit : document.getMediaUnit()) {
				if(!doneURIs.contains(mediaUnit.getUri())){
					logger.info("Add MediaUnit to Document");
					initialDocument.getMediaUnit().add(mediaUnit);
				}
			}
		}
	}

	private void mergeLowLevelDescriptor(LowLevelDescriptor initialLowLevelDescriptor,
			List<LowLevelDescriptor> lowLevelDescriptorsToMerge) {
		logger.warn("LowLevelDescriptor merge not handle for now");
	}
	
	private void mergeSegment(Segment initialSegment,
			List<Segment> segmentsToMerge) {
		logger.warn("Segment merge not handle for now");
	}

	private void mergeData(PieceOfKnowledge initialPOK, List<PieceOfKnowledge> poksToMerge) {
		AdvancedSelector initialSelector = RDFSelectorFactory.getSelector(true);
		initialSelector.limitToFirstLevelAnnotation(true);
		Statements initialData = initialSelector.searchFor(initialPOK);
		
		Map<String, Map<String, List<String>>> statementToAdd = new HashMap<String, Map<String,List<String>>>();
		for (PieceOfKnowledge pok : poksToMerge) {
			AdvancedSelector selector = RDFSelectorFactory.getSelector(true);
			selector.limitToFirstLevelAnnotation(true);
			Statements data = selector.searchFor(pok);
			for(String subject : data.keySet()){
				if(initialData.containsKey(subject)) {
					IPredicateValuePairs initialPairs = initialData.get(subject);
					IPredicateValuePairs currentPairs = data.get(subject);
					
					for (String predicate : currentPairs.allPredicates()) {
						Object initialValues = initialPairs.getValue(predicate);
						if(initialValues == null) {
							addStatement(statementToAdd, subject, predicate, currentPairs.getValue(predicate));
						}
						else if(initialValues instanceof String) {
							String initialString = (String) initialValues;
							Object currentValues = currentPairs.getValue(predicate);
							if(currentValues instanceof String) {
								String currentString = (String) currentValues;
								if(!currentString.equals(initialString)) {
									addStatement(statementToAdd, subject, predicate, currentString);
								}
							}
							else if(currentValues instanceof List) {
								List<?> currentList = (List<?>) currentValues;
								for (Object object : currentList) {
									String s = (String) object;
									if(!s.equals(initialString)) {
										addStatement(statementToAdd, subject, predicate, s);
									}
								}
							}
						}
						else if(initialValues instanceof List) {
							List<?> initialList = (List<?>) initialValues;
							Object currentValues = currentPairs.getValue(predicate);
							if(currentValues instanceof String) {
								String currentString = (String) currentValues;
								if(!initialList.contains(currentString)) {
									addStatement(statementToAdd, subject, predicate, currentString);
								}
							}
							else if(currentValues instanceof List) {
								List<?> currentList = (List<?>) currentValues;
								for (Object object : currentList) {
									String currentString = (String) object;
									boolean notFound = true;
									for (int i = 0; i < initialList.size() && notFound; i++) {
										String initialString = (String) initialList.get(i);
										if(initialString.equals(currentString)) {
											notFound = false;
										}
									}
									if(notFound) {
										addStatement(statementToAdd, subject, predicate, currentString);
									}
								}
							}
						}
					}
				}
				else {
					logger.error("Annotation not a the same subject");
				}
			}
		}
		writeStatements(statementToAdd, initialData, initialPOK);
	}

	private void addStatement(
			Map<String, Map<String, List<String>>> statementToAdd, String subject, String predicate, Object value) {
		if(value != null && statementToAdd != null) {
			Map<String, List<String>> mapSubject = null;
			if(statementToAdd.containsKey(subject)) {
				mapSubject = statementToAdd.get(subject);
			}
			else {
				mapSubject = new HashMap<String, List<String>>();
				statementToAdd.put(subject, mapSubject);
			}
			
			if(mapSubject != null) {
				List<String> listPred = null;
				if(mapSubject.containsKey(predicate)) {
					listPred = mapSubject.get(predicate);
				}
				else {
					listPred = new LinkedList<String>();
					mapSubject.put(predicate, listPred);
				}
				
				if(listPred != null) {
					if(value instanceof String) {
						String s = (String) value;
						if(!listPred.contains(s)) {
							listPred.add(s);
						}
					}
					else if(value instanceof List) {
						List<?> list = (List<?>) value;
						if(!listPred.containsAll(list)) {
							for (Object object : list) {
								if(object instanceof String && !listPred.contains(object)) {
									listPred.add((String) object);
								}
							}
						}
					}
					else {
						logger.error("Value class not known.");
					}
				}
			}
		}
	}
	
	private void writeStatements(
			Map<String, Map<String, List<String>>> statementToAdd,
			Statements initialData, PieceOfKnowledge initialPOK) {
		for (String subject : statementToAdd.keySet()) {
			Map<String, List<String>> currentStatements = statementToAdd.get(subject);
			if(initialData.containsKey(subject)) {
				for (String predicate : currentStatements.keySet()) {
					List<String> list = currentStatements.get(predicate);
					for (String string : list) {
						try {
							JenaPoKHelper jph = new JenaPoKHelper(initialPOK);
							jph.createLitStat(subject, predicate, string);
						} catch (Exception e) {
							logger.error("Failed to write Statement");
							e.printStackTrace();
						}
					}
					
				}
			}
			else {
				logger.error("Subject merge not yet implemented");
			}
		}
	}
}
