/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2010 CASSIDIAN
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.core.extended.factory;

import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;
import java.util.Map.Entry;

import org.ow2.weblab.core.extended.uri.WebLabRI;
import org.ow2.weblab.core.model.Annotation;
import org.ow2.weblab.core.model.ComposedResource;
import org.ow2.weblab.core.model.Document;
import org.ow2.weblab.core.model.LowLevelDescriptor;
import org.ow2.weblab.core.model.MediaUnit;
import org.ow2.weblab.core.model.Resource;
import org.ow2.weblab.core.model.Segment;

/**
 * @author Cassidian WebLab Team
 * @since 1.2
 * @date 2010-11-02
 */
abstract class AbstractFactory {

	/**
	 * This data structure is not well adapted to manage WebLab Resource
	 */
	final protected static Map<String, Set<String>> innerMap = Collections.synchronizedMap(new LinkedHashMap<String, Set<String>>());

	protected synchronized static void cleanMapping(final String uri) {
		final Iterator<Entry<String, Set<String>>> it = innerMap.entrySet().iterator();
		while (it.hasNext()) {
			final Entry<String, Set<java.lang.String>> entry = it.next();
			if (entry.getValue().contains(uri)) {
				it.remove();
			}
		}
	}

	protected synchronized static Set<String> getUris(final String key) {
		if (!innerMap.containsKey(key)) {
			for (Set<String> test : innerMap.values()) {
				if (test.contains(key)) {
					return test;
				}
			}
			return new HashSet<String>();
		}
		return innerMap.get(key);
	}

	protected synchronized static void setUris(final String key, final Set<String> uris) {
		while (innerMap.size() > 1000) {
			final String fkey = innerMap.keySet().iterator().next();
			innerMap.remove(fkey);
		}
		innerMap.put(key, uris);
	}

	/**
	 * List uris in the resource
	 * 
	 * @param resource
	 *            a resource
	 * @return a set of uri in this resource
	 */
	protected synchronized static Set<String> listUri(final Resource resource) {
		return listUri(resource, false);
	}

	private static Set<String> listUri(final String uri) {
		final Set<String> buffered_uris = getUris(uri);
		return buffered_uris;
	}

	protected synchronized static Set<String> listUri(final Resource resource, final boolean force) {
		final Set<String> buffered_uris = listUri(resource.getUri());
		if (!force && !buffered_uris.isEmpty()) {
			return buffered_uris;
		}
		final Set<String> set = Collections.synchronizedSet(new HashSet<String>());
		set.add(resource.getUri());
		if (resource instanceof Document) {
			final Document doc = (Document) resource;
			if (doc.isSetMediaUnit()) {
				for (final MediaUnit inner : doc.getMediaUnit()) {
					set.addAll(listUri(inner));
				}
			}
			if(doc.isSetSegment()){
				for (final Segment s : doc.getSegment()) {
					set.add(s.getUri());
				}
			}
		} else if (resource instanceof ComposedResource) {
			final ComposedResource cr = (ComposedResource) resource;
			for (final Resource inner : cr.getResource()) {
				set.addAll(listUri(inner));
			}
		} else if(resource instanceof MediaUnit){
			final MediaUnit mu = (MediaUnit) resource;
			if(mu.isSetSegment()){
				for (final Segment s : mu.getSegment()) {
					set.add(s.getUri());
				}
			}
		}
		
		if (resource.isSetAnnotation()) {
			for (final Annotation a : resource.getAnnotation()) {
				set.addAll(listUri(a));
			}
		}
		if (resource.isSetDescriptor()) {
			for (final LowLevelDescriptor lld : resource.getDescriptor()) {
				set.addAll(listUri(lld));
			}
		}
		setUris(resource.getUri(), set);

		return set;
	}

	protected synchronized static String createUniqueURIFrom(final Resource resource, final boolean isChild, final boolean isAnnotation) {
		return createUniqueURIin(listUri(resource), isChild, isAnnotation, resource.getUri());
	}

	protected synchronized static String createUniqueURIin(final Set<String> uris, final boolean isChild, final boolean isAnnotation, final String parentUri) {
		final String ref = "webLabFactory";
		final String res = "auto-parent";
		String template = "weblab://" + ref + "/" + res;
		if (isAnnotation) {
			if (parentUri.indexOf("#") != -1) {
				template = parentUri + "-a";
			} else {
				template = parentUri + "#a";
			}
		}
		if (isChild) {
			if (parentUri.indexOf("#") != -1) {
				template = parentUri + "-";
			} else {
				template = parentUri + "#";
			}
		}
		int i = uris.size() - 1;

		String uri = null;
		while (uri == null || uris.contains(uri)) {
			if (i + 1 % 1000 == 0) {
				template += i + "_";
				i = 0;
			}
			uri = template + (i++);
		}
		return new WebLabRI(uri).toString();
	}

	protected synchronized static void removeChild(final MediaUnit mu, final Resource parent, final Resource root) {
		final Set<String> uris = listUri(root);
		if (parent instanceof ComposedResource) {
			final ComposedResource cr = (ComposedResource) parent;
			cr.getResource().remove(mu);

			uris.remove(mu.getUri());
		}
	}

	protected synchronized static void addAnnotation(final Annotation annotation, final Resource resource, final Resource root) {
		resource.getAnnotation().add(annotation);
		listUri(root).add(annotation.getUri());
	}

	protected synchronized static  void addSegment(final Segment segment, final MediaUnit parent, final Resource root) {
		parent.getSegment().add(segment);
		listUri(root).add(segment.getUri());
	}

	protected synchronized static void addDescriptor(final LowLevelDescriptor descriptor, final Resource parent, final Resource root) {
		parent.getDescriptor().add(descriptor);
		listUri(root).add(descriptor.getUri());
	}

	protected synchronized static void addChild(final MediaUnit mu, final Resource parent, final Resource root) {
		final Set<String> uris = listUri(root);
		if (parent instanceof ComposedResource) {
			final ComposedResource cr = (ComposedResource) parent;
			cr.getResource().add(mu);
			uris.add(mu.getUri());
		} else if (parent instanceof Document) {
			final Document doc = (Document) parent;
			doc.getMediaUnit().add(mu);
			uris.add(mu.getUri());
		}

	}

	protected synchronized static Resource findParent(final MediaUnit mu, final Resource resource) {
		if (resource instanceof ComposedResource) {
			final ComposedResource cr = (ComposedResource) resource;
			if (cr.getResource().contains(mu)) {
				return resource;
			}
			for (final Resource inner : cr.getResource()) {
				final Resource r = findParent(mu, inner);
				if (r != null) {
					return r;
				}
			}
		} else if (resource instanceof Document) {
			final Document doc = (Document) resource;
			if (doc.isSetMediaUnit() && doc.getMediaUnit().contains(mu)) {
				return resource;
			}
		}
		return null;
	}

	/**
	 * @param res
	 *            The <code>Resource</code> to be used to create <code>WebLabRI</code>
	 * @return The <code>String</code> to be used as <code>URI</code> by the <code>MediaUnit</code> created
	 */
	protected synchronized final static String getUniqueWebLabRIFrom(final Resource res, final boolean isChild, final boolean isAnnotation) {
		return createUniqueURIFrom(res, isChild, isAnnotation);
	}
}
