/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2010 CASSIDIAN
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.core.extended.factory;

import org.ow2.weblab.core.extended.exception.WebLabRISyntaxException;
import org.ow2.weblab.core.extended.exception.WebLabResourceCreationException;
import org.ow2.weblab.core.model.MediaUnit;
import org.ow2.weblab.core.model.Resource;


/**
 * To create <code>MediaUnit</code>s that are not <code>Document</code> we strongly encourage to use this class. It will
 * enable the creation of a <code>MediaUnit</code> having an <code>URI</code> compliant with the <code>WebLabRI</code>
 * recommendations. <br />
 * Since a non-<code>Document</code> <code>MediaUnit</code> hasn't any sense without being attached to a
 * <code>Document</code>, we only provide a method for creating <code>MediaUnit</code>s that takes
 * a <code>Document</code> in parameter. As a result, the
 * created <code>MediaUnit</code> (instance of the class given in parameter) is added to the <code>MediaUnit</code>
 * <code>List</code> of the <code>Document</code>.
 * 
 * @author Cassidian WebLab Team
 * @date 2008-01-18
 */
public class MediaUnitFactory extends AbstractFactory {


	/**
	 * Constructors Do not use it.
	 */
	private MediaUnitFactory() {
		throw new UnsupportedOperationException("This class only contains static methods; no need to instantiate it.");
	}

	/**
	 * The separator char.
	 */
	protected final static char SEPARATOR = ResourceFactory.SEPARATOR;

	/**
	 * Automatically set a valid <code>WebLabRI</code> to the created <code>MediaUnit</code> (an instance of
	 * <code>muClass</code> in parameter) using <code>doc</code> to generate an unique id. Created
	 * <code>MediaUnit</code> will be added to the <code>MediaUnit</code> <code>List</code> of<code>doc</code>.
	 * 
	 * @param <T>
	 *            The subclass of <code>MediaUnit</code> to be created
	 * @param res
	 *            the <code>Resource</code> that will contains the created <code>MediaUnit</code>
	 * @param muClass
	 *            The class of the <code>MediaUnit</code> to be created
	 * @return a new instance of <code>muClass</code> with a valid <code>WebLabRI</code> and added to <code>cu</code>
	 */
	public final static <T extends MediaUnit> T createAndLinkMediaUnit(final Resource res, final Class<T> muClass) {
		final T mediaUnit;
		try {
			mediaUnit = muClass.newInstance();
		} catch (final InstantiationException ie) {
			throw new WebLabResourceCreationException(WebLabResourceCreationException.EXCEPTION + muClass.getSimpleName(), ie);
		} catch (final IllegalAccessException iae) {
			throw new WebLabResourceCreationException(WebLabResourceCreationException.EXCEPTION + muClass.getSimpleName(), iae);
		}

		try {
			mediaUnit.setUri(AbstractFactory.getUniqueWebLabRIFrom(res, true, false));
			AbstractFactory.addChild(mediaUnit, res, res);
		} catch (final WebLabRISyntaxException wlrise) {
			throw new WebLabResourceCreationException(WebLabResourceCreationException.EXCEPTION + muClass.getSimpleName(), wlrise);
		}

		return mediaUnit;
	}

}
