/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2010 CASSIDIAN
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.core.extended.factory;

import org.ow2.weblab.core.extended.exception.WebLabRISyntaxException;
import org.ow2.weblab.core.extended.exception.WebLabResourceCreationException;
import org.ow2.weblab.core.extended.uri.WebLabRI;
import org.ow2.weblab.core.model.PieceOfKnowledge;
import org.ow2.weblab.core.model.ResultSet;


/**
 * To create <code>PieceOfKnowledge</code>s we strongly encourage to use this class. It will enable the creation of an
 * <code>Annotation</code> having an <code>URI</code> compliant with the <code>WebLabRI</code> recommendations. <br />
 * You may need <code>PieceOfKnowledge</code>s in three main cases.
 * <ul>
 * <li>First, you just built an empty <code>ResultSet</code> and you need to add a <code>PoK</code> in it. In that case
 * you shall use this class.</li>
 * <li>Secondly, you just need a standalone <code>PoK</code> or a standalone <code>Ontology</code>. In that case, we
 * encourage you to use the method <code>createResource</code> in <code>ResourceFactory</code>.</li>
 * <li>Finally, you need to create an <code>Annotation</code>. In that case, we encourage you to use the method
 * <code>createAndLinkAnnotation</code> of the <code>AnnotationFactory</code> class.</li>
 * </ul>
 * 
 * @author Cassidian WebLab Team
 * @date 2008-07-20
 * @see AnnotationFactory
 * @see ResourceFactory
 * @since 1.1
 */
public class PoKFactory {

	/**
	 * Constructors Do not use it.
	 */
	private PoKFactory() {
		throw new UnsupportedOperationException("This class only contains static methods; no need to instantiate it.");
	}

	/**
	 * The char announcing an pok in fragments
	 */
	private final static char POK_CHAR = 'p';

	/**
	 * Automatically set a valid <code>WebLabRI</code> to the created <code>PieceOfKnowledge</code> using the
	 * <code>ResultSet</code> in parameter to generate an
	 * unique id. Created <code>PoK</code> will be added to <code>resultSet</code>.
	 * 
	 * @param resultSet
	 *            the <code>ResultSet</code> that will contains the created <code>PieceOfKnowledge</code>
	 * @return a new <code>PieceOfKnowledge</code> with a valid <code>WebLabRI</code> and added to
	 *         <code>resultSet</code>
	 */
	static PieceOfKnowledge createAndLinkPoK(ResultSet resultSet) {
		PieceOfKnowledge pok = new PieceOfKnowledge();
		try {
			pok.setUri(PoKFactory.getUniqueWelLabRIFrom(resultSet));
			resultSet.setPok(pok);
		} catch (final WebLabRISyntaxException wlrise) {
			throw new WebLabResourceCreationException(WebLabResourceCreationException.EXCEPTION + "PieceOfKnowledge", wlrise);
		}
		return pok;
	}

	/**
	 * @param resultSet
	 *            The <code>ResultSet</code> to be used to create <code>WebLabRI</code>
	 * @return The <code>String</code> to be used as <code>URI</code> by the <code>PieceOfKnowledge</code> created
	 */
	private final static String getUniqueWelLabRIFrom(final ResultSet resultSet) {
		// Check the parent WebLabRI validity (might throw WeblabRISyntaxException)
		WebLabRI uri = new WebLabRI(resultSet.getUri());
		uri.addFragment("" + PoKFactory.POK_CHAR);
		return uri.toString();
	}
}
