/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2010 CASSIDIAN
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.core.extended.ontologies;

import org.ow2.weblab.core.extended.exception.WebLabUncheckedException;
import org.ow2.weblab.core.model.Resource;


/**
 * @author Cassidian WebLab Team
 * @date 2008-05-07
 */
public final class WebLab {

	/**
	 * Constructors Do not use it.
	 */
	private WebLab() {
		throw new UnsupportedOperationException("This class only contains static methods; no need to instantiate it.");
	}

	// Base URIs

	/**
	 * http://weblab.ow2.org/model#
	 */
	public final static String MODEL_NAMESPACE = "http://weblab.ow2.org/core/1.2/model#";

	/**
	 * http://weblab.ow2.org/core/1.2/model#
	 */
	public final static String CLASS_NAMESPACE = WebLab.MODEL_NAMESPACE;

	/**
	 * The base URI of the WebLab Resources Retrieval properties.
	 * http://weblab.ow2.org/core/1.2/ontology/retrieval#
	 */
	public final static String RETRIEVAL_PROPERTY_NAMESPACE = "http://weblab.ow2.org/core/1.2/ontology/retrieval#";

	/**
	 * The base URI of the WebLab Resources Processing properties.
	 * http://weblab.ow2.org/core/1.2/ontology/processing#
	 */
	public final static String PROCESSING_PROPERTY_NAMESPACE = "http://weblab.ow2.org/core/1.2/ontology/processing#";

	/**
	 * The base URI of the WebLab structure properties.
	 * http://weblab.ow2.org/core/1.2/model#
	 */
	public final static String STRUCTURE_NAMESPACE = WebLab.MODEL_NAMESPACE;

	// Structure properties

	/**
	 * Links a MediaUnit to one of its Segment.
	 */
	public final static String HAS_SEGMENT = WebLab.STRUCTURE_NAMESPACE + "hasSegment";

	/**
	 * Links a Segment to its parent MediaUnit.
	 */
	public final static String IN_MEDIA_UNIT = WebLab.STRUCTURE_NAMESPACE + "inMediaUnit";

	/**
	 * Links a ComposedUnit to one of its sub MediaUnit.
	 */
	public final static String HAS_MEDIA_UNIT = WebLab.STRUCTURE_NAMESPACE + "hasMediaUnit";

	/**
	 * Links a MediaUnit to its parent ComposedUnit.
	 */
	public final static String IN_COMPOSED_UNIT = WebLab.STRUCTURE_NAMESPACE + "inComposedUnit";

	/**
	 * Links a Resource to one of its Annotation.
	 */
	public final static String HAS_ANNOTATION = WebLab.STRUCTURE_NAMESPACE + "hasAnnotation";

	/**
	 * Links an Annotation to its parent Resource.
	 */
	public final static String IN_RESOURCE_ANNOTATION = WebLab.STRUCTURE_NAMESPACE + "inResourceAnnotation";

	/**
	 * Links a Resource to one of its LowLevelDescriptor.
	 */
	public final static String HAS_LOW_LEVEL_DESCRIPTOR = WebLab.STRUCTURE_NAMESPACE + "hasLowLevelDescriptor";

	/**
	 * Links an LowLevelDescriptor to its parent Resource.
	 */
	public final static String IN_RESOURCE_LLD = WebLab.STRUCTURE_NAMESPACE + "inResourceLLD";

	/**
	 * Links a ComposedResource one of its Resource.
	 */
	public final static String HAS_RESOURCE = WebLab.STRUCTURE_NAMESPACE + "hasResource";

	/**
	 * Links a Resource to one of its parent ComposedResource.
	 */
	public final static String IN_COMPOSED_RESOURCE = WebLab.STRUCTURE_NAMESPACE + "inComposedResource";

	/**
	 * Links a ResultSet to one of its Resource result.
	 */
	public final static String HAS_RESOURCE_RESULT = WebLab.STRUCTURE_NAMESPACE + "hasResourceResult";

	/**
	 * Links a Resource to one of its parent ResultSet.
	 */
	public final static String IN_RESULT_SET_RESOURCE = WebLab.STRUCTURE_NAMESPACE + "inResultSetResource";

	/**
	 * Links a ResultSet its PieceOfKnowledge result.
	 */
	public final static String HAS_POK_RESULT = WebLab.STRUCTURE_NAMESPACE + "hasPoKResult";

	/**
	 * Links a Hit to one of its parent ResultSet.
	 */
	public final static String IN_RESULT_SET_HIT = WebLab.RETRIEVAL_PROPERTY_NAMESPACE + "inResultSetHit";
	
	/**
	 * Links a PieceOfKnowldege to one of its parent ResultSet.
	 */
	public final static String IN_RESULT_SET_POK = WebLab.STRUCTURE_NAMESPACE + "inResultSetPoK";

	/**
	 * Links a ComposedQuery one of its Query.
	 */
	public final static String HAS_QUERY_IN_COMPOSED = WebLab.STRUCTURE_NAMESPACE + "hasQuery";

	/**
	 * Links a Query to one of its parent ComposedQuery.
	 */
	public final static String IN_COMPOSED_QUERY = WebLab.STRUCTURE_NAMESPACE + "inComposedQuery";

	/**
	 * Links a SimilarityQuery to one of its Resource sample.
	 */
	public final static String HAS_RESOURCE_SIM_QUERY = WebLab.STRUCTURE_NAMESPACE + "hasResourceSimQuery";

	/**
	 * Links a Resource to one of its parent SimilarityQuery.
	 */
	public final static String IN_SIMILARITY_QUERY = WebLab.STRUCTURE_NAMESPACE + "inSimilarityQuery";

	// Processing properties

	/**
	 * Links a media unit to its native content.
	 */
	public final static String HAS_NATIVE_CONTENT = WebLab.PROCESSING_PROPERTY_NAMESPACE + "hasNativeContent";

	/**
	 * Links a media unit to its normalised content.
	 */
	public final static String HAS_NORMALISED_CONTENT = WebLab.PROCESSING_PROPERTY_NAMESPACE + "hasNormalisedContent";

	/**
	 * Links a media unit result of an automatic (translation, summarisation,
	 * transcription) process from a source resource.
	 */
	public final static String IS_GENERATED_FROM = WebLab.PROCESSING_PROPERTY_NAMESPACE + "isGeneratedFrom";

	/**
	 * Indicates the service that has produced automatically or
	 * semi-automatically the resource
	 */
	public final static String IS_PRODUCED_BY = WebLab.PROCESSING_PROPERTY_NAMESPACE + "isProducedBy";

	/**
	 * Indicates the service that has updated automatically or
	 * semi-automatically the resource.
	 */
	public final static String HAS_BEEN_UPDATED_BY = WebLab.PROCESSING_PROPERTY_NAMESPACE + "hasBeenUpdatedBy";

	/**
	 * Indicate the processing of this media unit can be ignored by a service.
	 */
	public final static String CAN_BE_IGNORED = WebLab.PROCESSING_PROPERTY_NAMESPACE + "canBeIgnored";

	/**
	 * Is exposed on a given protocol at a given address (for instance points to
	 * an URL).
	 */
	public final static String IS_EXPOSED_AS = WebLab.PROCESSING_PROPERTY_NAMESPACE + "isExposedAs";

	/**
	 * Refers to an given instance. It might be an existing instance from an
	 * Ontology or a instance created in the Document or Text Annotation.
	 */
	public final static String REFERS_TO = WebLab.PROCESSING_PROPERTY_NAMESPACE + "refersTo";

	/**
	 * Whether or not is the instance extracted a candidate instance.
	 */
	public final static String IS_CANDIDATE = WebLab.PROCESSING_PROPERTY_NAMESPACE + "isCandidate";

	/**
	 * Gives the number of bytes of the original file crawled.
	 */
	public final static String HAS_ORIGINAL_FILE_SIZE = WebLab.PROCESSING_PROPERTY_NAMESPACE + "hasOriginalFileSize";

	/**
	 * Gives the name of original file crawled.
	 */
	public final static String HAS_ORIGINAL_FILE_NAME = PROCESSING_PROPERTY_NAMESPACE + "hasOriginalFileName";

	/**
	 * Gives the date when the Document where crawled.
	 */
	public final static String HAS_GATHERING_DATE = WebLab.PROCESSING_PROPERTY_NAMESPACE + "hasGatheringDate";

	/**
	 * Gives the stem for a text.
	 */
	public final static String HAS_STEM = WebLab.PROCESSING_PROPERTY_NAMESPACE + "hasStem";

	// Retrieval

	// RDF classes
	/**
	 * A hit describes media document properties corresponding to a request.
	 */
	public final static String HIT = WebLab.RETRIEVAL_PROPERTY_NAMESPACE + "Hit";

	/**
	 * A SearchOrder describe a search order requested by a user or a service.
	 */
	public final static String SEARCH_ORDER = WebLab.RETRIEVAL_PROPERTY_NAMESPACE + "retrieval/SearchOrder";

	// Properties

	/**
	 * Links a ResultSet to the Query that has been used to retrieve the set of
	 * results.
	 */
	public final static String IS_RESULT_OF = WebLab.RETRIEVAL_PROPERTY_NAMESPACE + "isResultOf";

	/**
	 * Provides the hit score of a given media unit regarding a given query in
	 * the case of a result collection returns by a search service.
	 */
	public final static String HAS_SCORE = WebLab.RETRIEVAL_PROPERTY_NAMESPACE + "hasScore";

	/**
	 * Provides the offset about the query in the case of a result set returns
	 * by a search service.
	 */
	public final static String HAS_QUERY_OFFSET = WebLab.RETRIEVAL_PROPERTY_NAMESPACE + "hasQueryOffset";

	/**
	 * Provides the rank of a given media unit regarding a given query in the
	 * case of a result collection returns by a search service.
	 */
	public final static String HAS_RANK = WebLab.RETRIEVAL_PROPERTY_NAMESPACE + "hasRank";

	/**
	 * Links a ResultSet to one of its Hit.
	 */
	public final static String HAS_HIT = WebLab.RETRIEVAL_PROPERTY_NAMESPACE + "hasHit";

	/**
	 * Provides the snippet of a given media unit regarding a given query in the
	 * case of a result set returns by a search service.
	 */
	public final static String HAS_DESCRIPTION = WebLab.RETRIEVAL_PROPERTY_NAMESPACE + "hasDescription";

	/**
	 * Provides the snippet of a given media unit regarding a given query in the
	 * case of a result collection returns by a search service.
	 */
	public final static String IS_LINKED_TO = WebLab.RETRIEVAL_PROPERTY_NAMESPACE + "isLinkedTo";

	/**
	 * Gives the total number of results to a given query.
	 */
	public final static String HAS_NUMBER_OF_RESULTS = WebLab.RETRIEVAL_PROPERTY_NAMESPACE + "hasNumberOfResults";

	/**
	 * Gives the language of the query.
	 */
	public final static String IS_EXPRESSED_WITH = WebLab.RETRIEVAL_PROPERTY_NAMESPACE + "isExpressedWith";

	/**
	 * Gives the query for this order.
	 */
	public final static String HAS_QUERY = WebLab.RETRIEVAL_PROPERTY_NAMESPACE + "hasQuery";

	/**
	 * Gives the offset for the requested query for this order.
	 */
	public final static String HAS_EXPECTED_OFFSET = WebLab.RETRIEVAL_PROPERTY_NAMESPACE + "hasExpectedOffset";

	/**
	 * Gives the limit for the requested query for this order.
	 */
	public final static String HAS_EXPECTED_LIMIT = WebLab.RETRIEVAL_PROPERTY_NAMESPACE + "hasExpectedLimit";

	// Methods

	/**
	 * Use this to get the WebLab ontology owl:Class URI corresponding to a java
	 * Class.
	 * 
	 * @warning Segment are not resources since 1.2
	 * @see #getClassFromUri(String)
	 * @param resourceClass
	 *            the java Class
	 * @return the ontology URI class
	 */
	public static String getUriFromClass(final Class<? extends Resource> resourceClass) {
		return WebLab.CLASS_NAMESPACE + resourceClass.getSimpleName();
	}

	/**
	 * Inverse method of {@link #getUriFromClass(Class)}.
	 * 
	 * @warning Segment are not resources since 1.2
	 * @see #getUriFromClass(Class)
	 * @param uri
	 *            The URI denoting a WebLab class in the ontology
	 * @return The java class
	 */
	public static Class<? extends Resource> getClassFromUri(final String uri) {
		final Class<? extends Resource> clazz;

		if (uri == null || uri.length() <= WebLab.CLASS_NAMESPACE.length() || !uri.startsWith(WebLab.CLASS_NAMESPACE)) {
			throw new WebLabUncheckedException("Uri: " + uri + " was not a WebLab class. ");
		}
		final String className = '.' + uri.substring(WebLab.CLASS_NAMESPACE.length());
		final Class<?> clazz1;
		try {
			clazz1 = ClassLoader.getSystemClassLoader().loadClass(Resource.class.getPackage().getName() + className);
		} catch (final ClassNotFoundException cnfe) {
			throw new WebLabUncheckedException("Uri: " + uri + " was not a WebLab class.", cnfe);
		}
		try {
			clazz = clazz1.asSubclass(Resource.class);
		} catch (final ClassCastException cce) {
			throw new WebLabUncheckedException("Unable to cast class " + clazz1 + " into Resource.", cce);
		}

		return clazz;
	}
}
