/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2010 CASSIDIAN
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.core.extended.util;

import java.net.URL;
import java.util.Map;

import javax.xml.namespace.QName;
import javax.xml.ws.BindingProvider;

import org.ow2.weblab.core.extended.exception.WebLabUncheckedException;
import org.ow2.weblab.core.services.Analyser;
import org.ow2.weblab.core.services.Analyser_Service;
import org.ow2.weblab.core.services.Indexer;
import org.ow2.weblab.core.services.Indexer_Service;
import org.ow2.weblab.core.services.QueueManager;
import org.ow2.weblab.core.services.QueueManager_Service;
import org.ow2.weblab.core.services.Searcher;
import org.ow2.weblab.core.services.Searcher_Service;
import org.ow2.weblab.core.services.SourceReader;
import org.ow2.weblab.core.services.SourceReader_Service;

/**
 * Util class for changing the generated web services client address.
 * 
 * @author Cassidian WebLab Team
 * @date 2008-07-28
 */
public class ServiceUtil {

	private static final String SERVICES_NAMESPACE = "http://weblab.ow2.org/core/1.2/services";
	
	/**
	 * Constructors Do not use it.
	 */
	private ServiceUtil() {
		throw new UnsupportedOperationException("This class only contains static methods; no need to instantiate it.");
	}

	/**
	 * Replaces the SOAPAction and endpoint address of <code>port</code> by <code>SOAPAction</code> and <code>newAddress</code>.
	 * 
	 * @param port
	 *            An instance of <code>BindingProvider</code> to have it's SOAPAction and endpoint address changed.
	 * @param newAddress
	 *            The endpoint address to be used.
	 * @param SOAPAction
	 *            The SOAPAction to be used.
	 */
	public static void setEndpointAddress(final Object port, final String newAddress, final String SOAPAction) {
		if (!(port instanceof BindingProvider)) {
			throw new WebLabUncheckedException("Object: " + port + " doesn't appear to be a valid port.");
		}
		final Map<String, Object> context = ((BindingProvider) port).getRequestContext();
		context.put(BindingProvider.ENDPOINT_ADDRESS_PROPERTY, newAddress);
		context.put(BindingProvider.SOAPACTION_USE_PROPERTY, Boolean.TRUE);
		context.put(BindingProvider.SOAPACTION_URI_PROPERTY, SOAPAction);
	}

	/**
	 * Build a Analyser service from WSDL and web Service URL.
	 * 
	 * @warning Will not work if your Analyser contains more than one exposed method (SOAP Action).
	 * @param WSDLLocation
	 *            The WSDL Location.
	 * @param analyserEndpointAddress
	 *            The new endpoint address to be affected (using {@link #setEndpointAddress(Object, String, String)}.
	 * @return The Analyser stub using right SOAP action, WSDL location and endpoint address.
	 */
	public static Analyser getAnalyserService(final URL WSDLLocation, final URL analyserEndpointAddress) {
		final Analyser analyser = new Analyser_Service(WSDLLocation, new QName(SERVICES_NAMESPACE, "Analyser")).getAnalyserPort();
		ServiceUtil.setEndpointAddress(analyser, analyserEndpointAddress.toString(), "process");
		return analyser;
	}

	/**
	 * Build a QueueManager service from WSDL and web Service URL.
	 * 
	 * @warning Will not work if your QueueManager contains more than one exposed method (SOAP Action).
	 * @param WSDLLocation
	 *            The WSDL Location.
	 * @param queueManagerEndpointAddress
	 *            The new endpoint address to be affected (using {@link #setEndpointAddress(Object, String, String)}.
	 * @return The QueueManager stub using right SOAP action, WSDL location and endpoint address.
	 */
	public static QueueManager getQueueManagerService(final URL WSDLLocation, final URL queueManagerEndpointAddress) {
		final QueueManager manager = new QueueManager_Service(WSDLLocation, new QName(SERVICES_NAMESPACE, "QueueManager")).getQueueManagerPort();
		ServiceUtil.setEndpointAddress(manager, queueManagerEndpointAddress.toString(), "nextResource");
		return manager;
	}

	/**
	 * Build a Indexer service from WSDL and web Service URL.
	 * 
	 * @warning Will not work if your Indexer contains more than one exposed method (SOAP Action).
	 * @param WSDLLocation
	 *            The WSDL Location.
	 * @param indexerEndpointAddress
	 *            The new endpoint address to be affected (using {@link #setEndpointAddress(Object, String, String)}.
	 * @return The Indexer stub using right SOAP action, WSDL location and endpoint address.
	 */
	public static Indexer getIndexerService(final URL WSDLLocation, final URL indexerEndpointAddress) {
		final Indexer indexer = new Indexer_Service(WSDLLocation, new QName(SERVICES_NAMESPACE, "Indexer")).getIndexerPort();
		ServiceUtil.setEndpointAddress(indexer, indexerEndpointAddress.toString(), "index");
		return indexer;
	}

	/**
	 * Build a Searcher service from WSDL and web Service URL.
	 * 
	 * @warning Will not work if your Searcher contains more than one exposed method (SOAP Action).
	 * @param WSDLLocation
	 *            The WSDL Location.
	 * @param searcherEndpointAddress
	 *            The new endpoint address to be affected (using {@link #setEndpointAddress(Object, String, String)}.
	 * @return The Searcher stub using right SOAP action, WSDL location and endpoint address.
	 */
	public static Searcher getSearcherService(final URL WSDLLocation, final URL searcherEndpointAddress) {
		final Searcher searcher = new Searcher_Service(WSDLLocation, new QName(SERVICES_NAMESPACE, "Searcher")).getSearcherPort();
		ServiceUtil.setEndpointAddress(searcher, searcherEndpointAddress.toString(), "search");
		return searcher;
	}
	
	/**
	 * Build a SourceReader service from WSDL and web Service URL.
	 * 
	 * @warning Will not work if your SourceReader contains more than one exposed method (SOAP Action).
	 * @param WSDLLocation
	 *            The WSDL Location.
	 * @param searcherEndpointAddress
	 *            The new endpoint address to be affected (using {@link #setEndpointAddress(Object, String, String)}.
	 * @return The SourceReader stub using right SOAP action, WSDL location and endpoint address.
	 */
	public static SourceReader getSourceReaderService(final URL WSDLLocation, final URL searcherEndpointAddress) {
		final SourceReader searcher = new SourceReader_Service(WSDLLocation, new QName(SERVICES_NAMESPACE, "SourceReader")).getSourceReaderPort();
		ServiceUtil.setEndpointAddress(searcher, searcherEndpointAddress.toString(), "search");
		return searcher;
	}

}
