/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2011 CASSIDIAN
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.core.extended.comparator;

import java.io.Serializable;
import java.util.Comparator;
import java.util.List;

import org.apache.commons.logging.LogFactory;
import org.ow2.weblab.core.model.Coordinate;
import org.ow2.weblab.core.model.LinearSegment;
import org.ow2.weblab.core.model.Segment;
import org.ow2.weblab.core.model.SpatialSegment;
import org.ow2.weblab.core.model.SpatioTemporalSegment;
import org.ow2.weblab.core.model.TemporalSegment;
import org.ow2.weblab.core.model.TrackSegment;


/**
 * A <code>Comparator</code> for <code>Segment</code>s. <br />
 * First of all <code>Segment</code> are ordered using their type: <code>LinearSegment</code>, <code>TemporalSegment</code>, <code>SpatioTemporalSegment</code>,
 * <code>SpatialSegment</code>, <code>TrackSegment</code> and then any other <code>Segment</code>. <br />
 * Then <code>Segment</code>s of same types are compare together. <br />
 * For <code>LinearSegment</code> and <code>TemporalSegment</code>: The smaller is the one that starts first. If equals, the smaller is the first that stops
 * too. If equals, they are equals (in comparison terms, non consistent with <code>equals</code> of <code>Resource</code>). <br />
 * For <code>SpatialSegment</code>: The smaller is the one with the less <code>Coordinate</code>s. If equals, we process all the coordinates to find the first
 * difference between Xs and then Ys (<code>Coordinate</code> by <code>Coordinate</code>). The smaller is the smaller using this first difference. If all number
 * (x and y) in <code>Coordinate</code>s are equals, then <code>Segment</code>s are equals (in comparison terms, non consistent with <code>equals</code> of
 * <code>Resource</code>). <br />
 * For <code>SpatioTemporalSegment</code> we first use the same process than for <code>TemporalSegment</code> (using the timestamp property). If the timestamp
 * are we compare them using the same process than for <code>SpatialSegment</code>.<br />
 * Finally, for <code>TrackSegment</code>, the less Segment it contains, the "smaller" it is. Then one by one contained Segment are compared and the first
 * difference is the comparison rasult.<br />
 * 
 * 
 * @warning Comparison is inconsistent with <code>equals()</code> since no <code>equals()</code> was generated by JAX-WS and it will use the default one from
 *          <code>Object</code>.
 * @author Cassidian WebLab Team
 * @date 2008-01-03
 */
public class SegmentComparator implements Serializable, Comparator<Segment> {

	private static final long serialVersionUID = 121L;


	@Override
	public int compare(final Segment seg1, final Segment seg2) {
		if (seg1 == seg2) {
			return 0;
		}
		if (seg1.getClass().equals(seg2.getClass())) {
			return this.compareSameTypes(seg1, seg2);
		}
		// Not of the same types
		return this.compareDifferentTypes(seg1, seg2);
	}


	/**
	 * Here segment are not of the same type. The ordering is:
	 * <ol>
	 * <li>LinearSegment</li>
	 * <li>TemporalSegment</li>
	 * <li>SpatialSegment</li>
	 * <li>SpatioTemporalSegment</li>
	 * <li>TrackSegment</li>
	 * </ol>
	 * 
	 * @param seg1
	 *            The left segment
	 * @param seg2
	 *            The right one
	 * @return a negative integer, zero, or a positive integer as the first argument is less than, equal to, or greater than the second
	 */
	private int compareDifferentTypes(final Segment seg1, final Segment seg2) {
		if (seg1.getClass().equals(LinearSegment.class)) {
			// seg1 is of the smallest type
			return -1;
		}
		if (seg2.getClass().equals(LinearSegment.class)) {
			// seg2 is of the smallest type
			return +1;
		}
		// LinearSegment are excluded, smallest type is Temporal
		if (seg1.getClass().equals(TemporalSegment.class)) {
			// seg1 is of the smallest type
			return -1;
		}
		if (seg2.getClass().equals(TemporalSegment.class)) {
			// seg2 is of the smallest type
			return +1;
		}
		// LinearSegment and Temporal are excluded, smallest type is SpatialSegment
		if (seg1.getClass().equals(SpatialSegment.class)) {
			// seg1 is of the smallest type
			return -1;
		}
		if (seg2.getClass().equals(SpatialSegment.class)) {
			// seg2 is of the smallest type
			return +1;
		}
		// LinearSegment, Temporal and SpatialSegment are excluded, smallest type is SpatioTemporalSegment
		if (seg1.getClass().equals(SpatioTemporalSegment.class)) {
			// seg1 is of the smallest type
			return -1;
		}
		if (seg2.getClass().equals(SpatioTemporalSegment.class)) {
			// seg2 is of the smallest type
			return +1;
		}
		// LinearSegment, Temporal, SpatialSegment and SpatioTemporalSegment are excluded, smallest type is TrackSegment
		if (seg1.getClass().equals(TrackSegment.class)) {
			// seg1 is of the smallest type
			return -1;
		}
		if (seg2.getClass().equals(TrackSegment.class)) {
			// seg2 is of the smallest type
			return +1;
		}
		// Segments are of types that are not in the default model.
		LogFactory.getLog(SegmentComparator.class).debug(seg1.getClass().getName() + " not defined. Comparison done through hashCode.");
		return Integer.signum(seg1.hashCode() - seg2.hashCode());
	}


	/**
	 * Compares its two arguments for order. Returns a negative integer, zero, or a positive integer as the first argument is less than, equal to, or greater
	 * than the second.
	 * 
	 * Here we assume that both are of the same type. If not we will face a ClassCastException. If the segment types are not known, the comparison is done by a
	 * simple hashcode comparison.
	 * We also assume that none of them is null.
	 * 
	 * @param seg1
	 *            The segment left
	 * @param seg2
	 *            The segment right
	 * @return a negative integer, zero, or a positive integer as the first argument is less than, equal to, or greater than the second
	 * @see #compare(Segment, Segment)
	 */
	private int compareSameTypes(final Segment seg1, final Segment seg2) {
		if (seg1 instanceof LinearSegment) {
			final LinearSegment lSeg1 = (LinearSegment) seg1;
			final LinearSegment lSeg2 = (LinearSegment) seg2;
			final int startComparison = Integer.signum(lSeg1.getStart() - lSeg2.getStart());
			// If both starts at the same character, we compare their end.
			if (startComparison == 0) {
				return Integer.signum(lSeg1.getEnd() - lSeg2.getEnd());
			}
			return startComparison;
		}
		if (seg1 instanceof TemporalSegment) {
			final TemporalSegment tSeg1 = (TemporalSegment) seg1;
			final TemporalSegment tSeg2 = (TemporalSegment) seg2;
			final int startComparison = Integer.signum(tSeg1.getStart() - tSeg2.getStart());
			// If both starts at the same time, we compare their end.
			if (startComparison == 0) {
				return Integer.signum(tSeg1.getEnd() - tSeg2.getEnd());
			}
			return startComparison;
		}
		if (seg1 instanceof SpatioTemporalSegment) {
			final SpatioTemporalSegment stSeg1 = (SpatioTemporalSegment) seg1;
			final SpatioTemporalSegment stSeg2 = (SpatioTemporalSegment) seg2;
			final int timestampComparison = Integer.signum(stSeg1.getTimestamp() - stSeg2.getTimestamp());
			// If both are taken at the same time, we compare their coordinates
			if (timestampComparison == 0) {
				return this.compareCoordinates(stSeg1.getCoordinate(), stSeg2.getCoordinate());
			}
			return timestampComparison;
		}
		if (seg1 instanceof SpatialSegment) {
			final SpatialSegment sSeg1 = (SpatialSegment) seg1;
			final SpatialSegment sSeg2 = (SpatialSegment) seg2;
			return this.compareCoordinates(sSeg1.getCoordinate(), sSeg2.getCoordinate());
		}
		if (seg1 instanceof TrackSegment) {
			return this.compareTracks((TrackSegment) seg1, (TrackSegment) seg2);
		}

		// Segments are of types that are not in the default model.
		LogFactory.getLog(SegmentComparator.class).debug(seg1.getClass().getName() + " not defined. Comparison done through hashCode.");
		return Integer.signum(seg1.hashCode() - seg2.hashCode());
	}


	/**
	 * The TrackSegment with the less inner Segment is the smaller. Then one by one contained Segment are compared and the first difference is the comparison
	 * result.
	 * 
	 * @param seg1
	 *            The left segment
	 * @param seg2
	 *            The right segment
	 * @return a negative integer, zero, or a positive integer as the first argument is less than, equal to, or greater than the second
	 */
	private int compareTracks(final TrackSegment seg1, final TrackSegment seg2) {
		final int sizeDifference = Integer.signum(seg1.getSegment().size() - seg2.getSegment().size());
		if ((sizeDifference == 0) && (!seg2.getSegment().isEmpty())) {
			// Both are of the same size and are not empty
			int cpt = 0;
			final int size = seg2.getSegment().size();
			int innerSegmentDifference;
			do {
				innerSegmentDifference = this.compare(seg1.getSegment().get(cpt), seg2.getSegment().get(cpt));
				cpt++;
			} while ((innerSegmentDifference == 0) && (cpt < size));
			// Return the signum of the first difference between two inner segments.
			return innerSegmentDifference;
		}
		return sizeDifference;
	}


	/**
	 * Both coordinates lists are equals if (and only if) coordinates are one by one equals, i.e. same X, same Y.
	 * 
	 * @param coordinate
	 *            The left one
	 * @param coordinate2
	 *            The right one
	 * @return a negative integer, zero, or a positive integer as the first argument is less than, equal to, or greater than the second
	 * @warning The code is not safe against on the coordinates ordering.
	 */
	private int compareCoordinates(final List<Coordinate> coordinate1, final List<Coordinate> coordinate2) {
		final int sizeDifference = Integer.signum(coordinate1.size() - coordinate2.size());
		if ((sizeDifference == 0) && (!coordinate2.isEmpty())) {
			// Both are of the same size and are not empty
			int cpt = 0;
			final int size = coordinate1.size();
			int coordinateDifference;
			do {
				coordinateDifference = Float.compare(coordinate1.get(cpt).getX(), coordinate2.get(cpt).getX());
				if (coordinateDifference == 0) {
					coordinateDifference = Float.compare(coordinate1.get(cpt).getY(), coordinate2.get(cpt).getY());
				}
				cpt++;
			} while ((coordinateDifference == 0) && (cpt < size));
			// Return the signum of the first difference between two coordinates.
			return coordinateDifference;
		}
		return sizeDifference;
	}

}