/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2011 CASSIDIAN
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 * A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this library; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.core.extended.factory;

import org.ow2.weblab.core.extended.exception.WebLabRISyntaxException;
import org.ow2.weblab.core.extended.exception.WebLabResourceCreationException;
import org.ow2.weblab.core.extended.uri.WebLabRI;
import org.ow2.weblab.core.model.Resource;


/**
 * A factory to be used to create all the <code>Resource</code>s defined in the model. When creating a <code>Resource</code>, this factory will ensure that it
 * contains an <code>URI</code> compliant with <code>WebLabRI</code>. <br/>
 * We strongly encourage <b>NOT</b> to use this class in order to create non standalone resources. In fact, use this
 * class to create <code>Document</code>, <code>ComposedResource</code>, <code>Query</code> and <code>UsageContext</code>, the only standalone
 * <code>Resource</code>s. For other <code>Resource</code> s, you should
 * use other factories.
 * 
 * @see PoKFactory
 * @see MediaUnitFactory
 * @see SegmentFactory
 * @see LowLevelDescriptorFactory
 * @author Cassidian WebLab Team
 * @date 2008-01-15
 */
public class ResourceFactory extends AbstractFactory {

	/**
	 * Constructors
	 * Do not use it.
	 */
	private ResourceFactory() {
		throw new UnsupportedOperationException("This class only contains static methods; no need to instantiate it.");
	}

	/**
	 * The separator
	 */
	protected final static char SEPARATOR = '-';

	/**
	 * Use this method to create <code>Resource</code>s that have a sense alone, and that are not contained by anything
	 * (but a <code>ComposedResource</code>).<br />
	 * It's recommended to only use this class when creating <code>Document</code>, <code>ComposedResource</code>, <code>Ontology</code>, <code>Query</code>,
	 * <code>Service</code> and <code>HumanResource</code> instances.
	 * 
	 * @param idRef
	 *            Resource creator unique reference
	 * @param idRes
	 *            Resource unique reference in the creator
	 * @param resClass
	 *            The implementation class of the resource
	 * @param <T>
	 *            The <code>class</code> of the <code>Resource</code> to be instancied.F
	 * @return An instance of resourceClass with a valid WebLabRI
	 */
	public static <T extends Resource> T createResource(final String idRef, final String idRes, final Class<T> resClass) {
		final T resource;
		try {
			resource = resClass.newInstance();
		} catch (final InstantiationException ie) {
			throw new WebLabResourceCreationException(WebLabResourceCreationException.EXCEPTION + resClass.getSimpleName(), ie);
		} catch (final IllegalAccessException iae) {
			throw new WebLabResourceCreationException(WebLabResourceCreationException.EXCEPTION + resClass.getSimpleName(), iae);
		}

		try {
			final WebLabRI wri = new WebLabRI("weblab://" + idRef + "/" + idRes);
			if (wri.getFragment() != null) {
				throw new WebLabRISyntaxException(wri.toString());
			}
			resource.setUri(wri.toString());
			AbstractFactory.cleanMapping(wri.toString());
		} catch (final WebLabRISyntaxException wlrise) {
			throw new WebLabResourceCreationException(WebLabResourceCreationException.EXCEPTION + resClass.getSimpleName(), wlrise);
		}

		return resource;
	}

}
