/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2011 CASSIDIAN
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.core.extended.jaxb;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Reader;
import java.io.StringWriter;
import java.io.UnsupportedEncodingException;
import java.io.Writer;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBElement;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;
import javax.xml.bind.PropertyException;
import javax.xml.bind.Unmarshaller;
import javax.xml.namespace.QName;
import javax.xml.transform.stream.StreamSource;

import org.apache.commons.logging.LogFactory;
import org.ow2.weblab.core.extended.exception.WebLabCheckedException;
import org.ow2.weblab.core.extended.exception.WebLabUncheckedException;
import org.ow2.weblab.core.model.Resource;


/**
 * A wrapping class for a JAXB <code>Marshaller</code> and a JAXB <code>Unmarshaler</code>. Many functions are available on Writer since it's easy for the use
 * to create a Writer from a file or from an output stream. <code>new OutputStreamWriter(new FileOutputStream(file)), "UTF-8");</code> Note that you need to
 * close the I/O streams, readers and writers that you gave to this class. When you use a file, we close the opened stream.
 * 
 * @author Cassidian WebLab Team
 * @date 2008-01-15
 */
public class WebLabMarshaller {

	/**
	 * Constructors
	 */
	public WebLabMarshaller() {
		super();
	}

	/**
	 * The <code>JAXBContext</code> to be used
	 */
	private JAXBContext jc;

	/**
	 * @param qname
	 *            The <code>QName</code> of the field containing <code>resource</code>.
	 * @param resource
	 *            The <code>Resource</code> to be marshaled.
	 * @return a <code>JAXBElement</code> containing <code>resource</code>.
	 */
	private static JAXBElement<Resource> createJAXBElement(final QName qname, final Resource resource) {
		return new JAXBElement<Resource>(qname, Resource.class, resource);
	}

	/**
	 * @param qNameLocalPart
	 *            The name of the field containing <code>resource</code>.
	 * @param resource
	 *            The <code>Resource</code> to be marshaled.
	 * @return a <code>JAXBElement</code> containing <code>resource</code>.
	 */
	private static JAXBElement<Resource> createJAXBElement(final String qNameLocalPart, final Resource resource) {
		return new JAXBElement<Resource>(new QName(qNameLocalPart), Resource.class, resource);
	}

	/**
	 * @param resource
	 *            The <code>Resource</code> to be marshaled.
	 * @return a <code>JAXBElement</code> containing <code>resource</code>.
	 */
	private static JAXBElement<Resource> createResourceJAXBElement(final Resource resource) {
		return WebLabMarshaller.createJAXBElement(new QName("resource"), resource);
	}

	/**
	 * Initialises <code>jc</code> with packages as context path if needed.
	 */
	private void initJC() {
		if (this.jc == null) {
			try {
				this.jc = JAXBContext.newInstance(Resource.class);
			} catch (final JAXBException jaxbe) {
				this.reset();
				throw new WebLabUncheckedException("Unable to initialise JAXBContext", jaxbe);
			}
		}
	}

	/**
	 * Reset <code>marshaller</code>, <code>unmarshaller</code> and <code>jc</code>.
	 */
	public void reset() {
		this.jc = null;
	}

	/**
	 * Initialises the <code>JAXBContext</code> and a marshaler with packages as
	 * context, UTF-8 encoding and formatted output.
	 */
	private Marshaller createMarshaller() {
		this.initJC();
		final Marshaller marshaller;
		try {
			marshaller = this.jc.createMarshaller();
		} catch (final JAXBException jaxbe) {
			this.reset();
			throw new WebLabUncheckedException("Unable to initialise marshaler", jaxbe);
		}
		try {
			marshaller.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT, Boolean.TRUE);
			marshaller.setProperty(Marshaller.JAXB_ENCODING, "UTF-8");
		} catch (final PropertyException pe) {
			LogFactory.getLog(WebLabMarshaller.class).error("Unable to set property in marshaler", pe);
		}
		return marshaller;
	}

	/**
	 * Initialises the <code>JAXBContext</code> with packages as context path if
	 * needed and initialises the unmarshaler if needed
	 */
	private Unmarshaller createUnmarshaller() {
		this.initJC();
		final Unmarshaller unmarshaller;
		try {
			unmarshaller = this.jc.createUnmarshaller();
		} catch (final JAXBException jaxbe) {
			this.reset();
			throw new WebLabUncheckedException("Unable to initialise unmarshaler", jaxbe);
		}
		return unmarshaller;
	}

	/**
	 * @param jaxbElement
	 *            The <code>JAXBElement</code> to be marshaled
	 * @param writer
	 *            <code>Writer</code> to be written. Note that the writer need
	 *            to be closed after using this method.
	 * @param checkXML
	 *            Whether or not to clean the JAXBElement from any not
	 *            recommended XML char. Time of the process will be increased.
	 * @throws WebLabCheckedException
	 *             if a <code>JAXBException</code> occurred
	 * @see XMLStringCleaner
	 */
	private void marshal(final JAXBElement<?> jaxbElement, final Writer writer, final boolean checkXML) throws WebLabCheckedException {
		final Marshaller marshaler = this.createMarshaller();
		try {
			if (checkXML) {
				final StringWriter sw = new StringWriter();
				try {
					marshaler.marshal(jaxbElement, sw);
					writer.write(XMLStringCleaner.getXMLRecommendedString(sw.toString()));
				} catch (final IOException ioe) {
					LogFactory.getLog(WebLabMarshaller.class).error("Unable to clean xml string; Trying with the input.", ioe);
					marshaler.marshal(jaxbElement, writer);
				} catch (final JAXBException jaxbe) {
					this.reset();
					LogFactory.getLog(WebLabMarshaller.class).error("Unable to clean xml string; Trying with the input.", jaxbe);
					final Marshaller otherMarshaler = this.createMarshaller();
					otherMarshaler.marshal(jaxbElement, writer);
				} finally {
					try {
						writer.flush();
					} catch (final IOException ioe) {
						LogFactory.getLog(this.getClass()).warn("Unable to flush writer.");
					}
				}
			} else {
				marshaler.marshal(jaxbElement, writer);
			}
		} catch (final JAXBException jaxbe) {
			this.reset();
			throw new WebLabCheckedException("Unable to marshal resource: " + jaxbElement.getValue() + "in writer: " + writer, jaxbe);
		}
	}

	/**
	 * @param <T>
	 *            Class of the <code>Resource</code> to be unmarshaled
	 * @param file
	 *            <code>File</code> to be read.
	 * @param resourceClass
	 *            Class of the <code>Resource</code> to be unmarshaled. Using <code>Resource</code> will prevent from
	 *            cast errors when the
	 *            content of the <code>File</code> isn't compatible with <code>resourceClass</code>.
	 * @return an instance of <code>resourceClass</code> from the content of <code>File</code>.
	 * @throws WebLabCheckedException
	 *             if a <code>JAXBException</code> or a <code>ClassCastException</code> occurred.
	 */
	public <T extends Resource> T unmarshal(final File file, final Class<T> resourceClass) throws WebLabCheckedException {
		final T res;
		try {
			final FileInputStream fis = new FileInputStream(file);
			try {
				res = this.unmarshal(fis, resourceClass);
			} finally {
				try {
					fis.close();
				} catch (final IOException ioe) {
					LogFactory.getLog(this.getClass()).warn("Unable to close stream.", ioe);
				}
			}
		} catch (final FileNotFoundException fnfe) {
			throw new WebLabCheckedException("Unable to read from file: '" + file.getAbsolutePath() + "'.", fnfe);
		}
		return res;
	}

	/**
	 * @param <T>
	 *            Class of the <code>Resource</code> to be unmarshaled
	 * @param stream
	 *            <code>InputStream</code> to be read. Note that you need to
	 *            close the stream after using this method.
	 * @param resourceClass
	 *            Class of the <code>Resource</code> to be unmarshaled. Using <code>Resource</code> will prevent from
	 *            cast errors when the
	 *            content of the <code>InputStream</code> isn't compatible with <code>resourceClass</code>.
	 * @return an instance of <code>resourceClass</code> from the content of the <code>InputStream</code>.
	 * @throws WebLabCheckedException
	 *             if a <code>JAXBException</code> or a <code>ClassCastException</code> occurred.
	 */
	public <T extends Resource> T unmarshal(final InputStream stream, final Class<T> resourceClass) throws WebLabCheckedException {
		final InputStreamReader isr;
		try {
			isr = new InputStreamReader(stream, "UTF-8");
		} catch (final UnsupportedEncodingException uee) {
			throw new WebLabCheckedException("UTF-8 not supported! Unable to marshall resource", uee);
		}
		return this.unmarshal(isr, resourceClass);
	}

	/**
	 * @param <T>
	 *            Class of the <code>Resource</code> to be unmarshaled
	 * @param reader
	 *            <code>Reader</code> to be read. Note that you need to close
	 *            the reader after using this method.
	 * @param resourceClass
	 *            Class of the <code>Resource</code> to be unmarshaled. Using <code>Resource</code> will prevent from
	 *            cast errors when the
	 *            content of the <code>Reader</code> isn't compatible with <code>resourceClass</code>.
	 * @return an instance of <code>resourceClass</code> from the content of
	 *         source.
	 * @throws WebLabCheckedException
	 *             if a <code>JAXBException</code> or a ClassCastException
	 *             occurred.
	 */
	public <T extends Resource> T unmarshal(final Reader reader, final Class<T> resourceClass) throws WebLabCheckedException {
		final Unmarshaller unmarshaler = this.createUnmarshaller();
		final T res;
		try {
			final JAXBElement<T> jaxbElement = unmarshaler.unmarshal(new StreamSource(reader), resourceClass);
			res = jaxbElement.getValue();
			if (resourceClass.isInstance(res)) {
				resourceClass.cast(res);
			} else {
				throw new ClassCastException("Unable to cast the unmarshaled Resource into a " + resourceClass.getName());
			}
		} catch (final JAXBException jaxbe) {
			this.reset();
			throw new WebLabCheckedException(jaxbe);
		} catch (final ClassCastException cce) {
			throw new WebLabCheckedException(cce);
		}
		return res;
	}

	/**
	 * @param resource
	 *            The resource to be marshaled.
	 * @param file
	 *            The <code>File</code> to be written.
	 * @throws WebLabCheckedException
	 *             if a <code>JAXBException</code> occurred.
	 */
	public void marshalResource(final Resource resource, final File file) throws WebLabCheckedException {
		try {
			final FileOutputStream fos = new FileOutputStream(file);
			try {
				this.marshalResource(resource, fos);
			} finally {
				try {
					fos.close();
				} catch (final IOException ioe) {
					LogFactory.getLog(this.getClass()).warn("Unable to close stream.", ioe);
				}
			}
		} catch (final FileNotFoundException fnfe) {
			throw new WebLabCheckedException("Unable to marshal resource in file: '" + file.getAbsolutePath() + "'.", fnfe);
		}
	}

	/**
	 * @param resource
	 *            The <code>Resource</code> to be marshaled.
	 * @param stream
	 *            The <code>OutputStream</code> to be written. Note that you
	 *            need to close the stream after using this method.
	 * @throws WebLabCheckedException
	 *             if a <code>JAXBException</code> occurred.
	 */
	public void marshalResource(final Resource resource, final OutputStream stream) throws WebLabCheckedException {
		try {
			final OutputStreamWriter osw = new OutputStreamWriter(stream, "UTF-8");
			this.marshalResource(resource, osw);
		} catch (final UnsupportedEncodingException uee) {
			throw new WebLabCheckedException("UTF-8 not supported! Unable to marshall resource", uee);
		}

	}

	/**
	 * @param resource
	 *            The <code>Resource</code> to be marshaled.
	 * @param writer
	 *            The <code>Writer</code> to be written. Note that you need to
	 *            close the writer after using this method.
	 * @throws WebLabCheckedException
	 *             if a <code>JAXBException</code> occurred.
	 */
	public void marshalResource(final Resource resource, final Writer writer) throws WebLabCheckedException {
		this.marshal(WebLabMarshaller.createResourceJAXBElement(resource), writer, false);
	}

	/**
	 * @param resource
	 *            The <code>Resource</code> to be marshaled.
	 * @param writer
	 *            The <code>Writer</code> to be written. Note that you need to
	 *            close the writer after using this method.
	 * @param checkXML
	 *            Whether or not to clean the Resource from any not recommended
	 *            XML char. Time of the process will be increased.
	 * @throws WebLabCheckedException
	 *             if a <code>JAXBException</code> occurred.
	 * @see XMLStringCleaner
	 */
	public void marshalResource(final Resource resource, final Writer writer, final boolean checkXML) throws WebLabCheckedException {
		this.marshal(WebLabMarshaller.createResourceJAXBElement(resource), writer, checkXML);
	}

	/**
	 * @param resource
	 *            The <code>Resource</code> to be marshaled.
	 * @param writer
	 *            The <code>Writer</code> to be written. Note that you need to
	 *            close the writer after using this method.
	 * @param qname
	 *            The <code>QName</code> of the field containing <code>resource</code>.
	 * @throws WebLabCheckedException
	 *             if a <code>JAXBException</code> occurred.
	 */
	public void marshal(final Resource resource, final Writer writer, final QName qname) throws WebLabCheckedException {
		this.marshal(WebLabMarshaller.createJAXBElement(qname, resource), writer, false);
	}

	/**
	 * @param resource
	 *            The <code>Resource</code> to be marshaled.
	 * @param writer
	 *            The <code>Writer</code> to be written. Note that you need to
	 *            close the writer after using this method.
	 * @param qnameLocalPart
	 *            The name of the field containing <code>resource</code>.
	 * @throws WebLabCheckedException
	 *             if a <code>JAXBException</code> occurred.
	 */
	public void marshal(final Resource resource, final Writer writer, final String qnameLocalPart) throws WebLabCheckedException {
		this.marshal(WebLabMarshaller.createJAXBElement(qnameLocalPart, resource), writer, false);
	}

	/**
	 * @see Marshaller for key values.
	 * @param key
	 *            The String key of the property to set.
	 * @param value
	 *            The Object value of the property to set.
	 * @return <code>false</code> if an <code>PropertyException</code> occurred
	 *         while setting the property.
	 */
	public boolean setMarshallerProperty(final String key, final Object value) {
		final Marshaller marshaler = this.createMarshaller();
		boolean set;
		try {
			marshaler.setProperty(key, value);
			set = true;
		} catch (final PropertyException pe) {
			set = false;
			LogFactory.getLog(WebLabMarshaller.class).error("Unable to set property in Marshaller", pe);
		}
		return set;
	}

	/**
	 * @see Marshaller for key values
	 * @param key
	 *            The String key of the property to set
	 * @param value
	 *            The Object value of the property to set
	 * @return false if an PropertyException occurred while setting the property
	 */
	public boolean setUnmarshallerProperty(final String key, final Object value) {
		final Unmarshaller unmarshaler = this.createUnmarshaller();
		boolean set;
		try {
			unmarshaler.setProperty(key, value);
			set = true;
		} catch (final PropertyException pe) {
			set = false;
			LogFactory.getLog(WebLabMarshaller.class).error("Unable to set property in Unmarshaller", pe);
		}
		return set;
	}
}
