/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2011 Cassidian, an EADS company
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */
package org.ow2.weblab.core.extended.ontologies;

import org.ow2.weblab.core.extended.exception.WebLabUncheckedException;
import org.ow2.weblab.core.model.Resource;
import org.ow2.weblab.core.model.Segment;


/**
 * Contains every properties that are defined in the model.owl file of the WebLab model project.
 * 
 * This version includes annotable object of the model i.e. Resources and their relationships. It also provides some properties to be used in annotations.
 * 
 * @author WebLab Cassidian team
 * @date 2011-06-08
 */
public class WebLabModel {


	/**
	 * Constructors Do not use it.
	 */
	private WebLabModel() {
		throw new UnsupportedOperationException("This class only contains static methods; no need to instantiate it.");
	}



	/**
	 * Namespace of the WebLab model ontology.
	 */
	public final static String NAMESPACE = "http://weblab.ow2.org/core/1.2/ontology/model#";

	/**
	 * Preferred prefix to be used to represent the WebLab Model ontology namespace.
	 */
	public final static String PREFIX = "model";


	/*
	 * URI of the classes
	 */


	/**
	 * URI of the Annotation object
	 */
	public final static String ANNOTATION = "http://weblab.ow2.org/core/1.2/ontology/model#Annotation";

	/**
	 * URI of the Audio object
	 */
	public final static String AUDIO = "http://weblab.ow2.org/core/1.2/ontology/model#Audio";

	/**
	 * URI of the ComposedQuery object
	 */
	public final static String COMPOSED_QUERY = "http://weblab.ow2.org/core/1.2/ontology/model#ComposedQuery";

	/**
	 * URI of the ComposedResource object
	 */
	public final static String COMPOSED_RESOURCE = "http://weblab.ow2.org/core/1.2/ontology/model#ComposedResource";

	/**
	 * URI of the Coordinate object
	 */
	public final static String COORDINATE = "http://weblab.ow2.org/core/1.2/ontology/model#Coordinate";

	/**
	 * URI of the Document object
	 */
	public final static String DOCUMENT = "http://weblab.ow2.org/core/1.2/ontology/model#Document";

	/**
	 * URI of the ElementaryQuery object
	 */
	public final static String ELEMENTARY_QUERY = "http://weblab.ow2.org/core/1.2/ontology/model#ElementaryQuery";

	/**
	 * URI of the Image object
	 */
	public final static String IMAGE = "http://weblab.ow2.org/core/1.2/ontology/model#Image";

	/**
	 * URI of the LinearSegment object
	 */
	public final static String LINEAR_SEGMENT = "http://weblab.ow2.org/core/1.2/ontology/model#LinearSegment";

	/**
	 * URI of the LowLevelDescriptor object
	 */
	public final static String LOW_LEVEL_DESCRIPTOR = "http://weblab.ow2.org/core/1.2/ontology/model#LowLevelDescriptor";

	/**
	 * URI of the MediaUnit object
	 */
	public final static String MEDIA_UNIT = "http://weblab.ow2.org/core/1.2/ontology/model#MediaUnit";

	/**
	 * URI of the PieceOfKnowledge object
	 */
	public final static String PIECE_OF_KNOWLEDGE = "http://weblab.ow2.org/core/1.2/ontology/model#PieceOfKnowledge";

	/**
	 * URI of the Query object
	 */
	public final static String QUERY = "http://weblab.ow2.org/core/1.2/ontology/model#Query";

	/**
	 * URI of the Resource object
	 */
	public final static String RESOURCE = "http://weblab.ow2.org/core/1.2/ontology/model#Resource";

	/**
	 * URI of the ResultSet object
	 */
	public final static String RESULT_SET = "http://weblab.ow2.org/core/1.2/ontology/model#ResultSet";

	/**
	 * URI of the Segment object
	 */
	public final static String SEGMENT = "http://weblab.ow2.org/core/1.2/ontology/model#Segment";

	/**
	 * URI of the SimilarityQuery object
	 */
	public final static String SIMILARITY_QUERY = "http://weblab.ow2.org/core/1.2/ontology/model#SimilarityQuery";

	/**
	 * URI of the SpatialSegment object
	 */
	public final static String SPATIAL_SEGMENT = "http://weblab.ow2.org/core/1.2/ontology/model#SpatialSegment";

	/**
	 * URI of the SpatioTemporalSegment object
	 */
	public final static String SPATIO_TEMPORAL_SEGMENT = "http://weblab.ow2.org/core/1.2/ontology/model#SpatioTemporalSegment";

	/**
	 * URI of the StringQuery object
	 */
	public final static String STRING_QUERY = "http://weblab.ow2.org/core/1.2/ontology/model#StringQuery";

	/**
	 * URI of the TemporalSegment object
	 */
	public final static String TEMPORAL_SEGMENT = "http://weblab.ow2.org/core/1.2/ontology/model#TemporalSegment";

	/**
	 * URI of the Text object
	 */
	public final static String TEXT = "http://weblab.ow2.org/core/1.2/ontology/model#Text";

	/**
	 * URI of the TrackSegment object
	 */
	public final static String TRACK_SEGMENT = "http://weblab.ow2.org/core/1.2/ontology/model#TrackSegment";

	/**
	 * URI of the Video object
	 */
	public final static String VIDEO = "http://weblab.ow2.org/core/1.2/ontology/model#Video";


	/*
	 * URI of the properties
	 */


	/**
	 * URI of the hasAnnotation property. Links a Resource to one of its Annotation.
	 */
	public final static String HAS_ANNOTATION = "http://weblab.ow2.org/core/1.2/ontology/model#hasAnnotation";

	/**
	 * URI of the hasCoordinate property. Links a SpatialSegment to one of its Coordinate.
	 */
	public final static String HAS_COORDINATE = "http://weblab.ow2.org/core/1.2/ontology/model#hasCoordinate";

	/**
	 * URI of the hasLowLevelDescriptor property. Links a Resource to one of its LowLevelDescriptor.
	 */
	public final static String HAS_LOW_LEVEL_DESCRIPTOR = "http://weblab.ow2.org/core/1.2/ontology/model#hasLowLevelDescriptor";

	/**
	 * URI of the hasPok property. Links a ResultSet to its PieceOfKnowledge.
	 */
	public final static String HAS_POK = "http://weblab.ow2.org/core/1.2/ontology/model#hasPok";

	/**
	 * URI of the hasSegment property. Links a MediaUnit to one of its Segment.
	 */
	public final static String HAS_SEGMENT = "http://weblab.ow2.org/core/1.2/ontology/model#hasSegment";

	/**
	 * URI of the isComposedBySegment property. Links a TrackSegment to one of its Segment.
	 */
	public final static String IS_COMPOSED_BY_SEGMENT = "http://weblab.ow2.org/core/1.2/ontology/model#isComposedBySegment";

	/**
	 * URI of the inComposedQuery property. Links a Query to its parent ComposedQuery.
	 */
	public final static String IN_COMPOSED_QUERY = "http://weblab.ow2.org/core/1.2/ontology/model#inComposedQuery";

	/**
	 * URI of the isComposedByQuery property. Links a ComposedQuery to one of its Query.
	 */
	public final static String IS_COMPOSED_BY_QUERY = "http://weblab.ow2.org/core/1.2/ontology/model#isComposedByQuery";

	/**
	 * URI of the inComposedResource property. Links a resource to its parent ComposedResource or SimilarityQuery
	 */
	public final static String IN_COMPOSED_RESOURCE = "http://weblab.ow2.org/core/1.2/ontology/model#inComposedResource";

	/**
	 * URI of the isComposedByResource property. Links a ComposedResource or a SimilarityQuery to one of its Resource.
	 */
	public final static String IS_COMPOSED_BY_RESOURCE = "http://weblab.ow2.org/core/1.2/ontology/model#isComposedByResource";

	/**
	 * URI of the inDocument property. Links a MediaUnit to its parent Document.
	 */
	public final static String IN_DOCUMENT = "http://weblab.ow2.org/core/1.2/ontology/model#inDocument";

	/**
	 * URI of the isComposedByMediaUnit property. Links a Document to one of its sub-MediaUnit.
	 */
	public final static String IS_COMPOSED_BY_MEDIA_UNIT = "http://weblab.ow2.org/core/1.2/ontology/model#isComposedByMediaUnit";

	/**
	 * URI of the inResultSet property. Links a resource to its parent ResultSet.
	 */
	public final static String IN_RESULT_SET = "http://weblab.ow2.org/core/1.2/ontology/model#inResultSet";

	/**
	 * URI of the isComposedByResourceResultSet property. Links a ResultSet to one of its Resource.
	 */
	public final static String IS_COMPOSED_BY_RESOURCE_RESULT_SET = "http://weblab.ow2.org/core/1.2/ontology/model#isComposedByResourceResultSet";

	/**
	 * URI of the endAt property. The end value of a LinearSegment (char in UTF-8) or a TemporalSegment (milliseconds).
	 */
	public final static String END_AT = "http://weblab.ow2.org/core/1.2/ontology/model#endAt";

	/**
	 * URI of the startAt property. The start value of a LinearSegment (char in UTF-8) or a TemporalSegment (milliseconds).
	 */
	public final static String START_AT = "http://weblab.ow2.org/core/1.2/ontology/model#startAt";

	/**
	 * URI of the hasTimeStamp property. The timestamp of a SpatioTemporalSegment
	 */
	public final static String HAS_TIMESTAMP = "http://weblab.ow2.org/core/1.2/ontology/model#hasTimeStamp";

	/**
	 * URI of the hasWeight property. A weight used for Query combination.
	 */
	public final static String HAS_WEIGHT = "http://weblab.ow2.org/core/1.2/ontology/model#hasWeight";

	/**
	 * URI of the hasData property. Should be an ARFF content for a LowLevelDescriptor and a RDF/XML content for a PoK.
	 */
	public final static String HAS_DATA = "http://weblab.ow2.org/core/1.2/ontology/model#hasData";

	/**
	 * URI of the hasUri property. The unique identifier of a Resource or a Segment.
	 */
	public final static String HAS_URI = "http://weblab.ow2.org/core/1.2/ontology/model#hasUri";

	/**
	 * URI of the hasRequest property. The string query in a free text syntax.
	 */
	public final static String HAS_REQUEST = "http://weblab.ow2.org/core/1.2/ontology/model#hasRequest";

	/**
	 * URI of the hasContent property. The optional content as a bytes array.
	 */
	public final static String HAS_CONTENT = "http://weblab.ow2.org/core/1.2/ontology/model#hasContent";

	/**
	 * URI of the hasTextContent property. The String content of the text. Might be empty, in that case an annotation referring to the whole large content
	 * should be present.
	 */
	public final static String HAS_TEXT_CONTENT = "http://weblab.ow2.org/core/1.2/ontology/model#hasTextContent";

	/**
	 * URI of the hasOperator property. This is a simple restriction of String to reduce the values to the three boolean operators AND/OR/NOT. Note that in a
	 * n-ary context, the unary operator NOT is considered as a AND NOT.
	 */
	public final static String HAS_OPERATOR = "http://weblab.ow2.org/core/1.2/ontology/model#hasOperator";

	/**
	 * URI of the hasAbscissa property. Abscissa value in pixels, with 0 at the left border. Other measure references are possible.
	 */
	public final static String HAS_ABSCISSA = "http://weblab.ow2.org/core/1.2/ontology/model#hasAbscissa";

	/**
	 * URI of the hasOrdonate property. Ordinate value in pixels, with 0 at the bottom border. Other measure references are possible.
	 */
	public final static String HAS_ORDINATE = "http://weblab.ow2.org/core/1.2/ontology/model#hasOrdinate";



	// Methods

	/**
	 * Use this to get the WebLab ontology owl:Class URI corresponding to a Resource class.
	 * 
	 * @see #getClassFromUri(String)
	 * @see #getUriFromSegment(Class)
	 * @param resourceClass
	 *            the java Class
	 * @return the ontology URI class
	 */
	public static String getUriFromResource(final Class<? extends Resource> resourceClass) {
		return WebLabModel.NAMESPACE + resourceClass.getSimpleName();
	}

	/**
	 * Use this to get the WebLab ontology owl:Class URI corresponding to a Segment class.
	 * 
	 * @see #getClassFromUri(String)
	 * @see #getUriFromResource(Class)
	 * @param segmentClass
	 *            the java Class
	 * @return the ontology URI class
	 */
	public static String getUriFromSegment(final Class<? extends Segment> segmentClass) {
		return WebLabModel.NAMESPACE + segmentClass.getSimpleName();
	}

	/**
	 * Inverse method of {@link #getUriFromResource(Class)} and {@link #getUriFromSegment(Class)}
	 * 
	 * @see #getUriFromResource(Class)
	 * @see #getUriFromSegment(Class)
	 * @param uri
	 *            The URI denoting a WebLab class in the ontology
	 * @return The java class
	 * @throws ClassNotFoundException If the URI does not represent a class that exists.
	 * @throws ClassCastException
	 *             If the URI does not represent a Resource or a Segment class defined in the model.
	 */
	public static Class<?> getClassFromUri(final String uri) throws ClassNotFoundException {
		if ((uri == null) || (uri.length() <= WebLabModel.NAMESPACE.length()) || !uri.startsWith(WebLabModel.NAMESPACE)) {
			throw new WebLabUncheckedException("Uri: " + uri + " was not a WebLab class. ");
		}
		final String className = '.' + uri.substring(WebLabModel.NAMESPACE.length());

		final Class<?> clazz = ClassLoader.getSystemClassLoader().loadClass(Resource.class.getPackage().getName() + className);
		if (Resource.class.isAssignableFrom(clazz)) {
			return clazz.asSubclass(Resource.class);
		} else if (Segment.class.isAssignableFrom(clazz)) {
			return clazz.asSubclass(Segment.class);
		}
		throw new ClassCastException("The URI '" + uri + "' denotes the class " + clazz.getCanonicalName() + " which is neither a resource nor a segment.");
	}

}
