/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2011 CASSIDIAN
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.core.extended.util;

import java.io.OutputStream;
import java.io.PrintWriter;
import java.io.StringReader;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.ow2.weblab.core.extended.exception.WebLabCheckedException;
import org.ow2.weblab.core.extended.jaxb.WebLabMarshaller;
import org.ow2.weblab.core.model.ComposedQuery;
import org.ow2.weblab.core.model.ComposedResource;
import org.ow2.weblab.core.model.Document;
import org.ow2.weblab.core.model.Resource;
import org.ow2.weblab.core.model.ResultSet;
import org.ow2.weblab.core.model.SimilarityQuery;


/**
 * A class that contain a useful method when wanting to visualise the content and the structure of a <code>Resource</code>
 * 
 * @author Cassidian WebLab Team
 */
public class ResourceUtil {

	/**
	 * Constructors Do not use it.
	 */
	private ResourceUtil() {
		throw new UnsupportedOperationException("This class only contains static methods; no need to instantiate it.");
	}

	/**
	 * Simple visualisation method that enable to print a user friendly (non-xml) description of <code>res</code>. Uses
	 * a tree representation. For each type of <code>Resource</code> that might me
	 * contained by the given <code>Resource</code> and it's contained <code>Resource</code>s print their <code>URI</code> and their class simple name. For any <code>Resource</code>, it's contained
	 * <code>Resource</code> s might be:
	 * <ul>
	 * <li>the <code>Annotations</code>,</li>
	 * <li>the <code>LowLevelDescriptors</code>,</li>
	 * <li>the <code>Resources</code> if it's a <code>ComposedResource, a Similarity or a ResultSet</code>,</li>
	 * <li>the <code>MediaUnits</code> if it's a <code>Document</code></li>
	 * <li>the <code>Queries</code> if it's a <code>ComposedQuery</code></li>
	 * <li>the <code>PieceOfKnowledge</code> if it's a <code>ResultSet</code>.</li>
	 * </ul>
	 * 
	 * @param res
	 *            The <code>Resource</code> to be printed
	 * @param stream
	 *            The <code>OutputStream</code> to write into
	 */
	public static void printTree(final Resource res, final OutputStream stream) {
		final PrintWriter printWriter = new PrintWriter(stream, true);
		ResourceUtil.printTree(res, printWriter, "");
	}

	/**
	 * Simply transforms <code>resource</code> into an XML <code>String</code>. Use the {@link WebLabMarshaller} for
	 * more configuration, and usage with files or
	 * streams.
	 * 
	 * @see WebLabMarshaller
	 * @param resource
	 *            The <code>Resource</code> to be transformed into an XML <code>String</code>.
	 * @return A <code>String</code> containing the XML view of the <code>Resource</code>
	 * @throws WebLabCheckedException
	 *             If a <code>JAXBException</code> occurred when marshaling the resource.
	 */
	public static String saveToXMLString(final Resource resource) throws WebLabCheckedException {
		final StringWriter sw = new StringWriter();
		new WebLabMarshaller().marshalResource(resource, sw);
		return sw.toString();
	}

	/**
	 * Simply transform the XML <code>String</code> into a <code>Resource</code>. Use the {@link WebLabMarshaller} for
	 * more configuration, and usage with files
	 * or streams.
	 * 
	 * @param xml
	 *            The XML <code>String</code> representing the Resource
	 * @return The <code>Resource</code> defined in the XML <code>String</code>.
	 * @throws WebLabCheckedException
	 *             If a <code>JAXBException</code> occurred when marshaling the resource.
	 */
	public static Resource loadFromXMLString(final String xml) throws WebLabCheckedException {
		return new WebLabMarshaller().unmarshal(new StringReader(xml), Resource.class);
	}

	/**
	 * @param res
	 *            The <code>Resource</code> to be printed.
	 * @param writer
	 *            The <code>PrintWriter</code> to write into.
	 * @param gap
	 *            The space to be added at the beginning of the lines.
	 */
	private static void printTree(final Resource res, final PrintWriter writer, final String gap) {
		writer.println(gap + res.getUri() + "  " + res.getClass().getSimpleName());
		for (final Resource subResource : ResourceUtil.getDirectSubResources(res)) {
			ResourceUtil.printTree(subResource, writer, gap + "  ");
		}

	}

	/**
	 * Retrieve the direct sub <code>Resource</code>s of <code>res</code>. In fact, it retrieves:
	 * <ul>
	 * <li>the <code>Annotations</code>,</li>
	 * <li>the <code>LowLevelDescriptors</code>,</li>
	 * <li>the <code>Resources</code> if it's a <code>ComposedResource, a Similarity or a ResultSet</code>,</li>
	 * <li>the <code>MediaUnits</code> if it's a <code>Document</code></li>
	 * <li>the <code>Queries</code> if it's a <code>ComposedQuery</code></li>
	 * <li>the <code>PieceOfKnowledge</code> if it's a <code>ResultSet</code>.</li>
	 * </ul>
	 * 
	 * @param res
	 *            The <code>Resource</code> the resource to process.
	 * @return The <code>List</code> containing <code>Resource</code>s defined just before.
	 */
	private static List<Resource> getDirectSubResources(final Resource res) {
		final List<Resource> subResources = new ArrayList<Resource>();
		if (res.isSetAnnotation()) {
			subResources.addAll(res.getAnnotation());
		}
		if (res.isSetDescriptor()) {
			subResources.addAll(res.getDescriptor());
		}
		if (res instanceof ResultSet) {
			final ResultSet rs = (ResultSet) res;
			if (rs.isSetPok()) {
				subResources.add(rs.getPok());
			}
			if (rs.isSetResource()) {
				subResources.addAll(rs.getResource());
			}
		} else if (res instanceof ComposedResource) {
			subResources.addAll(((ComposedResource) res).getResource());
		} else if (res instanceof ComposedQuery) {
			subResources.addAll(((ComposedQuery) res).getQuery());
		} else if (res instanceof SimilarityQuery) {
			subResources.addAll(((SimilarityQuery) res).getResource());
		} else if (res instanceof Document) {
			final Document doc = (Document) res;
			if (doc.isSetMediaUnit()) {
				subResources.addAll(doc.getMediaUnit());
			}
		}
		return subResources;
	}

	/**
	 * @param res
	 *            The <code>Resource</code> to retrieve children.
	 * @return The sub<code>Resource</code> <code>List</code> without <code>res</code> it self.
	 */
	public static List<Resource> getSubResources(final Resource res) {
		return new ArrayList<Resource>(ResourceUtil.getSubResources1(res).keySet());
	}

	/**
	 * @param res
	 *            The <code>Resource</code> to retrieve children.
	 * @return An ordered Map containing every subresource of <code>res</code> as keys and parent of those resources as
	 *         values.
	 */
	private static Map<Resource, Resource> getSubResources1(final Resource res) {
		final LinkedHashMap<Resource, Resource> resources = new LinkedHashMap<Resource, Resource>();
		final List<Resource> subs = ResourceUtil.getDirectSubResources(res);
		for (final Resource sub : subs) {
			resources.put(sub, res);
			resources.putAll(ResourceUtil.getSubResources1(sub));
		}
		return resources;
	}



	/**
	 * The same as <code>ResourceUtil.getSubResources(Resource)</code>, but with a filter on the <code>Resource</code> class to be retrieved by the <code>List</code>. I.e. the only retrieved
	 * <code>Resource</code>s are those that are
	 * instances of <code>resClass</code>.
	 * 
	 * @param <T>
	 *            Subclass of Resource to be retrieved
	 * @see ResourceUtil#getSubResources(Resource)
	 * @param res
	 *            The <code>Resource</code> to retrieve children.
	 * @param resClass
	 *            The <code>Class</code> of the only children you need in result <code>List</code>.
	 * @return The subResource <code>List</code>.
	 */
	public static <T extends Resource> List<T> getSelectedSubResources(final Resource res, final Class<T> resClass) {
		final List<T> result = new ArrayList<T>();
		for (final Resource r : ResourceUtil.getSubResources1(res).keySet()) {
			if (resClass.isInstance(r)) {
				final T muAsT = resClass.cast(r);
				result.add(muAsT);
			}
		}
		return result;
	}


	/**
	 * The same as <code>ResourceUtil.getSubResources(Resource, Class&lt;T&gt)</code>; But in
	 * addition to the sub-resources by themselves, this method return an reference to the resource
	 * containing every subresource.
	 * 
	 * @param <T>
	 *            Subclass of Resource to be retrieved
	 * @see ResourceUtil#getSelectedSubResources(Resource, Class)
	 * @param res
	 *            The <code>Resource</code> to retrieve children.
	 * @param resClass
	 *            The <code>Class</code> of the only children you need in result <code>List</code>.
	 * @return The subResource <code>Map</code> with their parents.
	 */
	public static <T extends Resource> Map<T, Resource> getSelectedSubResourcesMap(final Resource res, final Class<T> resClass) {
		final Map<T, Resource> result = new LinkedHashMap<T, Resource>();
		for (final Entry<Resource, Resource> entry : ResourceUtil.getSubResources1(res).entrySet()) {
			if (resClass.isInstance(entry.getKey())) {
				final T muAsT = resClass.cast(entry.getKey());
				result.put(muAsT, entry.getValue());
			}
		}
		return result;
	}

}
