/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2010 CASSIDIAN
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.core.extended.comparator;

import java.io.Serializable;
import java.util.Comparator;

import org.ow2.weblab.core.extended.exception.WebLabNotYetImplementedException;
import org.ow2.weblab.core.model.LinearSegment;
import org.ow2.weblab.core.model.Segment;
import org.ow2.weblab.core.model.SpatialSegment;
import org.ow2.weblab.core.model.TemporalSegment;


/**
 * A <code>Comparator</code> for <code>Segment</code>s. <br />
 * First of all <code>Segment</code> are ordered using their type: <code>LinearSegment</code>,
 * <code>TemporalSegment</code>, <code>SpatialSegment</code> an then any other <code>Segment</code>. <br />
 * Then <code>Segment</code>s of same types are compare together. <br />
 * For <code>LinearSegment</code> and <code>TemporalSegment</code>: The smaller is the one that starts first. If equals,
 * the smaller is the first that stops too. If equals, they are equals (in comparison terms, non consistent with
 * <code>equals</code> of <code>Resource</code>). <br />
 * For <code>SpatialSegment</code>: The smaller is the one with the less <code>Coordinate</code>s. If equals, we process
 * all the coordinates to find the first difference between Xs and then Ys (<code>Coordinate</code> by
 * <code>Coordinate</code>). The smaller is the smaller using this first difference. If all number (x and y) in
 * <code>Coordinate</code>s are equals, then <code>Segment</code>s are equals (in comparison terms, non consistent with
 * <code>equals</code> of <code>Resource</code>). <br />
 * 
 * @warning Comparison is inconsistent with <code>equals()</code> since no <code>equals()</code> was generated by JAX-WS
 *          and it will use the default one from <code>Object</code>.
 * @author Cassidian WebLab Team
 * @date 2008-01-03
 */
public class SegmentComparator implements Serializable, Comparator<Segment> {

	private static final long serialVersionUID = 12L;

	/*
	 * (non-Javadoc)
	 * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
	 */
	@Override
	public int compare(final Segment seg1, final Segment seg2) {
		if (seg1 instanceof LinearSegment) {
			return SegmentComparator.compareLinear((LinearSegment) seg1, seg2);
		} else if (seg2 instanceof LinearSegment) {
			return -SegmentComparator.compareLinear((LinearSegment) seg2, seg1);
		} else if (seg1 instanceof TemporalSegment) {
			// seg2 cannot be a Linear
			return SegmentComparator.compareTemporal((TemporalSegment) seg1, seg2);
		} else if (seg2 instanceof TemporalSegment) {
			// seg1 cannot be a Linear
			return -SegmentComparator.compareTemporal((TemporalSegment) seg2, seg1);
		} else if (seg1 instanceof SpatialSegment) {
			return SegmentComparator.compareSpacial((SpatialSegment) seg1, seg2);
		} else if (seg2 instanceof SpatialSegment) {
			return -SegmentComparator.compareSpacial((SpatialSegment) seg2, seg1);
		} else {
			throw new WebLabNotYetImplementedException("These Segments types " + seg1.getClass().getCanonicalName() + " " + seg2.getClass().getCanonicalName()
					+ " are not yet handle in SegmentComparator");
		}
	}

	/**
	 * If two <code>Segment</code>s are spatial one, calls <code>compareSpatials</code>, else <code>seg1</code> is
	 * assume to be
	 * smaller than <code>seg2</code> since it can't be neither a Linear nor a
	 * Temporal one.
	 * 
	 * @param seg1
	 *            The <code>SpatialSegment</code>
	 * @param seg2
	 *            Any type of <code>Segment</code> except </code>LinearSegment</code>
	 *            and <code>TemporalSegment</code>
	 * @return a negative integer, zero, or a positive integer as the first
	 *         argument is less than, equal to, or greater than the second
	 */
	private static int compareSpacial(final SpatialSegment seg1, final Segment seg2) {
		final int toRet;
		if (seg2 instanceof SpatialSegment) {
			toRet = SegmentComparator.compareSpatials(seg1, (SpatialSegment) seg2);
		} else {
			toRet = -1;
		}
		return toRet;
	}

	/**
	 * The smaller is the one with the less <code>Coordinate</code>s. If
	 * equals, we process all the coordinates to find the first difference
	 * between xs and then ys (<code>Coordinate</code> by <code>Coordinate</code>). The smaller is the smaller using
	 * this first
	 * difference. If all number (x and y) in <code>Coordinate</code>s are
	 * equals, then <code>Segment</code>s are equals (in comparison terms,
	 * inconsistent with <code>equals</code> of <code>Resource</code>).
	 * 
	 * @param seg1
	 *            The <code>SpatialSegment</code>
	 * @param seg2
	 *            Other <code>SpatialSegment</code>
	 * @warning Inconsistent with <code>equals</code>
	 * @return a negative integer, zero, or a positive integer as the first
	 *         argument is less than, equal to, or greater than the second
	 */
	private static int compareSpatials(final SpatialSegment seg1, final SpatialSegment seg2) {
		float diff = seg1.getCoordinate().size() - seg2.getCoordinate().size();
		if (diff == 0) {
			for (int i = 0; i < seg1.getCoordinate().size(); i++) {
				diff = seg1.getCoordinate().get(i).getX() - seg2.getCoordinate().get(i).getX();
				if (diff != 0)
					break;
				diff = seg1.getCoordinate().get(i).getY() - seg2.getCoordinate().get(i).getY();
				if (diff != 0)
					break;
			}
		}
		return (int) Math.signum(diff);
	}

	/**
	 * If two segments are temporal one, calls <code>compareTemporals</code>,
	 * else <code>seg1</code> is assume to be smaller than <code>seg2</code> since it can't be a Linear one.
	 * 
	 * @param seg1
	 *            The <code>TemporalSegment</code>
	 * @param seg2
	 *            Any type of <code>Segment</code> except <code>LinearSegment</code>
	 * @return a negative integer, zero, or a positive integer as the first
	 *         argument is less than, equal to, or greater than the second
	 */
	private static int compareTemporal(final TemporalSegment seg1, final Segment seg2) {
		final int toRet;
		if (seg2 instanceof TemporalSegment) {
			toRet = SegmentComparator.compareTemporals(seg1, (TemporalSegment) seg2);
		} else {
			toRet = -1;
		}
		return toRet;
	}

	/**
	 * The smaller is the one that starts first. If equals, the smaller is the
	 * first that stops too. If equals, they are equals (in comparison terms,
	 * non consistent with equals of <code>Resource</code>).
	 * 
	 * @param seg1
	 *            The <code>TemporalSegment</code>
	 * @param seg2
	 *            Other <code>TemporalSegment</code>
	 * @warning Inconsistent with <code>equals</code>
	 * @return a negative integer, zero, or a positive integer as the first
	 *         argument is less than, equal to, or greater than the second
	 */
	private static int compareTemporals(final TemporalSegment seg1, final TemporalSegment seg2) {
		final int diff = seg1.getStart() - seg2.getStart();
		final int toRet;
		if (diff == 0) {
			toRet = (int) Math.signum(seg1.getEnd() - seg2.getEnd());
		} else {
			toRet = (int) Math.signum(diff);
		}
		return toRet;
	}

	/**
	 * If two segments are linear one, calls <code>compareLinears</code>,
	 * else <code>seg1</code> is assume to be smaller than <code>seg2</code>.
	 * 
	 * @param seg1
	 *            The <code>LinearSegment</code>
	 * @param seg2
	 *            Any type of <Segment</code>
	 * @return a negative integer, zero, or a positive integer as the first
	 *         argument is less than, equal to, or greater than the second
	 */
	private static int compareLinear(final LinearSegment seg1, final Segment seg2) {
		final int toRet;
		if (seg2 instanceof LinearSegment) {
			toRet = SegmentComparator.compareLinears(seg1, (LinearSegment) seg2);
		} else {
			toRet = -1;
		}
		return toRet;
	}

	/**
	 * The smaller is the one that starts first. If equals, the smaller is the
	 * first that stops too. If equals, they are equals (in comparison terms,
	 * non consistent with equals of <code>Resource</code>).
	 * 
	 * @param seg1
	 *            The <code>LinearSegment</code>
	 * @param seg2
	 *            The other <code>LinearSegment</code>
	 * @warning Inconsistent with <code>equals</code>
	 * @return a negative integer, zero, or a positive integer as the first
	 *         argument is less than, equal to, or greater than the second
	 */
	private static int compareLinears(final LinearSegment seg1, final LinearSegment seg2) {
		final int toRet;
		final int diff = seg1.getStart() - seg2.getStart();
		if (diff == 0) {
			toRet = (int) Math.signum(seg1.getEnd() - seg2.getEnd());
		} else {
			toRet = (int) Math.signum(diff);
		}
		return toRet;
	}

}