/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2010 CASSIDIAN
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.core.extended.factory;

import org.ow2.weblab.core.extended.exception.WebLabRISyntaxException;
import org.ow2.weblab.core.extended.exception.WebLabResourceCreationException;
import org.ow2.weblab.core.model.Annotation;
import org.ow2.weblab.core.model.Resource;


/**
 * To create <code>Annotation</code>s we strongly encourage to use this
 * class. It will enable the creation of an <code>Annotation</code> having an <code>URI</code> compliant with the
 * <code>WebLabRI</code> recommendations. <br />
 * Since an <code>Annotation</code> hasn't any sense without being attached to
 * a <code>Resource</code>, we only provide a method for creating <code>Annotation</code>s that takes a
 * <code>Resource</code> in
 * parameter. As a result, the created <code>Annotation</code> is added to the <code>Annotation</code> <code>List</code>
 * of the <code>Resource</code>.
 * 
 * @author EADS WebLab Team
 * @date 2008-01-18
 */
public class AnnotationFactory extends AbstractFactory {

	/**
	 * Constructors
	 * Do not use it.
	 */
	private AnnotationFactory() {
		throw new UnsupportedOperationException("This class only contains static methods; no need to instantiate it.");
	}

	/**
	 * /**
	 * Automatically set a valid <code>WebLabRI</code> to the created <code>Annotation</code> using the
	 * <code>Resource</code> in parameter
	 * to generate an unique id. Created <code>Annotation</code> will be added
	 * to <code>resource</code>.
	 * 
	 * @param resource
	 *            the <code>Resource</code> that will contains the created <code>Annotation</code>
	 * @return a new <code>Annotation</code> with a valid <code>WebLabRI</code> and added to <code>resource</code>
	 */
	public final static Annotation createAndLinkAnnotation(Resource resource) {
		Annotation annotation = new Annotation();
		try {
			annotation.setUri(AbstractFactory.createUniqueURIFrom(resource, false, true));
			AbstractFactory.addAnnotation(annotation, resource, resource);
		} catch (final WebLabRISyntaxException wlrise) {
			throw new WebLabResourceCreationException(WebLabResourceCreationException.EXCEPTION + "Annotation", wlrise);
		}
		return annotation;
	}

}