/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2010 CASSIDIAN
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.core.extended.factory;

import org.ow2.weblab.core.extended.exception.WebLabResourceCreationException;
import org.ow2.weblab.core.extended.exception.WebLabUncheckedException;
import org.ow2.weblab.core.model.LinearSegment;
import org.ow2.weblab.core.model.MediaUnit;
import org.ow2.weblab.core.model.Segment;
import org.ow2.weblab.core.model.Text;


/**
 * To create <code>Segment</code>s we strongly encourage to use this class. It will enable the creation of a
 * <code>Segment</code> having an <code>URI</code> compliant with the <code>WebLabRI</code> recommendations. <br />
 * Since an <code>Segment</code> hasn't any sense without being attached to a <code>MediaUnit</code>, we only provide a
 * method for creating <code>Segment</code> s that takes a <code>MediaUnit</code> in parameter. As a result, the created
 * <code>Segment</code> (an instance of the <code>Segment</code> class in
 * parameter) is added to the <code>Segment</code> <code>List</code> of the <code>MediaUnit</code>.
 * 
 * @author Cassidian WebLab Team
 * @date 2008-01-18
 */
public class SegmentFactory extends AbstractFactory {

	/**
	 * Constructors Do not use it.
	 */
	private SegmentFactory() {
		throw new UnsupportedOperationException("This class only contains static methods; no need to instantiate it.");
	}

	/**
	 * Look in <code>text</code> if a <code>LinearSegment</code> already exists at <code>start</code> and
	 * <code>end</code>. If it's the case, returns it.
	 * Otherwise, it creates a <code>LinearSegment</code> using {@link #createAndLinkSegment(MediaUnit, Class)} method,
	 * and setting the <code>start</code> and <code>end</code>.
	 * 
	 * @param text
	 *            The <code>Text</code> containing the <code>Segment</code>
	 * @param start
	 *            char index of the <code>Segment</code> start
	 * @param end
	 *            char index of the <code>Segment</code> end
	 * @return Either a newly created <code>LinearSegment</code> or the existing one at the same position in the
	 *         <code>Text</code>.
	 */
	public final static LinearSegment createAndLinkLinearSegment(final Text text, final int start, final int end) {
		if (text == null) {
			throw new WebLabUncheckedException("Argument was null");
		}
		if (start >= end) {
			throw new WebLabUncheckedException("Starting index was bigger than ending one");
		}

		LinearSegment theSeg = null;
		for (final Segment seg : text.getSegment()) {
			if (seg instanceof LinearSegment) {
				final LinearSegment linSeg = (LinearSegment) seg;
				if (linSeg.getStart() == start && linSeg.getEnd() == end) {
					theSeg = linSeg;
					// Just to win time, no matter to remove break
					break;
				}
			}
		}

		if (theSeg == null) {
			theSeg = SegmentFactory.createAndLinkSegment(text, LinearSegment.class);
			theSeg.setStart(start);
			theSeg.setEnd(end);
		}

		return theSeg;
	}

	/**
	 * Automatically set a valid <code>WebLabRI</code> to the created <code>Segment</code> using the
	 * <code>MediaUnit</code> to generate an unique id. Created <code>Segment</code> will be added to the
	 * <code>MediaUnit</code>.
	 * 
	 * @param <T>
	 *            The subclass of <code>Segment</code> to be created
	 * @param mediaUnit
	 *            the <code>MediaUnit</code> that will contains the created <code>Segment</code>
	 * @param segClass
	 *            The class of the <code>Segment</code> to be created
	 * @return a new instance of <code>segClass</code> with a valid <code>WebLabRI</code> and added to the
	 *         <code>MediaUnit</code>
	 */
	public final static <T extends Segment> T createAndLinkSegment(final MediaUnit mediaUnit, final Class<T> segClass) {
		final T segment;
		try {
			segment = segClass.newInstance();
		} catch (final InstantiationException ie) {
			throw new WebLabResourceCreationException(WebLabResourceCreationException.EXCEPTION + segClass.getName(), ie);
		} catch (final IllegalAccessException iae) {
			throw new WebLabResourceCreationException(WebLabResourceCreationException.EXCEPTION + segClass.getName(), iae);
		}
		segment.setUri(AbstractFactory.createUniqueURIFrom(mediaUnit, true, false));
		AbstractFactory.addSegment(segment, mediaUnit, mediaUnit);

		return segment;
	}

}
