/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2010 CASSIDIAN
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.core.extended.util;

import java.io.StringReader;
import java.io.StringWriter;

import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMResult;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;

import org.ow2.weblab.core.extended.exception.WebLabCheckedException;
import org.ow2.weblab.core.extended.exception.WebLabUncheckedException;
import org.ow2.weblab.core.extended.ontologies.RDF;
import org.ow2.weblab.core.model.PieceOfKnowledge;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * Uses to manipulate <code>PieceOfKnowledge</code> data field with a <code>String</code>.
 * 
 * @author Cassidian WebLab Team
 */
public class PoKUtil {

	/**
	 * Constructors Do not use it.
	 */
	private PoKUtil() {
		throw new UnsupportedOperationException("This class only contains static methods; no need to instantiate it.");
	}

	private final static String START_ELEMENT = "<data>";

	private final static String END_ELEMENT = "</data>";

	/**
	 * Returns a <code>String</code> containing the RDF statements of the <code>PieceOfKnowledge</code>.
	 * 
	 * @param pok
	 *            <code>PieceOfKnowledge</code> you want to get the <code>String</code> content
	 * @return The <code>String</code> content of the <code>PieceOfKnowledge</code>
	 */
	public static String getPoKData(final PieceOfKnowledge pok) {
		/*
		 * if the annotation is empty, return an empty String.
		 */
		if (pok.getData() == null) {
			return "";
		}

		final Element el = ((Element) pok.getData());
		if (el.getFirstChild() == null) {
			return "";
		}

		boolean rdfRDFFound = false;
		final StringBuilder sb = new StringBuilder();
		final NodeList nodes = el.getChildNodes();
		for (int k = 0; k < nodes.getLength(); k++) {
			final Node domElement = nodes.item(k);
			if (domElement.getNodeType() == Node.TEXT_NODE && domElement.getNodeValue().trim().length() == 0) {
				// Just a bad indentation. No matter. Skipping it.
				continue;
			}
			if (domElement.getNodeType() == Node.TEXT_NODE) {
				// Someone added texts directly in the Node ?
				throw new WebLabUncheckedException("PieceOfKnowledge contains a text node. Is it really valid RDF/XML?");
			}
			if (rdfRDFFound) {
				throw new WebLabUncheckedException("PieceOfKnowledge contains a least a node in addition to a rdf:RDF one. It is not allowed in RDF/XML.");
			}

			final String fullName = domElement.getNamespaceURI() + domElement.getLocalName();
			if (fullName.compareToIgnoreCase(RDF.NAMESPACE + "rdf") == 0) {
				try {
					sb.append(PoKUtil.extractRDFXML(domElement));
				} catch (final TransformerException te) {
					throw new WebLabUncheckedException("Unable to transform data field into a String. PieceOfKnowledge is corrupted!", te);
				}
				rdfRDFFound = true;
			} else if (fullName.compareToIgnoreCase(RDF.NAMESPACE + "description") == 0) {
				try {
					sb.append(PoKUtil.extractRDFXML(domElement));
				} catch (final TransformerException te) {
					throw new WebLabUncheckedException("Unable to transform data field into a String. PieceOfKnowledge is corrupted!", te);
				}
			} else {
				throw new WebLabUncheckedException("PieceOfKnowledge contains a least a non rdf node. It is not allowed in RDF/XML.");
			}
		}
		return sb.toString();
	}

	private static String extractRDFXML(final Node domElement) throws TransformerException {
		final Transformer trans = TransformerFactory.newInstance().newTransformer();
		/*
		 * removes xml declaration
		 */
		trans.setOutputProperty(OutputKeys.OMIT_XML_DECLARATION, "yes");
		trans.setOutputProperty(OutputKeys.ENCODING, "UTF-8");
		trans.setOutputProperty(OutputKeys.INDENT, "true");

		final StringWriter strW = new StringWriter();
		trans.transform(new DOMSource(domElement), new StreamResult(strW));
		return strW.toString();
	}

	/**
	 * Set the <code>PieceOfKnowledge</code> data using a <code>String</code>, the parameter have to be a valid XML <code>String</code>. If the parameter don't
	 * start with '<data' it add it automatically.
	 * 
	 * @param pok
	 *            the <code>PieceOfKnowledge</code> you want to modify
	 * @param pokStr
	 *            contains the RDF/XML <code>String</code>
	 * @param strict
	 *            whether or not to be string and block some bad RDF/XML data
	 * @throws WebLabCheckedException
	 *             if the <code>String</code> parameter is not XML valid; and also, if strict and the <code>String</code> parameter is not RDF/XML valid
	 */
	public static void setPoKData(final PieceOfKnowledge pok, final String pokStr, final boolean strict) throws WebLabCheckedException {

		final String tmp = (START_ELEMENT + pokStr.trim() + END_ELEMENT).replaceAll(">\\s+<", "><");
		final DOMResult domRes = new DOMResult();
		final StringReader strW = new StringReader(tmp);
		final Transformer trans;
		try {
			trans = TransformerFactory.newInstance().newTransformer();
			trans.transform(new StreamSource(strW), domRes);
		} catch (final TransformerConfigurationException tce) {
			throw new WebLabUncheckedException(tce);
		} catch (final TransformerException te) {
			throw new WebLabCheckedException("Unvalid String parameter:'" + tmp + "'.", te);
		}
		pok.setData(((Document) domRes.getNode()).getDocumentElement());
		if (strict) {
			getPoKData(pok);
		}
	}

	/**
	 * Same as <code>setPoKData(pok, pokStr, true);</code>
	 * 
	 * @param pok
	 *            the <code>PieceOfKnowledge</code> you want to modify
	 * @param pokStr
	 *            contains the RDF/XML <code>String</code>
	 * @throws WebLabCheckedException
	 *             if the <code>String</code> parameter is not RDF/XML valid
	 * @see #setPoKData(PieceOfKnowledge, String, boolean)
	 */
	public static void setPoKData(final PieceOfKnowledge pok, final String pokStr) throws WebLabCheckedException {
		setPoKData(pok, pokStr, true);
	}

}
