/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2009 EADS DEFENCE AND SECURITY SYSTEMS
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.core.helper.impl;

import java.util.List;

import org.ow2.weblab.core.model.Resource;

/**
 * Selection functions for triples retrieval and WebLab model association.
 * 
 */
public interface AdvancedSelector {
	
	/**
	 * Find Triplets Subject, Predicate, Object) in the resource 
	 * according to a filter. If the filter is null then results of
	 * findInResource(Resource r, null, null, null) is returned.
	 * If the resource is null, then a NullPointerException is thrown. 
	 * @param resource a resource
	 * @param gc a filter (can be null)
	 * @return a list of triples
	 */
	List<WTriple> findInResource(Resource resource, Filter filter);
	
	/**
	 * Find Triplet (Subject, Predicate, Object) in the resource and 
	 * its sub elements (Segments, Annotations, sub-MediaUnit ...).
	 * If subject, predicate or object is null, it acts as a wildcard parameter.<br/>
	 * Here is an example how to retrieve all Triplets which match the property "http://www.weblab-project.org/gld-country"
	 * <br/><br/>
	 * <code>
	 * 		List&lt;WTriple&gt; results: findInResource(myResource, null, "http://www.weblab-project.org/gld-country", null);
	 * </code>
	 * If the resource is null, then a NullPointerException is thrown. 
	 * @param subject the subject that should match in the triplet (can be null)
	 * @param predicate the predicate that should match in the triplet (can be null)
	 * @param object the object that should match in the triplet (can be null)
	 * @return a list of WTriple that match the subject, predicate and object. 
	 */
	List<WTriple> findInResource(Resource resource, String subject, String predicate, String object);
	
	/**
	 * Use this function to avoid to aggregate reified annotations
	 * If the resource is null, then a NullPointerException is thrown.
	 * @param resource a resource
	 * @param subject the subject that should match in the triplet (can be null)
	 * @param predicate the predicate that should match in the triplet (can be null)
	 * @param object the object that should match in the triplet (can be null)
	 * @param reif if true, reified triple will not be aggregated
	 * @return a list of WTriple that match the subject, predicate and object.
	 */
	List<WTriple> findInResource(Resource resource, String subject, String predicate, String object, boolean reif);
	
	/**
	 * Build a collection of objects with given properties.<br/>
	 * Let's say you search for : searchFor(myResource, EVENT, NAME, DATE, LOCATION)
	 * where name is a type of resource and NAME, DATE and LOCATION are predicates.<br/> 
	 * Then it will return a map :<br/>
	 * <code>
	 *  {<br/>
	 *   subject0uri => { NAME => nameObject0, DATE => dateObject0, LOCATION => locationObject0 },<br/>
	 *   subject1uri => { NAME => nameObject1, DATE => dateObject1, LOCATION => locationObject1 },<br/>
	 *   subject2uri => { NAME => nameObject2, DATE => dateObject2, LOCATION => locationObject2 }<br/>
	 *  } <br/>
	 * </code>
	 * It can access any EVENT in O(1) and any predicate value in O(1).<br/>
	 * If the resource is null, then a NullPointerException is thrown.
	 * If predicates are null, then all predicates will match the search. 
	 * @param resource an annotated Resource
	 * @param predicates the list of searched predicates
	 * @return an easy and fast access structure collection of triples 
	 */
	Statements searchFor(Resource resource, String... predicates);
	
	/**
	 * If limit is true, then all the searches made with this RDFSelector
	 * will stop after the first level of annotation.<br/>
	 * Else the search will recursively browse all the searched Resource structure.
	 * By default limit is false.
	 * @param limit if true, it will only search annotation on resource else it will search until the leaf annotations
	 */
	void limitToFirstLevelAnnotation(boolean limit);
}
