/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2009 EADS DEFENCE AND SECURITY SYSTEMS
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.core.helper.impl;

import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.ow2.weblab.core.helper.impl.Rule.Domain;
import org.ow2.weblab.core.model.PieceOfKnowledge;
import org.ow2.weblab.core.model.Resource;

import com.hp.hpl.jena.graph.Graph;
import com.hp.hpl.jena.graph.Node;
import com.hp.hpl.jena.graph.Triple;
import com.hp.hpl.jena.rdf.model.Model;
import com.hp.hpl.jena.rdf.model.RDFNode;
import com.hp.hpl.jena.rdf.model.SimpleSelector;
import com.hp.hpl.jena.rdf.model.Statement;
import com.hp.hpl.jena.rdf.model.StmtIterator;
import com.hp.hpl.jena.util.iterator.ExtendedIterator;

public class ComplexTripleSelector extends TripleSelectors implements RDFRulesSelector, AdvancedSelector, org.ow2.weblab.core.helper.impl.SimpleSelector {
	
	protected transient List<Rule> rules = new LinkedList<Rule>();
	private static final Log log = LogFactory.getLog(ComplexTripleSelector.class);	
	
	public ComplexTripleSelector(){}
	
	public ComplexTripleSelector(boolean supportNullData,String... namespaces){
		super(supportNullData,namespaces);
	}
	
	/**
	 * Add a filter on an association subject/predicat
	 * @param subject
	 * @param predicat
	 */
	public void addSPRule(final String subject, final String predicat){
//		filter.addSPFilter(subject, predicat);
		rules.add(new Rule(Domain.SUBJECT,subject,Domain.PREDICATE,predicat));
	}
	
	/**
	 * Add a filter on an association subject/object
	 * @param subject
	 * @param object
	 */
	public void addSORule(final String subject, final String object){
//		filter.addSPFilter(subject, object);
		rules.add(new Rule(Domain.SUBJECT,subject,Domain.OBJECT,object));
	}
	
	/**
	 * Add a filter on an association predicat/object
	 * @param predicat
	 * @param object
	 */
	public void addPORule(final String predicat, final String object){
//		filter.addSPFilter(predicat, object);
		rules.add(new Rule(Domain.PREDICATE,predicat,Domain.OBJECT,object));
	}
	
	/**
	 * TODO: A tester
	 * @param resource
	 * @return
	 */
	public List<WTriple> findIn(Resource resource){
		List<WTriple> list = super.findInResource(resource, null, null, null); 
		LinkedHashSet<WTriple> triples = new LinkedHashSet<WTriple>();
		for(WTriple wt:list){
			triples.add(wt);
		}
		list.clear();
		list.addAll(triples);
		return list;
	}
	
	/**
	 * Faster
	 */
	protected List<WTriple> applySelection(org.w3c.dom.Node data, Node s, Node p, Node o,
			String text, Resource annotated, List<Resource> resources,
			PieceOfKnowledge annotation){
		if (rules.isEmpty()){
			return super.applySelection(data, s, p, o, text, annotated, resources, annotation);
		}
		List<WTriple> list = new LinkedList<WTriple>();
		
		
		Model model = loadModel(data);
		Graph graph = model.getGraph();
		
		StmtIterator iter = model.listStatements(
	    new SimpleSelector(null, null, (RDFNode) null) {
	        public boolean selects(Statement s) {
                Triple t = s.asTriple();
                for(Rule rule:rules){
                	if(matches(rule,t)){
                		return true;
                	}
                }
                return false;
	        }
		});
		
		while(iter.hasNext()){
			Object ob = iter.next();
			if (ob instanceof Statement){
				Triple t = ((Statement)ob).asTriple();
				Node n = t.getSubject();
				WTriple w = new WTriple(t,text, annotated,copyAndAdd(resources, annotation));
				w.setAnnotation(annotation);
				if (n.isBlank()){
					ExtendedIterator in = graph.find(null, null, n);
					while(in.hasNext()){
						Object oz = in.next();
						if (oz instanceof Triple){
							w.setReifiedWTriple(new WTriple((Triple)oz,text, annotated, resources));
						}
					}
				}
				list.add(w);
			}
		}
		model.close();
		return list;
	}

	/**
	 * Uses rules
	 */
	public Statements searchFor(Resource resource, String... predicates) {
		rules.clear();
		Statements res = new Statements(this.uris);
		log.debug("Resource : search on "+resource);
		if (predicates == null){
			List<WTriple> l = findInResource(resource, null, null, null, true);
			res.addAll(l);
			return res;
		}
		
		for(String pred:predicates){
			addPORule(pred, "(.|\\s)*");
		}
		res.addAll(findIn(resource));
		rules.clear();
		return res;
	}
	
	public List<WTriple> findInResource(Resource resource, Filter filter){
		rules.clear();
		return super.findInResource(resource, filter);
	}
	
	public List<WTriple> findInResource(Resource resource, String subject, String predicate, String object){
		rules.clear();
		return super.findInResource(resource, subject, predicate, object);
	}
	
	public List<WTriple> findInResource(Resource resource, String subject, String predicate, String object, boolean reif){
		rules.clear();
		return super.findInResource(resource, subject, predicate, object, reif);
	}
	
	public void addRule(Rule rule) {
		rules.add(rule);
	}
	
	protected boolean matches(Rule rule, Triple t) {
		String subject = t.getSubject().toString(); 
		String predicate= t.getPredicate().toString();
		String object= t.getObject().toString();
		
		String first = null;
		String second = null;
		switch (rule.first) {
			case SUBJECT:
				first = subject;
				break;
			case PREDICATE:
				first = predicate;
				break;
			case OBJECT:
				first = object;
				break;
		}
		if (rule.second != null){
			switch (rule.second) {
			case SUBJECT:
				second = subject;
				break;
			case PREDICATE:
				second = predicate;
				break;
			case OBJECT:
				second = object;
				break;
			}
		}
		if (first != null && rule.firstS.equals("(.|\\s)*") || first.matches("(?i)"+rule.firstS)){
			if (second == null){
				return true;
			}
			if(rule.secondS.equals("(.|\\s)*") || second.matches("(?i)"+rule.secondS)){
				return true;
			}
		}
		
		return false;
	}

	@Override
	public Results select(Resource resource, String... predicates) {
		rules.clear();
		Statements statements = searchFor(resource, predicates);
		return new Results(statements);
	}
}
