/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2009 EADS DEFENCE AND SECURITY SYSTEMS
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.core.helper.impl;


public final class RDFSelectorFactory {
	
	private RDFSelectorFactory(){
		throw new UnsupportedOperationException("This class only contains "
				+ "static methods; no need to instantiate it.");
	}
	
	
	/**
	 * Get a searcher in triples in RDF for literals.
	 * If you want to select objects as uri, see getSelector(boolean supportNullData, String... uris).
	 * 
	 * @return a SimpleSelector searching on literals
	 */
	public static SimpleSelector getSelector(){
		return new ComplexTripleSelector(true);
	}
	
	/**
	 * Create a rdf selector with a list of namespace.
	 * If you construct an RDFSelector with the namespace http://www.example.com in parameter,
	 * then if you search for "http://www.example.com#Object" as the object of the triple.
	 * It will look for object with an URI stating with one of the given namespace instead searching for only literals.
	 *  
	 * @param namespaces a list of namespace
	 * @return a SimpleSelector
	 */
	public static SimpleSelector getSelector(final String... namespaces){
		return new ComplexTripleSelector(true,namespaces);
	}
	
	/**
	 * Create a rdf selector with robustness to bad or null annotations with a list of namespaces.
	 * Robustness to bad annotation is desactivated by default. 
	 * If you construct an RDFSelector with the uri http://www.example.com in parameter,
	 * then if you search for "http://www.example.com#Object" as the object of the triple.
	 * It will look for object with an URI stating with one of the given namespace instead searching for only literals.
	 *  
	 * @param supportNullData if true null data annotation are ignored else an error is thrown.  
	 * @param uris a list of namespace
	 * @return a AdvancedSelector
	 */
	public static AdvancedSelector getSelector(final boolean supportNullData,final String... namespaces){
		return new ComplexTripleSelector(supportNullData,namespaces);
	}
	
	/**
	 * Get a searcher in triples in RDF for literals.<br/>
	 * You can add rules on tuples:<br/>
	 * <ul>
	 * <li>subject/predicate</li>
	 * <li>predicate/object</li>
	 * <li>subject/object</li>
	 * </ul>
	 * <br/>
	 * You can also use regular expression in rules. For instance:<br/>
	 * <code>
	 * RDFRulesSelector rrs = RDFSelectorFactory.getRuleSelector();<br/>
	 * rss.addPORule("http://www.weblab-project.org/entity/.*","(.|\\s)*");<br/>
	 * List<WTriple> results = rss.findIn(myResource);<br/>
	 * </code>
	 * <br/>
	 * If you want to select objects as uri, see getRuleSelector(boolean supportNullData, String... uris).
	 * 
	 * @param supportNullData if true null data annotation are ignored else an error is thrown.  
	 * @param namespaces a list of namespace
	 * @return a RDFRulesSelector
	 */
	public static RDFRulesSelector getSelectorByRules(final boolean supportNullData,final String... namespaces){
		return new ComplexTripleSelector(supportNullData,namespaces);
	}
}
