/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2009 EADS DEFENCE AND SECURITY SYSTEMS
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.core.helper.impl;

import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;

import javax.xml.parsers.ParserConfigurationException;

import org.apache.commons.logging.LogFactory;
import org.ow2.weblab.core.extended.exception.WebLabCheckedException;
import org.ow2.weblab.core.helper.impl.parser.Parsers;
import org.ow2.weblab.core.helper.impl.parser.TypeParser;
import org.ow2.weblab.core.model.PieceOfKnowledge;
import org.ow2.weblab.core.model.Resource;

/**
 * An efficient structure to gather and access triples and their 
 * subject/predicates/values
 * 
 * This structure supports a list of values if multiple values 
 * are found for a given subject and predicate.
 * 
 * @author Arnaud
 *
 */
public class Statements extends LinkedHashMap<String, IPredicateValuePairs> {
	
	/**
	 * An ascending comparator the value "a" will be after the value "b"
	 */
	public static final Comparator<String> ASC = ascComparator();
	
	/**
	 * A descending comparator the value "a" will be before the value "b"
	 */
	public static final Comparator<String> DESC = descComparator();
	
	protected Map<String, Resource> resourcesMap = new HashMap<String, Resource>();
	
	protected Map<String, Resource> annotatedOn = new HashMap<String, Resource>();
	
	protected List<String> namespaces = new LinkedList<String>();
	
	public Statements(String... namespaces){
		for(String namespace:namespaces){
			this.namespaces.add(namespace);
		}
	}
	
	/**
	 * Add a list of WTriple in ths Statements
	 * @param wtl a list of Wtriple
	 */
	public void addAll(List<WTriple> wtl){
		for(WTriple wt:wtl){
			add(wt);
		}
	}
	
	/**
	 * Return the WebLab Resource associated with the given URI.
	 * If it was found during the search.
	 * @param uri  an URI
	 * @return a WebLab Resource
	 */
	public Resource getResource(String uri){
		return resourcesMap.get(uri);
	}
	
	/**
	 * Returns a resource on which a statement with this uri has been found  
	 * @param uri
	 * @return a WebLab Resource annotated with a statement concerning this uri (subject or object) 
	 */
	public Resource getAnnotatedResourceWith(String uri){
		return annotatedOn.get(uri);
	}
	
	protected void add(WTriple wt){
		WTriple wrei = wt.getReifiedWTriple(); 
		String resubject = wrei == null?null:""+wrei.getSubject();
		String subject = resubject == null ?wt.getSubject():resubject;
//		String predicate = wt.getPredicate();
		String object = wt.getObject();
		
		Resource rsubject = wrei != null?
				wrei.getSubjectResource():wt.getSubjectResource();
		Resource robject = wt.getObjectResource();
		
		if (rsubject != null){
			resourcesMap.put(subject, rsubject);
		}
		if (robject != null){
			resourcesMap.put(object, robject);
		}
		
		IPredicateValuePairs value = get(subject);
//		Object objectvalue = object;s
		if (value == null){
			value = new AdvancedLinkedHashMap(this);
		}
//		else{
//			// Multiple values support for the same subject and predicate
//			Object o = value.getValue(pred);
//			List<String> l = null;
//			if (o != null){
//				l = new LinkedList<String>();
//				if (o instanceof List){
//					l = (List<String>)o;
//				}else{
//					l.add(o.toString());
//				}
//				l.add(object);
//				ov = l;
//				if(l.size() == 1){
//					ov = l.iterator().next();
//				}
//			}
//		}
		Resource res = annotatedOn.get(subject);
		if (res == null){
			annotatedOn.put(subject,wt.getAnnotatedOn());
		}
		((AdvancedLinkedHashMap)value).put(wt);
		put(subject, value);
	}
	
	public String toString(){
		StringBuffer res = new StringBuffer();
		for(String uri:keySet()){
			res.append(uri+" : \n");
			IPredicateValuePairs values = get(uri);
			for(String pred:values.allPredicates()){
				res.append(pred+" = "+values.getValue(pred)+"\n");
			}
			res.append("\n");
		}
		return res.toString();
	}
	
	/**
	 * Returns a list of URI in the order of predicates sorted by a comparator.
	 * If the value is not defined for a subject/predicate, its uri will be at the end if 
	 * <code>defined</code> is false, else it will not be added in the final list. 
	 * @param predicate the predicate on which value will be sorted 
	 * @param comparable the sorter
	 * @param defined if false uri without predicate defined will be added at the end of the list, else it will be ignored 
	 * @return a sorted list of URIs
	 */
	public LinkedList<String> sortBy(String predicate, Comparator<String> comparable, boolean defined){
		LinkedList<String> listURI = new LinkedList<String>();
		LinkedList<String> nullValues = new LinkedList<String>();
		TreeMap<String, List<String>> tree = new TreeMap<String, List<String>>(comparable);
		for(String uri:super.keySet()){
			Object value = get(uri).getValue(predicate);
			if (value != null){
				List<String> suri = tree.get(value);
				if (suri == null){
					suri = new LinkedList<String>();
				}
				suri.add(uri);
				tree.put(value.toString(), suri);
			}else if (defined){
				nullValues.add(uri);
			}
			
		}
		
		for(List<String> l:tree.values()){
			listURI.addAll(l);
		}
		listURI.addAll(nullValues);
		return listURI;
	}
	
	/**
	 * Returns a list of URI in the order of predicates sorted by a comparator.
	 * If the value is not defined for a suject/predicate, its uri will be at the end. 
	 * @param predicate the predicate on which value will be sorted 
	 * @param comparable the sorter
	 * @return a sorted list of URIs
	 */
	public LinkedList<String> sortBy(String predicate, Comparator<String> comparable){
		return sortBy(predicate, comparable, false);
	}
	
	/**
	 * A descending comparator the value "a" will be before the value "b"
	 * @return a descending comparator
	 */
	private static Comparator<String> descComparator(){
		return new Comparator<String>(){

			public int compare(String o1, String o2) {
				return o1.compareTo(o2);
			}
			
		};
		
	}
	
	/**
	 * An ascending comparator the value "a" will be after the value "b"
	 * @return an ascending comparator
	 */
	private static Comparator<String> ascComparator(){
		return new Comparator<String>(){

			public int compare(String o1, String o2) {
				return o2.compareTo(o1);
			}
			
		};
	}
	
	/**
	 * Returns the first value corresponding to the predicate given.
	 * TODO: rewrite using getTypedValue(...)
	 * @param subject optional, use null as wildcard
	 * @param predicate the uri of predicate
	 * @param locale the locale, optional set to null as wildcard
	 * @return a value
	 */
	public String getFirstValue(String subject, String predicate, Locale locale){
		if (isEmpty()){
			return null;
		}
		Object value = null;
		if (subject == null){
			for(String uri:keySet()){
				value = get(uri).getValue(predicate);
				if (value != null){
					
					break;
				}
			}
		}else{
			value = get(subject).getValue(predicate);
		}
		String result = null;
		if (value instanceof List<?>) {
			List<String> labelList = (List<String>) value;
			for (String label : labelList) {
				if(locale == null){
					return clean(label);
				}
				if (label.contains("@" + locale.getLanguage())
						|| label.contains("@"
								+ locale.getLanguage().toUpperCase(locale))) {
					result = clean(label);
				}
			}
		} else if(value != null){
			if (locale == null){
				return clean(value.toString());
			}
			if (((String) value)
					.toUpperCase(locale).contains("@" + locale.getLanguage())
					|| ((String) value).toUpperCase().contains(
							"@" + locale.getLanguage().toUpperCase(locale))) {
				result = clean((String) value);
			}
		}
		return result;
	}
	
	/**
	 * Suivant partiellement la RFC 4647
	 * TODO: faites mieux...
	 * @param langedValue
	 * @return
	 */
	private String clean(String langedValue){
		if (langedValue != null && langedValue.startsWith("\"") && 
			langedValue.matches(".*\"@[a-zA-Z0-8-]+")){
			int fin = langedValue.lastIndexOf("\"");
			langedValue = langedValue.substring(1,fin);
		}
		return langedValue;
	}
	
	/**
	 * @deprecated Please avoid to use this thing 
	 */
	public LinkedHashMap<String, LinkedHashMap<String, Object>> getMap(){
		LinkedHashMap<String, LinkedHashMap<String, Object>> res = new LinkedHashMap<String, LinkedHashMap<String,Object>>();
		for(String uri:keySet()){
			LinkedHashMap<String, List<WTriple>> preds = ((AdvancedLinkedHashMap)get(uri)).getMap(); 
			LinkedHashMap<String, Object> preObj = new LinkedHashMap<String, Object>();
			for(String pred:preds.keySet()){
				List<WTriple> lw = preds.get(pred);
				if (lw != null){
					if (lw.size() == 1){
						preObj.put(pred, lw.get(0).getObject());
					}else{
						List<Object> lo = new LinkedList<Object>();
						for(WTriple w:lw){
							lo.add(w.getObject());
						}
						preObj.put(pred, lo);
					}
				}
			}
			res.put(uri, preObj);
		}
		return res;
	}
	
	/**
	 * Return a typed value rather than an object.
	 * @param <T> the type to be returned
	 * @param subject a subject
	 * @param property a property
	 * @param clazz a class of the type of the value 
	 * @return An instance of <T> if the value can be mapped on this type, else null. 
	 */
	public <T> T getTypedValue(String subject, String property, Class<T> clazz){
		if (subject == null){
			Set<String> set = getSubjects(property, ".*"); 
			if (set != null && !set.isEmpty()){
				subject = set.iterator().next();
			}
		}
		
		IPredicateValuePairs properties = get(subject);
		if (properties == null || clazz == null){
			return null;
		}
		Object value = properties.getValue(property);
		if (value == null){
			return null;
		}
		T result = null;
		if (List.class.isAssignableFrom(value.getClass())){
			List l = getTypedValues(subject, property, clazz);
			if (l.isEmpty()){
				return null;
			}else{
				return (T)l.get(0);
			}
		}
		if (clazz.isInstance(value)){
			result = (T)value;
		} else {
			result = parse(clazz, value,subject,property);
		}
		return result;
	}
	
	private <T> T parse(Class<T> clazz, Object value, String subject, String property) throws ClassCastException {
		T result = null;
		try {
			TypeParser<T> tp = Parsers.getParser(clazz);
			result = tp.parseValue(value);
			if (result != null && !clazz.isInstance(result)){
				LogFactory.getLog(Statements.class).error(new ClassCastException("the value of ("+subject+", "+property+"):"+result+" of "+result.getClass()+" is not an instance of the asked class "+clazz+" !"));
				result = null;
			}
		} catch (ParserConfigurationException e) {
			LogFactory.getLog(Statements.class).error(e);
		}
		return result;
	}
	
	/**
	 * Returns a list of typed values.
	 * @param <T> the type of the value 
	 * @param subject the uri of the subject
	 * @param property the uri of the subject
	 * @param clazz the class of the value
	 * @return a list of typed values.
	 */
	public <T> LinkedList<T> getTypedValues(String subject, String property, Class<T> clazz){
		LinkedList<T> result = new LinkedList<T>();
		
		if (subject == null){
			// We search for all subject
			if (subject == null){
				for(String isubject:keySet()){
					result.addAll(getTypedValues(isubject, property, clazz));
				}
			}
			
		}else{

			IPredicateValuePairs properties = get(subject);
			if (properties == null || clazz == null){
				return result;
			}
			Object value = properties.getValue(property);
			if (value == null){
				return result;
			}
			if (value instanceof List){
				for(Object o:(List)value){
					T parsed = parse(clazz, o,subject,property);
					if (parsed != null){
						result.add(parsed);
					}else{
						LogFactory.getLog(Statements.class).warn("Object "+o+" is not defined as a subtype of "+clazz.getName());
					}
				}
			}else{
				T parsed = parse(clazz, value,subject,property);
				if (parsed != null){
					result.add(parsed);
				}
			}
		}
		return result;
	}
	
	/**
	 * Returns all the subjects that matches the tuple (predicate,object)
	 * @param predicate
	 * @param object
	 * @return a set of subject
	 */
	public Set<String> getSubjects(String predicate, String object){
		if (predicate == null){
			return resourcesMap.keySet();
		}
		if (object == null){
			object = ".*";
		}
		Set<String> res = new HashSet<String>();
		for(String subject:keySet()){
			List<String> values = getTypedValues(subject, predicate, String.class);
			for(String value:values){
				if (value != null && value.matches(object)){
					res.add(subject);
					break;
				}
			}
			continue;
		}
		return res;
	}
	
	/**
	 * Write all statements from a Statements on the given Pok and add 
	 * these statements on this Statements.
	 * @param an other Statements
	 * @param pok a PieceOfKnoelegde on which statements will be written
	 * @param uris a mapping of uris. Uris are mapped using couples (previous_uri,future_uri)
	 * @throws WebLabCheckedException
	 */
	public void writeAndAdd(Statements map, PieceOfKnowledge pok, String... uris) throws WebLabCheckedException{
		if (map == null || pok == null){
			throw new WebLabCheckedException("Can not write the map :"+map+" on the PoK : "+pok);
		}
		resourcesMap.putAll(map.resourcesMap);
		List<WTriple> toWrite = new LinkedList<WTriple>();
		
		HashMap<String,String> mapping = new HashMap<String, String>();
		if (uris != null && uris.length%2 == 0){
			for(int i = 0;i<uris.length;i+=2){
				mapping.put(uris[i], uris[i+1]);
			}
		}
		
		for(String uri:map.keySet()){
			IPredicateValuePairs iplm = map.get(uri);
			String muri = mapping.get(uri);
			muri = muri == null?uri:muri;
			IPredicateValuePairs local_iplm = get(muri);
			if (local_iplm == null){
				local_iplm = new AdvancedLinkedHashMap(this,muri);
				put(muri,local_iplm);
			}
			
			Collection<WTriple> list = iplm.getStatements(); 
			if (list != null){
				toWrite.addAll(list);
			}
		}
		
		AdvancedLinkedHashMap.write(toWrite, pok, this, mapping);
		addAll(toWrite);
	}
}