/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2009 EADS DEFENCE AND SECURITY SYSTEMS
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.core.helper.impl;

import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import java.util.Map.Entry;

import org.ow2.weblab.core.extended.ontologies.RDF;
import org.ow2.weblab.core.model.Annotation;
import org.ow2.weblab.core.model.ComposedResource;
import org.ow2.weblab.core.model.Document;
import org.ow2.weblab.core.model.MediaUnit;
import org.ow2.weblab.core.model.PieceOfKnowledge;
import org.ow2.weblab.core.model.Resource;
import org.ow2.weblab.core.model.ResultSet;
import org.xml.sax.SAXParseException;

import com.hp.hpl.jena.graph.Graph;
import com.hp.hpl.jena.graph.Node;
import com.hp.hpl.jena.graph.Triple;
import com.hp.hpl.jena.rdf.arp.DOM2Model;
import com.hp.hpl.jena.rdf.model.Model;
import com.hp.hpl.jena.rdf.model.ModelFactory;
import com.hp.hpl.jena.util.iterator.ExtendedIterator;

/**
 * Selectors Helper to ease annotations retrieval. 
 * Annotations triples are referenced by WTriple in which
 * subject and object can be references to WebLab Resources.
 * By default, it transforms reified triples as normal triples.
 * It is useful if you just bother on data selection. 
 * Example:<br/>
 * <code>
 * 		List&lt;WTriple&gt; results = findInResource(myResource, null, "http://www.weblab-project.org/gld-country", null); 
 * </code>
 * 
 * @author EADS WebLab Team
 * @date 2009-03-11
 */
public class TripleSelectors {

	public boolean limit = false;
	public static boolean debug = false;

	public boolean supportNullData = false;

	// Multithread support
	public static boolean multi = false;
	private final Object lock = new Object();
	private final List<WTriple> syncList = Collections.synchronizedList(new LinkedList<WTriple>());
	private final static int numberofProc = Runtime.getRuntime().availableProcessors();
	private final static int multiplier = 1;
	private int numberOfJob = 0;

	public transient String[] uris = new String[0];

	public void limitToFirstLevelAnnotation(boolean limit){
		this.limit = limit;
	}

	public TripleSelectors(){};

	public TripleSelectors(boolean supportNullData, final String... uris) {
		this.supportNullData = supportNullData;
		if (uris == null){
			return;
		}
		this.uris = uris;
	}


	/**
	 * Select Triplet (Subject, Predicate, Object) in the resource and 
	 * its sub elements (Segments, Annotations, sub-MediaUnit ...).
	 * If subject, predicate or object is null, it acts as a wildcard parameter.
	 * Here is an example how to retrieve all Triplets which match the property "http://www.weblab-project.org/gld-country"
	 * <br/><br/>
	 * <code>
	 * 		List&lt;WTriple&gt; results: findInResource(myResource, null, "http://www.weblab-project.org/gld-country", null);
	 * </code>
	 *   
	 * @param subject the subject that should match in the triplet (can be null)
	 * @param predicate the predicate that should match in the triplet (can be null)
	 * @param object the object that should match in the triplet (can be null)
	 * @return a list of WTriple that match the subject, predicate and object. 
	 */
	public List<WTriple> findInResource(final Resource resource, final String subject, final String predicate, final String object){
		Node s = subject == null?Node.ANY:Node.createURI(subject);
		Node p = predicate == null?Node.ANY:Node.createURI(predicate);

		boolean buri = false;
		for(String uri:uris){
			if (object != null && object.startsWith(uri)){
				buri = true;
				break;
			}
		}

		Node o = object == null?Node.ANY:buri?Node.createURI(object):Node.createLiteral(object);
		List<WTriple> list = findInResource(resource,s,p,o, new LinkedList<Resource>());
		if (multi){
			while(numberOfJob > 0){
				Thread.yield();
				try {
					Thread.sleep(2);
				} catch (InterruptedException e) {
				}
			}
			list.addAll(syncList);
			syncList.clear();
		}
		return aggregateReified(list);
	}

	/**
	 * Select WTriple in resource according to given subject, predicate and object.
	 * Null parameters act as wildcard. It will recursively loop through Resource structure
	 * (children and their annotations) to retrieve RDF in annotations.
	 * 
	 * @param resource a WebLab Resource on which there are annotations containing RDF
	 * @param subject the subject that should match in the triplet (can be null)
	 * @param predicate the predicate that should match in the triplet (can be null)
	 * @param object the object that should match in the triplet (can be null)
	 * @param reif if true, reified triple will not be aggregated
	 * @return a list of WTriple that match the subject, predicate and object.
	 */
	public List<WTriple> findInResource(final Resource resource, final String subject, final String predicate, final String object, final boolean reif){
		Node s = subject == null?Node.ANY:Node.createURI(subject);
		Node p = predicate == null?Node.ANY:Node.createURI(predicate);

		boolean buri = false;
		for(String uri:uris){
			if (object != null && object.startsWith(uri)){
				buri = true;
				break;
			}
		}

		Node o = object == null?Node.ANY:buri?Node.createURI(object):Node.createLiteral(object);
		if (reif){
			List<WTriple> list = findInResource(resource,s,p,o, new LinkedList<Resource>());
			if (multi){
				list.addAll(syncList);
				syncList.clear();
			}
			return aggregateReified(list);
		}
		List<WTriple> list = findInResource(resource, s, p, o, new LinkedList<Resource>());
		if (multi){
			list.addAll(syncList);
			syncList.clear();
		}
		return list;
	}

	/**
	 * Select WTriple in the resource according to constrains in a filter.
	 * @param resource a WebLab Resource on which there are annotations containing RDF
	 * @param filter a set of constrains 
	 */
	public List<WTriple> findInResource(Resource resource, Filter filter){
		List<WTriple> list = findInResource(resource, null, null, null);
		if (filter == null){
			return list;
		}
		Set<WTriple> res = new HashSet<WTriple>();
		for(WTriple wt:list){
			if (!res.contains(wt) && filter.matches(wt)){
				res.add(wt);
			}
		}
		list.clear();
		list.addAll(res);
		return list;
	}

	/**
	 * Aggregate reified triple in normal ones
	 * @param list a list of WTriple reified or not
	 * @return a list of WTriple not reified. 
	 */
	private List<WTriple> aggregateReified(List<WTriple> list){
		List<WTriple> res = new LinkedList<WTriple>(list);
		LinkedHashMap<String, List<WTriple>> buffer = new LinkedHashMap<String, List<WTriple>>();
		for(WTriple wt:list){
			String pred = wt.getPredicate();
			if (pred.startsWith("http://www.w3.org/1999/02/22-rdf-syntax-ns#")){
				List<WTriple> inner = buffer.get(wt.getSubject());
				if (inner == null){
					inner = new LinkedList<WTriple>();
				}
				inner.add(wt);
				buffer.put(wt.getSubject(),inner);
				//res.remove(wt);
			}
		}

		for(Entry<String,List<WTriple>> entry: buffer.entrySet()){
			String subject = null;
			String predicate = null;
			String object = null;
			Resource subjectr = null;
			Resource objectr = null;
			final List<WTriple> later = new LinkedList<WTriple>();
			for(WTriple wt:entry.getValue()){
				String pred = wt.getPredicate(); 
				if(pred.equals("http://www.w3.org/1999/02/22-rdf-syntax-ns#predicate")){
					predicate = wt.getObject();
				}else if(pred.equals("http://www.w3.org/1999/02/22-rdf-syntax-ns#object")){
					object = wt.getObject();
					objectr = wt.getObjectResource();
				}else if(pred.equals("http://www.w3.org/1999/02/22-rdf-syntax-ns#subject")){
					subject = wt.getObject();
					subjectr = wt.getObjectResource();
				}else if(!pred.equals("http://www.w3.org/1999/02/22-rdf-syntax-ns#type")){
					later.add(wt);
				}
			}
			if (subject != null && predicate != null && object != null){
				WTriple wtriple = new WTriple(subject,predicate,object,subjectr,objectr,null);
				res.add(wtriple);
			}
			for(WTriple wt:later){
				res.add(new WTriple(subject,wt.getPredicate(),wt.getObject(),subjectr,wt.getObjectResource(),wt.getAnnotatedOn()));
			}
		}
		return res;
	}

	private List<WTriple> findInResource(Resource resource, Node subject, Node predicate, Node object, List<Resource> resources){
		List<WTriple> list = new LinkedList<WTriple>();

		if (limit){ // limit to the first level only
			Resource r = new Resource();
			r.setUri(resource.getUri());
			r.getAnnotation().addAll(resource.getAnnotation());
			resource = r;
		}

		if (resource instanceof ComposedResource){
			list.addAll(find((ComposedResource)resource, subject, predicate, object, resources));
		
		}else if (resource instanceof MediaUnit){
			list.addAll(find((MediaUnit)resource, subject, predicate, object, resources));
		}else if (resource instanceof ResultSet){
			list.addAll(find((ResultSet)resource, subject, predicate, object, resources));
		}else if (resource instanceof PieceOfKnowledge){
			list.addAll(find((PieceOfKnowledge)resource, subject, predicate, object,null, resource, resources));
		}else{
			for(Annotation a:resource.getAnnotation()){
				list.addAll(find(a,subject,predicate,object,null,resource, resources));
			}
		}

		return list;
	}

	/**
	 * Copies a list of resources and add an object
	 * @param resources list of resources
	 * @param r a resource
	 * @return a copy the list and add the resource r
	 */
	protected List<Resource> copyAndAdd(List<Resource> resources, Resource r){
		List<Resource> copie = new LinkedList<Resource>();
		copie.addAll(resources);
		copie.add(r);
		return copie;
	}

	private List<WTriple> find(ResultSet resultset,Node subject, Node predicate, Node object, List<Resource> resources) {
		List<WTriple> l = new LinkedList<WTriple>();

		PieceOfKnowledge pok = resultset.getPok();
		if (pok != null){
			l.addAll(find(pok, subject, predicate, object, null, resultset, resources));
		}

		for(Resource r:resultset.getResource()){
			l.addAll(findInResource(r, subject, predicate, object, copyAndAdd(resources, r)));
		}

		for(Annotation a:resultset.getAnnotation()){
			l.addAll(find(a,subject,predicate,object,null,resultset,copyAndAdd(resources, a)));
		}
		return l;
	}

	private List<WTriple> find(ComposedResource rc, Node subject, Node predicate, Node object, List<Resource> resources){
		List<WTriple> l = new LinkedList<WTriple>();

		for(Resource r:rc.getResource()){
			l.addAll(findInResource(r, subject, predicate, object, copyAndAdd(resources, r)));
		}

		for(Annotation a:rc.getAnnotation()){
			l.addAll(find(a,subject,predicate,object,null,rc,copyAndAdd(resources, a)));
		}
		return l;
	}



	private List<WTriple> find(MediaUnit mu, Node subject, Node predicate, Node object, List<Resource> resources){
		List<WTriple> l = new LinkedList<WTriple>();

		resources.add(mu);

		if (mu instanceof Document){
			Document doc = (Document)mu;
			for(MediaUnit mu_in_doc:doc.getMediaUnit()){
				l.addAll(find(mu_in_doc, subject, predicate, object, resources));
			}
		}
		

		for(Annotation a:mu.getAnnotation()){
			l.addAll(find(a,subject,predicate,object,null,mu, resources));
		}

		return l;
	}



	/**
	 * Select RDF Triplet in an annotation
	 * @param annotation an annotation
	 * @param subject the subject that should match in the triplet (can be null)
	 * @param predicate the predicate that should match in the triplet (can be null)
	 * @param object the object that should match in the triplet (can be null)
	 * @param text the text associated with the annotation if any
	 * @param resources the parent resources
	 * @return a list of triplet which matches the data in the annotation.
	 */
	protected List<WTriple> find(final PieceOfKnowledge annotation,
			final Node subject,final  Node predicate,final Node object,
			final String text,final  Resource annotated,final  List<Resource> resources){

		if (multi && numberOfJob < multiplier*numberofProc){
			return findMT(annotation, subject, predicate, object, text, annotated, resources);
		}

		Object node = annotation.getData();
		if (supportNullData && (node == null)){
			return new LinkedList<WTriple>();
		}

		resources.add(annotation);
		List<WTriple> l = applySelection((org.w3c.dom.Node)node, subject, predicate, object, text, annotated, resources, annotation);

		for(Annotation a:annotation.getAnnotation()){
			l.addAll(find(a, subject, predicate, object, null, annotated, copyAndAdd(resources, a)));
		}
		return l;
	}


	protected List<WTriple> findMT(final PieceOfKnowledge annotation, 
			final Node subject, final Node predicate, final Node object,
			final String text, final Resource annotated, final List<Resource> resources){
		synchronized(lock){
			numberOfJob++;
		}

		Thread worker = new Thread("MTFind#"+numberOfJob){

			public void run(){
				List<WTriple> list = new LinkedList<WTriple>(); 

				Object node = annotation.getData();
				if (supportNullData && (node == null)){
					synchronized(lock){
						numberOfJob--;
					}
					return;
				}

				resources.add(annotation);
				list.addAll(applySelection((org.w3c.dom.Node)node, subject, predicate, object, text, annotated, resources, annotation));

				for(Annotation a:annotation.getAnnotation()){
					list.addAll(find(a, subject, predicate, object, null, annotated, copyAndAdd(resources, a)));
				}
				synchronized(lock){
					syncList.addAll(list);
					numberOfJob--;
				}
			}
		};
		worker.start();

		return new LinkedList<WTriple>();
	}

	/**
	 * Load a model really fast
	 * @param data a DOM
	 * @return a Jena Model
	 */
	protected static Model loadModel(final org.w3c.dom.Node data){
		final Model model = ModelFactory.createDefaultModel();
		try {
			final DOM2Model dm = DOM2Model.createD2M(RDF.NAMESPACE, model);
			dm.load(data.getFirstChild());
			dm.close();
		} catch (SAXParseException e) {
			e.printStackTrace();
		}
		return model;
	}
	
	protected List<WTriple> applySelection(org.w3c.dom.Node data, Node s, Node p, Node o,
			String text, Resource annotated, List<Resource> resources,
			PieceOfKnowledge annotation){
		List<WTriple> triples = new LinkedList<WTriple>();

		Model model = loadModel(data);
		Graph graph = model.getGraph();

		ExtendedIterator i = graph.find(s, p, o);
		while(i.hasNext()){
			Object ob = i.next();
			if (ob instanceof Triple){
				Triple t = (Triple)ob;
				Node n = t.getSubject();
				WTriple w = new WTriple((Triple)ob,text, annotated,copyAndAdd(resources, annotation));
				w.setAnnotation(annotation);
				if (n.isBlank()){
					ExtendedIterator in = graph.find(null, null, n);
					while(in.hasNext()){
						Object oz = in.next();
						if (oz instanceof Triple){
							w.setReifiedWTriple(new WTriple((Triple)oz,text, annotated, resources));
						}
					}
				}
				triples.add(w);
			}
		}
		model.close();
		
		return triples;
	}
}