/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2009 EADS DEFENCE AND SECURITY SYSTEMS
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.core.helper.impl;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.List;

import org.ow2.weblab.core.model.PieceOfKnowledge;
import org.ow2.weblab.core.model.Resource;

import com.hp.hpl.jena.graph.Node;
import com.hp.hpl.jena.graph.Triple;

/**
 * A container class composed by a subject, a predicate and an object.
 * The subject and the object can be either an URI or an URI and a reference to 
 * a WebLab Resource. 
 *  
 * @author EADS WebLab Team
 * @date 2009-03-11
 */
public class WTriple {
	
	
	private PieceOfKnowledge annotation = null;
	private Resource annotatedOn = null;
	private WTriple reifiedWTriple = null;
	private Resource subject = null;
	private String subjectURI = null;
	private String predicate = null;
	private String objectString = null;
	private Resource object = null;
	private String annotatedText = null;
	private Triple triple;

	WTriple(String subject, String predicate, String object, Resource subjectr,Resource objectr, Resource annotatedOn){
		this.annotatedOn = annotatedOn;
		this.triple = null;
		this.subjectURI = subject;
		this.subject = subjectr;
		this.predicate = predicate;
		this.objectString = check(object);
		this.object = objectr;
		
	}

	public WTriple(Node s, Node p, Node o, String annotatedText, Resource annotatedOn, List<Resource> resources){
		this(new Triple(s,p,o),annotatedText,annotatedOn,resources);
	}

	public WTriple(Triple triple, String annotatedText, Resource annotatedOn, List<Resource> resources){
		this.annotatedOn = annotatedOn;
		this.annotatedText = annotatedText;
		this.triple = triple;
		predicate  = triple.getPredicate().toString();
		subjectURI = triple.getSubject().toString();
		objectString = check(triple.getObject().toString());

		boolean checked = false;
		try {
			new URI(objectString);
		} catch (URISyntaxException e) {
			checked = true;
		}
		if (resources == null){
			return;
		}
		for(Resource r:resources){
			if(r != null && r.getUri() != null && r.getUri().equals(subjectURI)){
				subject = r;
				if (checked){
					return;
				}
			}
			if(!checked && r != null && r.getUri() != null && r.getUri().equals(objectString)){
				object = r;
				if (subject != null){
					return;
				}
			}
		}
	}

	public void setAnnotation(PieceOfKnowledge annotation){
		this.annotation = annotation;
	}
	
	public PieceOfKnowledge getAnnotation(){
		return annotation;
	}
	
	/**
	 * Returns the PieceOfKnowledge in which this triplet is defined.  
	 * @return a PieceOfKnowledge
	 */
	public PieceOfKnowledge from(){
		return annotation;
	}
	
	public int hashCode(){
		return (subjectURI+predicate+objectString).hashCode();
	}
	
	public boolean equals(Object o){
		if (o instanceof WTriple){
			WTriple wt = (WTriple)o;
			return subjectURI.equals(wt.subjectURI) &&
			   predicate.equals(wt.predicate) &&
			   objectString.equals(wt.objectString);
		}
		return false;
		
	}
	
	private String check(String object){
		String subobject = object;
		if (object != null && 
			object.startsWith("\"") && 
			object.endsWith("\"")){
			subobject = object.substring(1, object.length()-1); 
		}
		return subobject;
	}
	
	/**
	 * Returns the Resource on which the annotation define this triple.  
	 * @return a resource
	 */
	public Resource getAnnotatedOn(){
		return this.annotatedOn;
	}
	
	protected void setReifiedWTriple(WTriple reifiedWTriple){
		this.reifiedWTriple = reifiedWTriple;
		reifiedWTriple.setAnnotation(annotation);
	}

	protected WTriple getReifiedWTriple(){
		return reifiedWTriple;
	}
	
	/**
	 * Return the text corresponding to the segment on which the annotation 
	 * @return the text for a segment
	 */
	public String getText(){
		return annotatedText;
	}

	/**
	 * Return the URI of the predicate 
	 * @return the URI
	 */
	public String getPredicate(){
		return predicate;
	}

	/**
	 * Return the Jena Triple use to build the WTriple 
	 * @return a Triple
	 */
	public Triple getTriple(){
		return triple;
	}

	/**
	 * Return the URI of the subject 
	 * @return the URI
	 */
	public String getSubject(){
		return subjectURI;
	}

	/**
	 * Returns the Resource if the subject is a WebLab Resource else null 
	 * @return a Resource or null
	 */
	public Resource getSubjectResource(){
		return subject;
	}

	/**
	 * Returns the value of the object 
	 * @return the value
	 */
	public String getObject(){
		return objectString;
	}

	/**
	 * Returns the Resource if the object is a WebLab Resource else null 
	 * @return a Resource or null
	 */
	public Resource getObjectResource(){
		return object;
	}

	public String toString(){
		return subjectURI+": "+subject+"\n"+
		annotatedText+" "+predicate+
		"\n"+objectString+": "+object; 
	}
}
