/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2009 EADS DEFENCE AND SECURITY SYSTEMS
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.core.helper;

import org.ow2.weblab.core.model.PieceOfKnowledge;

/**
 * <p>
 * API to manipulate a unique <code>PieceOfKnowledge</code> object.
 * </p>
 * <p>
 * In this new version of helpers, a lots of method have been added. They shall be easier to understand and use. No more boolean in parameters, and only the
 * much simpler part of RDF will be used (i.e. no anonymous resources and no reified statements).<br />
 * Some former methods gone <tt>Deprecated</tt>.
 * </p>
 * <p>
 * In this class a lot of abbreviation are used to reduce the length of the methods. Those convention are commons with {@link PoKHelperExtended},
 * {@link ResourceHelper} and {@link ResourceHelperExtended}.
 * <ul>
 * <li><tt>Subj</tt>: In method names, means subject.</li>
 * <li><tt>Pred</tt>: In method names, means predicate (property).</li>
 * <li><tt>Lit</tt>: In method names, means object that is literal.</li>
 * <li><tt>Res</tt>: In method names, means object that is resource.</li>
 * <li><tt>Stat</tt>: In method names, means statement.</li>
 * <li><tt>uriSubj</tt>: In parameter, means the URI of a subject that is a named resource.</li>
 * <li><tt>uriPred</tt>: In parameter, means the URI of a predicate that is a named resource.</li>
 * <li><tt>uriObj</tt>: In parameter, means the URI of an object that is a named resource.</li>
 * <li><tt>litObj</tt>: In parameter, means the value of an object that is a literal.</li>
 * </ul>
 * <p>
 * If you are performing a lots of changes on the same <code>PieceOfKnowledge</code>. Best practice is to first call {@link #setAutoCommitMode(boolean)} with
 * <code>false</code> parameter, and then at the end of your process to call the {@link #commit()} method. It will enable to win a lot of time, preventing from
 * serialising and writing the RDF/XML into the <code>data</code> field the <code>PieceOfKnowledge</code> at each writing method call.
 * </p>
 * 
 * @author gdupont, ymombrun Copyright EADS 18 Jan 2008
 * @date 2009-06-23
 * @see PoKHelperExtended
 * @see ResourceHelper
 * @see ResourceHelperExtended
 */
public interface PoKHelper extends RDFReader, RDFWriter {

	/**
	 * A call to commit is needed when <code>PoKHelper</code> do not works in "<tt>autoCommitMode</tt>". Other with this call is not needed.
	 * 
	 * @see #setAutoCommitMode(boolean)
	 */
	public void commit();

	/**
	 * Retrieve the current <code>PieceOfKnowledge</code> which is manipulated. Please not that this will not commit any changes made into the RDF/XML if
	 * <code>PoKHelper</code> do not works in "<tt>autoCommitMode</tt>" and if you did not call {@link #commit()} method.
	 * 
	 * @return The loaded <code>PieceOfKnowledge</code>.
	 */
	public PieceOfKnowledge getPoK();

	/**
	 * Fixes the current <code>PieceOfKnowledge</code> which is manipulated by the helper. Note that using an <code>PoKHelper</code> without
	 * <code>PieceOfKnowledge</code> will throw <code>RuntimeException</code>.<br />
	 * 
	 * @param pok
	 *            An existing <code>PieceOfKnowledge</code>
	 */
	public void setPoK(PieceOfKnowledge pok);

	/**
	 * In auto commit mode, remove and create operations are writing into the <code>PieceOfKnowledge</code> each time they are called. When performing a lot's
	 * of actions on the RDF, best practice is to set auto commit mode to <code>false</code> and to commit at the end of the process.
	 * 
	 * @param autoCommitMode
	 *            Whether or not to commit any changes (made by create and remove operations) on the <code>PieceOfKnowledge</code>.
	 */
	public void setAutoCommitMode(final boolean autoCommitMode);
}
