/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2009 EADS DEFENCE AND SECURITY SYSTEMS
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.core.helper;

import java.util.HashSet;
import java.util.Locale;
import java.util.Map;
import java.util.Set;

import org.apache.commons.logging.LogFactory;
import org.ow2.weblab.core.extended.exception.WebLabUncheckedException;
import org.ow2.weblab.core.extended.properties.PropertiesLoader;
import org.ow2.weblab.core.model.PieceOfKnowledge;
import org.ow2.weblab.core.model.Resource;

/**
 * Factory used to dynamically load an implementation of <code>PoKHelper</code>, <code>PoKHelperExtended</code>, <code>ResourceHelper</code> and
 * <code>ResourceHelperExtended</code>.
 * 
 * @author EADS WebLab Team
 * @see PoKHelperExtended
 * @see PoKHelper
 * @see ResourceHelper
 * @see ResourceHelperExtended
 */
public class RDFHelperFactory {

	/**
	 * Constructors Do not use it.
	 */
	private RDFHelperFactory() {
		throw new UnsupportedOperationException("This class only contains "
				+ "static methods; no need to instantiate it.");
	}

	/**
	 * The <code>PoKHelper</code> <code>class</code> to use.
	 */
	private static Class<? extends PoKHelper> pokHelpClass = null;

	/**
	 * The <code>PoKHelperExtended</code> <code>class</code> to use.
	 */
	private static Class<? extends PoKHelperExtended> pokHelpExtClass = null;

	/**
	 * The <code>PoKHelperExtended</code> name of the <code>class</code> to use.
	 */
	private static String pokHelpExtImplClassName = null;

	/**
	 * The <code>PoKHelper</code> name of the <code>class</code> to use.
	 */
	private static String pokHelpImplClassName = null;

	/**
	 * The properties file used.
	 */
	public static final String FILENAME = "rdf-helper.properties";

	/**
	 * The <code>ResourceHelper</code> <code>class</code> to use.
	 */
	private static Class<? extends ResourceHelper> resHelpClass = null;

	/**
	 * The <code>ResourceHelperExtended</code> <code>class</code> to use.
	 */
	private static Class<? extends ResourceHelperExtended> resHelpExtClass = null;

	/**
	 * The <code>ResourceHelperExtended</code> <code>class</code> to use.
	 */
	private static String resHelpExtImplClassName = null;

	/**
	 * The <code>ResourceHelper</code> <code>class</code> to use.
	 */
	private static String resHelpImplClassName = null;

	/**
	 * Creates dynamically an instance of <code>PoKHelper</code> using the <code>class</code> defined in <tt>helper.properties</tt> as the implementation class.
	 * 
	 * @return the <code>PoKHelper</code>'s implementation (without any <code>PieceOfKnowledge</code> set).
	 */
	private static PoKHelper getPOKHelp() {
		if (RDFHelperFactory.pokHelpClass == null) {
			RDFHelperFactory.pokHelpClass =
					RDFHelperFactory.loadDynamicClass(RDFHelperFactory.getPoKHelperImplClassName(), PoKHelper.class);
		}
		return RDFHelperFactory.getInstance(RDFHelperFactory.pokHelpClass);
	}

	/**
	 * Creates dynamically an instance of <code>PoKHelper</code> using the <code>PieceOfKnowledge</code> in parameter and the <code>class</code> defined in
	 * <tt>helper.properties</tt> as the implementation class.
	 * 
	 * @param pok
	 *            The <code>PieceOfKnowledge</code> to be handled with an <code>PoKHelper</code>.
	 * @return the <code>PoKHelper</code>'s implementation for this <code>PieceOfKnowledge</code>.
	 */
	public static PoKHelper getPoKHelper(PieceOfKnowledge pok) {
		PoKHelper h = RDFHelperFactory.getPOKHelp();
		h.setPoK(pok);
		return h;
	}

	/**
	 * Creates dynamically an instance of <code>PoKHelper</code> using the <code>PieceOfKnowledge</code> and the implementation <code>class</code> in parameter.
	 * 
	 * @param pok
	 *            The <code>PieceOfKnowledge</code> to be handled with an <code>PoKHelper</code>.
	 * @param pokHelperImpl
	 *            The implementation class name.
	 * @return the <code>PoKHelper</code>'s implementation for this <code>PieceOfKnowledge</code>.
	 */
	public static PoKHelper getSpecificPoKHelper(PieceOfKnowledge pok, final String pokHelperImpl) {
		final Class<? extends PoKHelper> clazz = RDFHelperFactory.loadDynamicClass(pokHelperImpl, PoKHelper.class);
		PoKHelper h = RDFHelperFactory.getInstance(clazz);
		h.setPoK(pok);
		return h;
	}

	/**
	 * Creates dynamically an instance of <code>PoKHelperExtended</code> using the <code>class</code> defined in <tt>helper.properties</tt> as the
	 * implementation class.
	 * 
	 * @return the <code>PoKHelperExtended</code>'s implementation (without any <code>PieceOfKnowledge</code> set).
	 */
	private static PoKHelperExtended getPOKHelpExt() {
		if (RDFHelperFactory.pokHelpExtClass == null) {
			RDFHelperFactory.pokHelpExtClass =
					RDFHelperFactory.loadDynamicClass(RDFHelperFactory.getPoKHelperExtendedImplClassName(),
							PoKHelperExtended.class);
		}
		return RDFHelperFactory.getInstance(RDFHelperFactory.pokHelpExtClass);
	}

	/**
	 * Creates dynamically an instance of <code>PoKHelperExtended</code> using the <code>PieceOfKnowledge</code> in parameter and the <code>class</code> defined
	 * in <tt>helper.properties</tt> as the implementation class.
	 * 
	 * @param pok
	 *            The <code>PieceOfKnowledge</code> to be handled with an <code>PoKHelperExtended</code>.
	 * @return the <code>PoKHelperExtended</code>'s implementation for this <code>PieceOfKnowledge</code>.
	 */
	public static PoKHelperExtended getPoKHelperExtended(PieceOfKnowledge pok) {
		PoKHelperExtended h = RDFHelperFactory.getPOKHelpExt();
		h.setPoK(pok);
		return h;
	}

	/**
	 * Creates dynamically an instance of <code>PoKHelperExtended</code> using the <code>PieceOfKnowledge</code> and the implementation <code>class</code> in
	 * parameter.
	 * 
	 * @param pok
	 *            The <code>PieceOfKnowledge</code> to be handled with an <code>PoKHelperExtended</code>.
	 * @param pokHelperExtImpl
	 *            The implementation class name.
	 * @return the <code>PoKHelperExtended</code>'s implementation for this <code>PieceOfKnowledge</code>.
	 */
	public static PoKHelperExtended getSpecificPoKHelperExtended(PieceOfKnowledge pok, final String pokHelperExtImpl) {
		final Class<? extends PoKHelperExtended> clazz =
				RDFHelperFactory.loadDynamicClass(pokHelperExtImpl, PoKHelperExtended.class);
		PoKHelperExtended h = RDFHelperFactory.getInstance(clazz);
		h.setPoK(pok);
		return h;
	}

	/**
	 * Use this to know the full name of the used implementation class of <code>PoKHelperExtended</code>.
	 * 
	 * @return implementation class name.
	 */
	public static String getPoKHelperExtendedImplClassName() {
		if (RDFHelperFactory.pokHelpExtImplClassName == null) {
			RDFHelperFactory.loadProperties();
		}
		return RDFHelperFactory.pokHelpExtImplClassName;
	}

	/**
	 * Use this to know the full name of the used implementation class of <code>PoKHelper</code>.
	 * 
	 * @return implementation class name.
	 */
	public static String getPoKHelperImplClassName() {
		if (RDFHelperFactory.pokHelpImplClassName == null) {
			RDFHelperFactory.loadProperties();
		}
		return RDFHelperFactory.pokHelpImplClassName;
	}

	/**
	 * Creates an instance of the parameter <code>Class</code> using the <code>newInstance()</code> method.
	 * 
	 * @param classToInstantiate
	 *            The <code>Class</code> to instantiate.
	 * @param <T>
	 *            An helper class implementation.
	 * @return An instantiated class.
	 */
	private static <T> T getInstance(final Class<T> classToInstantiate) {
		try {
			final T ret = classToInstantiate.newInstance();
			LogFactory.getLog(RDFHelperFactory.class).trace("Instantiate: " + ret);
			return ret;
		} catch (final InstantiationException ie) {
			throw new WebLabUncheckedException("Cannot instantiate " + classToInstantiate.getName() + ".", ie);
		} catch (final IllegalAccessException iae) {
			throw new WebLabUncheckedException("Cannot instantiate " + classToInstantiate.getName() + ".", iae);
		}
	}

	/**
	 * Creates dynamically an instance of <code>ResourceHelper</code> using the <code>class</code> defined in <tt>helper.properties</tt> as the implementation
	 * class.
	 * 
	 * @return The <code>ResourceHelper</code>'s implementation (without any <code>Resource</code> set).
	 */
	private static ResourceHelper getResHelp() {
		if (RDFHelperFactory.resHelpClass == null) {
			RDFHelperFactory.resHelpClass =
					RDFHelperFactory.loadDynamicClass(RDFHelperFactory.getResourceHelperImplClassName(),
							ResourceHelper.class);
		}
		return RDFHelperFactory.getInstance(RDFHelperFactory.resHelpClass);
	}

	/**
	 * Creates dynamically an instance of <code>ResourceHelperExtended</code> using the <code>class</code> defined in <tt>helper.properties</tt> as the
	 * implementation class.
	 * 
	 * @return The <code>ResourceHelperExtended</code>'s implementation (without any <code>Resource</code> set).
	 */
	private static ResourceHelperExtended getResHelpExt() {
		if (RDFHelperFactory.resHelpExtClass == null) {
			RDFHelperFactory.resHelpExtClass =
					RDFHelperFactory.loadDynamicClass(RDFHelperFactory.getResourceHelperExtendedImplClassName(),
							ResourceHelperExtended.class);
		}
		return RDFHelperFactory.getInstance(RDFHelperFactory.resHelpExtClass);
	}

	/**
	 * Creates dynamically an instance of <code>ResourceHelperExtended</code> using the <code>Resource</code> in parameter and the <code>class</code> defined in
	 * <tt>helper.properties</tt> as the implementation class.
	 * 
	 * @param resource
	 *            The <code>Resource</code> to be handled with an <code>ResourceHelperExtended</code>.
	 * @return the <code>ResourceHelperExtended</code>'s implementation for this <code>Resource</code>.
	 */
	public static ResourceHelperExtended getResourceHelperExtended(final Resource resource) {
		ResourceHelperExtended h = RDFHelperFactory.getResHelpExt();
		h.setResource(resource);
		return h;
	}

	/**
	 * Creates dynamically an instance of <code>ResourceHelperExtended</code> using the <code>Resource</code> and the implementation <code>class</code> in
	 * parameter.
	 * 
	 * @param resource
	 *            The <code>Resource</code> to be handled with an <code>ResourceHelperExtended</code>.
	 * @param resHelperExtImpl
	 *            The implementation class name.
	 * @return the <code>ResourceHelperExtended</code>'s implementation for this <code>Resource</code>.
	 */
	public static ResourceHelperExtended getSpecificResourceHelperExtended(final Resource resource,
			final String resHelperExtImpl) {
		final Class<? extends ResourceHelperExtended> clazz =
				RDFHelperFactory.loadDynamicClass(resHelperExtImpl, ResourceHelperExtended.class);
		ResourceHelperExtended h = RDFHelperFactory.getInstance(clazz);
		h.setResource(resource);
		return h;
	}

	/**
	 * Creates dynamically an instance of <code>ResourceHelper</code> using the <code>Resource</code> in parameter and the <code>class</code> defined in
	 * <tt>helper.properties</tt> as the implementation class.
	 * 
	 * @param resource
	 *            The <code>Resource</code> to be handled with an <code>ResourceHelper</code>.
	 * @return the <code>ResourceHelper</code>'s implementation for this <code>Resource</code>.
	 */
	public static ResourceHelper getResourceHelper(final Resource resource) {
		ResourceHelper h = RDFHelperFactory.getResHelp();
		h.setResource(resource);
		return h;
	}

	/**
	 * Creates dynamically an instance of <code>ResourceHelper</code> using the <code>Resource</code> and the implementation <code>class</code> in parameter.
	 * 
	 * @param resource
	 *            The <code>Resource</code> to be handled with an <code>ResourceHelper</code>.
	 * @param resHelperImpl
	 *            The implementation class name.
	 * @return the <code>ResourceHelper</code>'s implementation for this <code>Resource</code>.
	 */
	public static ResourceHelper getSpecificResourceHelper(final Resource resource, final String resHelperImpl) {
		final Class<? extends ResourceHelper> clazz =
				RDFHelperFactory.loadDynamicClass(resHelperImpl, ResourceHelper.class);
		ResourceHelper h = RDFHelperFactory.getInstance(clazz);
		h.setResource(resource);
		return h;
	}

	/**
	 * Use this to know the full name of the used implementation class of <code>ResourceHelper</code>.
	 * 
	 * @return Implementation class name.
	 */
	public static String getResourceHelperImplClassName() {
		if (RDFHelperFactory.resHelpImplClassName == null) {
			RDFHelperFactory.loadProperties();
		}
		return RDFHelperFactory.resHelpImplClassName;
	}

	/**
	 * Use this to know the full name of the used implementation class of <code>ResourceHelperExtended</code>.
	 * 
	 * @return Implementation class name.
	 */
	public static String getResourceHelperExtendedImplClassName() {
		if (RDFHelperFactory.resHelpExtImplClassName == null) {
			RDFHelperFactory.loadProperties();
		}
		return RDFHelperFactory.resHelpExtImplClassName;
	}

	/**
	 * Calls the <code>classLoader</code> to return the <code>Class</code> corresponding to the parameter.
	 * 
	 * @param className
	 *            The complete class name you want to load.
	 * @param helperType
	 *            The interface to be implemented by the helper.
	 * @return A <code>Class</code> corresponding to the class name parameter.
	 */
	private static <T> Class<? extends T> loadDynamicClass(final String className, final Class<T> helperType) {
		try {
			Class<?> impl = RDFHelperFactory.class.getClassLoader().loadClass(className);
			Class<? extends T> toReturn;
			if (helperType.isAssignableFrom(impl)) {
				toReturn = impl.asSubclass(helperType);
			} else {
				throw new ClassNotFoundException("Unable load the class: " + className + " as a "
						+ helperType.getSimpleName());
			}
			LogFactory.getLog(RDFHelperFactory.class).debug(
					"Loaded: '" + className + "'; instance of '" + helperType.getSimpleName() + "'");
			return toReturn;
		} catch (final ClassNotFoundException cnfe) {
			throw new WebLabUncheckedException("Cannot instantiate " + className + ".", cnfe);
		}
	}

	/**
	 * Loads class names from the <tt>helper.properties<tt> file.
	 * 
	 * @warning If there is more than one <code>helper.properties</code> file.
	 */
	private static void loadProperties() {
		final String pokHelpStr = PoKHelper.class.getSimpleName().toLowerCase(Locale.ENGLISH);
		final String pokHelpExtStr = PoKHelperExtended.class.getSimpleName().toLowerCase(Locale.ENGLISH);
		final String resHelpStr = ResourceHelper.class.getSimpleName().toLowerCase(Locale.ENGLISH);
		final String resHelpExtStr = ResourceHelperExtended.class.getSimpleName().toLowerCase(Locale.ENGLISH);
		Set<String> prop = new HashSet<String>();
		prop.add(pokHelpStr);
		prop.add(pokHelpExtStr);
		prop.add(resHelpStr);
		prop.add(resHelpExtStr);

		Map<String, String> values =
				PropertiesLoader.loadProperties(RDFHelperFactory.FILENAME, RDFHelperFactory.class);
		RDFHelperFactory.pokHelpImplClassName = values.get(pokHelpStr);
		RDFHelperFactory.pokHelpExtImplClassName = values.get(pokHelpExtStr);
		RDFHelperFactory.resHelpImplClassName = values.get(resHelpStr);
		RDFHelperFactory.resHelpExtImplClassName = values.get(resHelpExtStr);

	}

	/**
	 * Use this to set the class implementation name for <code>PoKHelperExtended</code>. Only use it if you are working with two implementations. You'd rather
	 * use one implementation contained by the <tt>helper.properties</tt> file.
	 * 
	 * @param pokHelperExtendedImplClassName
	 *            The full name of the implementation <code>class</code>.
	 */
	public static void setPoKHelperExtendedImplClassName(final String pokHelperExtendedImplClassName) {
		RDFHelperFactory.pokHelpExtImplClassName = pokHelperExtendedImplClassName;
		RDFHelperFactory.pokHelpExtClass = null;
	}

	/**
	 * Use this to set the class implementation name for <code>PoKHelper</code>. Only use it if you are working with two implementations. You'd rather use one
	 * implementation contained by the <tt>helper.properties</tt> file.
	 * 
	 * @param pokHelperImplClassName
	 *            The full name of the implementation <code>class</code>.
	 */
	public static void setPoKHelperImplClassName(final String pokHelperImplClassName) {
		RDFHelperFactory.pokHelpImplClassName = pokHelperImplClassName;
		RDFHelperFactory.pokHelpClass = null;
	}

	/**
	 * Use this to set the class implementation name for <code>ResourceHelper</code>. Only use it if you are working with two implementations. You'd rather use
	 * one implementation contained by the <tt>helper.properties</tt> file.
	 * 
	 * @param resourceHelperImplClassName
	 *            The full name of the implementation <code>class</code>.
	 */
	public static void setResourceHelperImplClassName(final String resourceHelperImplClassName) {
		RDFHelperFactory.resHelpImplClassName = resourceHelperImplClassName;
		RDFHelperFactory.resHelpClass = null;
	}

	/**
	 * Use this to set the class implementation name for <code>ResourceHelperExtended</code>. Only use it if you are working with two implementations. You'd
	 * rather use one implementation contained by the <tt>helper.properties</tt> file.
	 * 
	 * @param resourceHelperExtendedImplClassName
	 *            The full name of the implementation <code>class</code>.
	 */
	public static void setResourceHelperExtendedImplClassName(final String resourceHelperExtendedImplClassName) {
		RDFHelperFactory.resHelpExtImplClassName = resourceHelperExtendedImplClassName;
		RDFHelperFactory.resHelpExtClass = null;
	}

}
