/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2009 EADS DEFENCE AND SECURITY SYSTEMS
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.core.helper;

import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * <p>
 * This class is an extract from former <code>AnnotationHelperExtended</code>, in order to be joint with new <code>ResourceHelperExtended</code> methods.
 * </p>
 * More comments are done in {@link PoKHelperExtended}.
 * 
 * @author EADS WebLab Team
 * @date 2008-09-16
 * @see PoKHelperExtended
 * @see ResourceHelperExtended
 */
interface RDFReaderExtended extends RDFReader {

	/**
	 * @param uriSubj
	 *            The URI of the subject resource of the statement.
	 * @param uriPred
	 *            The URI of the predicate resource of the statement.
	 * @param idObj
	 *            The internal ID of the object resource of the statement.
	 * @param uriReifPred
	 *            The URI of the predicate reifing the statement
	 * @return A <code>List</code> containing internal IDs of every anonymous resources that are object of statements having <code>uriReifPred</code> as
	 *         predicate and the reified statement (S:<code>uriSubj</code>, P:<code>uriPred</code>, O: <code>idObj</code>) as subject in the
	 *         <code>PieceOfKnowledge</code>.
	 */
	public List<String> getAnonRessOnPredAnonResStatReif(final String uriSubj, final String uriPred,
			final String idObj, final String uriReifPred);

	/**
	 * @param idSubj
	 *            The internal ID of the subject resource of the statement.
	 * @param uriPred
	 *            The URI of the predicate resource of the statement.
	 * @param idObj
	 *            The internal ID of the object resource of the statement.
	 * @param uriReifPred
	 *            The URI of the predicate reifing the statement
	 * @return A <code>List</code> containing internal IDs of every anonymous resources that are object of statements having <code>uriReifPred</code> as
	 *         predicate and the reified statement (S:<code>idSubj</code>, P:<code>uriPred</code>, O: <code>idObj</code>) as subject in the
	 *         <code>PieceOfKnowledge</code>.
	 */
	public List<String> getAnonRessOnPredAnonResStatReifOnAnonSubj(final String idSubj, final String uriPred,
			final String idObj, final String uriReifPred);

	/**
	 * @param idSubj
	 *            The internal Id of the subject resource.
	 * @param uriPred
	 *            The URI of the predicate resource.
	 * @return A <code>List</code> containing internal IDs of every anonymous resources that are object of statements having <code>uriPred</code> as predicate
	 *         and <code>idSubj</code> as subject in the <code>PieceOfKnowledge</code>.
	 */
	public List<String> getAnonRessOnPredAnonSubj(final String idSubj, final String uriPred);

	/**
	 * @param uriSubj
	 *            The URI of the subject resource of the statement.
	 * @param uriPred
	 *            The URI of the predicate resource of the statement.
	 * @param litObj
	 *            The value of the object literal of the statement.
	 * @param uriReifPred
	 *            The URI of the predicate reifing the statement
	 * @return A <code>List</code> containing internal IDs of every anonymous resources that are object of statements having <code>uriReifPred</code> as
	 *         predicate and the reified statement (S:<code>uriSubj</code>, P:<code>uriPred</code>, O: <code>litObj</code>) as subject in the
	 *         <code>PieceOfKnowledge</code>.
	 */
	public List<String> getAnonRessOnPredLitStatReif(final String uriSubj, final String uriPred, final String litObj,
			final String uriReifPred);

	/**
	 * @param idSubj
	 *            The internal ID of the subject resource of the statement.
	 * @param uriPred
	 *            The URI of the predicate resource of the statement.
	 * @param litObj
	 *            The value of the object literal of the statement.
	 * @param uriReifPred
	 *            The URI of the predicate reifing the statement
	 * @return A <code>List</code> containing internal IDs of every anonymous resources that are object of statements having <code>uriReifPred</code> as
	 *         predicate and the reified statement (S:<code>idSubj</code>, P:<code>uriPred</code>, O: <code>litObj</code>) as subject in the
	 *         <code>PieceOfKnowledge</code>.
	 */
	public List<String> getAnonRessOnPredLitStatReifOnAnonSubj(final String idSubj, final String uriPred,
			final String litObj, final String uriReifPred);

	/**
	 * @param uriSubj
	 *            The URI of the subject resource of the statement.
	 * @param uriPred
	 *            The URI of the predicate resource of the statement.
	 * @param uriObj
	 *            The URI of the object resource of the statement.
	 * @param uriReifPred
	 *            The URI of the predicate reifing the statement
	 * @return A <code>List</code> containing internal IDs of every anonymous resources that are object of statements having <code>uriReifPred</code> as
	 *         predicate and the reified statement (S:<code>uriSubj</code>, P:<code>uriPred</code>, O: <code>uriObj</code>) as subject in the
	 *         <code>PieceOfKnowledge</code>.
	 */
	public List<String> getAnonRessOnPredResStatReif(final String uriSubj, final String uriPred, final String uriObj,
			final String uriReifPred);

	/**
	 * @param idSubj
	 *            The internal ID of the subject resource of the statement.
	 * @param uriPred
	 *            The URI of the predicate resource of the statement.
	 * @param uriObj
	 *            The URI of the object resource of the statement.
	 * @param uriReifPred
	 *            The URI of the predicate reifing the statement
	 * @return A <code>List</code> containing internal IDs of every anonymous resources that are object of statements having <code>uriReifPred</code> as
	 *         predicate and the reified statement (S:<code>idSubj</code>, P:<code>uriPred</code>, O: <code>uriObj</code>) as subject in the
	 *         <code>PieceOfKnowledge</code>.
	 */
	public List<String> getAnonRessOnPredResStatReifOnAnonSubj(final String idSubj, final String uriPred,
			final String uriObj, final String uriReifPred);

	/**
	 * @param uriSubj
	 *            The URI of the subject resource.
	 * @param uriPred
	 *            The URI of the predicate resource.
	 * @return A <code>List</code> containing internal IDs of every anonymous resources that are object of statements having <code>uriPred</code> as predicate
	 *         and <code>uriSubj</code> as subject in the <code>PieceOfKnowledge</code>.
	 */
	public List<String> getAnonRessOnPredSubj(final String uriSubj, final String uriPred);

	/**
	 * @return A <code>Set</code> containing internal IDs of every anonymous resources that are subject of any statement in the <code>PieceOfKnowledge</code>.
	 */
	public Set<String> getAnonSubjs();

	/**
	 * @param uriPred
	 *            The URI of the predicate resource.
	 * @return A <code>Set</code> containing internal IDs of every anonymous resources that are subject of statements having <code>uriPred</code> as predicate
	 *         in the <code>PieceOfKnowledge</code>.
	 */
	public Set<String> getAnonSubjsOnPred(final String uriPred);

	/**
	 * @param uriPred
	 *            The URI of the predicate resource.
	 * @param idObj
	 *            The internal ID of the object resource.
	 * @return A <code>Set</code> containing internal IDs of every anonymous resources that are subject of statements having <code>uriPred</code> as predicate
	 *         and <code>idObj</code> as object resource URI in the <code>PieceOfKnowledge</code>.
	 */
	public Set<String> getAnonSubjsOnPredAnonRes(final String uriPred, final String idObj);

	/**
	 * @param uriPred
	 *            The URI of the predicate resource.
	 * @param litObj
	 *            The literal value of the object.
	 * @return A <code>Set</code> containing internal IDs of every anonymous resources that are subject of statements having <code>uriPred</code> as predicate
	 *         and <code>litObj</code> as object literal value in the <code>PieceOfKnowledge</code>.
	 */
	public Set<String> getAnonSubjsOnPredLit(final String uriPred, final String litObj);

	/**
	 * @param uriPred
	 *            The URI of the predicate resource.
	 * @param uriObj
	 *            The literal value of the object.
	 * @return A <code>Set</code> containing internal IDs of every anonymous resources that are subject of statements having <code>uriPred</code> as predicate
	 *         and <code>uriObj</code> as object resource URI in the <code>PieceOfKnowledge</code>.
	 */
	public Set<String> getAnonSubjsOnPredRes(final String uriPred, final String uriObj);

	/**
	 * @param uriSubj
	 *            The URI of the subject resource of the statement.
	 * @param uriPred
	 *            The URI of the predicate resource of the statement.
	 * @param idObj
	 *            The internal ID of the object resource of the statement.
	 * @param uriReifPred
	 *            The URI of the predicate reifing the statement
	 * @return A <code>List</code> containing values of every literals that are object of statements having <code>uriReifPred</code> as predicate and the
	 *         reified statement (S:<code>uriSubj</code>, P:<code>uriPred</code>, O: <code>idObj</code>) as subject in the <code>PieceOfKnowledge</code>.
	 */
	public List<String> getLitsOnPredAnonResStatReif(final String uriSubj, final String uriPred, final String idObj,
			final String uriReifPred);

	/**
	 * @param idSubj
	 *            The internal ID of the subject resource of the statement.
	 * @param uriPred
	 *            The URI of the predicate resource of the statement.
	 * @param idObj
	 *            The internal ID of the object resource of the statement.
	 * @param uriReifPred
	 *            The URI of the predicate reifing the statement
	 * @return A <code>List</code> containing values of every literals that are object of statements having <code>uriReifPred</code> as predicate and the
	 *         reified statement (S:<code>idSubj</code>, P:<code>uriPred</code>, O: <code>idObj</code>) as subject in the <code>PieceOfKnowledge</code>.
	 */
	public List<String> getLitsOnPredAnonResStatReifOnAnonSubj(final String idSubj, final String uriPred,
			final String idObj, final String uriReifPred);

	/**
	 * @param idSubj
	 *            The internal Id of the subject resource.
	 * @param uriPred
	 *            The URI of the predicate resource.
	 * @return A <code>List</code> containing values of every literal that are object of statements having <code>uriPred</code> as predicate and
	 *         <code>idSubj</code> as subject in the <code>PieceOfKnowledge</code>.
	 */
	public List<String> getLitsOnPredAnonSubj(final String idSubj, final String uriPred);

	/**
	 * @param uriSubj
	 *            The URI of the subject resource of the statement.
	 * @param uriPred
	 *            The URI of the predicate resource of the statement.
	 * @param litObj
	 *            The value of the object literal of the statement.
	 * @param uriReifPred
	 *            The URI of the predicate reifing the statement.
	 * @return A <code>List</code> containing values of every literal that are object of statements having <code>uriReifPred</code> as predicate and the reified
	 *         statement (S:<code>uriSubj</code>, P:<code>uriPred</code>, O: <code>litObj</code>) as subject in the <code>PieceOfKnowledge</code>.
	 */
	public List<String> getLitsOnPredLitStatReif(final String uriSubj, final String uriPred, final String litObj,
			final String uriReifPred);

	/**
	 * @param idSubj
	 *            The internal ID of the subject resource of the statement.
	 * @param uriPred
	 *            The URI of the predicate resource of the statement.
	 * @param litObj
	 *            The value of the object literal of the statement.
	 * @param uriReifPred
	 *            The URI of the predicate reifing the statement
	 * @return A <code>List</code> containing values of every literal that are object of statements having <code>uriReifPred</code> as predicate and the reified
	 *         statement (S:<code>idSubj</code>, P:<code>uriPred</code>, O: <code>litObj</code>) as subject in the <code>PieceOfKnowledge</code>.
	 */
	public List<String> getLitsOnPredLitStatReifOnAnonSubj(final String idSubj, final String uriPred,
			final String litObj, final String uriReifPred);

	/**
	 * @param uriSubj
	 *            The URI of the subject resource of the statement.
	 * @param uriPred
	 *            The URI of the predicate resource of the statement.
	 * @param uriObj
	 *            The URI of the object resource of the statement.
	 * @param uriReifPred
	 *            The URI of the predicate reifing the statement
	 * @return A <code>List</code> containing values of every literal that are object of statements having <code>uriReifPred</code> as predicate and the reified
	 *         statement (S:<code>uriSubj</code>, P:<code>uriPred</code>, O: <code>uriObj</code>) as subject in the <code>PieceOfKnowledge</code>.
	 */
	public List<String> getLitsOnPredResStatReif(final String uriSubj, final String uriPred, final String uriObj,
			final String uriReifPred);

	/**
	 * @param idSubj
	 *            The internal ID of the subject resource of the statement.
	 * @param uriPred
	 *            The URI of the predicate resource of the statement.
	 * @param uriObj
	 *            The URI of the object resource of the statement.
	 * @param uriReifPred
	 *            The URI of the predicate reifing the statement
	 * @return A <code>List</code> containing values of every literal that are object of statements having <code>uriReifPred</code> as predicate and the reified
	 *         statement (S:<code>idSubj</code>, P:<code>uriPred</code>, O: <code>uriObj</code>) as subject in the <code>PieceOfKnowledge</code>.
	 */
	public List<String> getLitsOnPredResStatReifOnAnonSubj(final String idSubj, final String uriPred,
			final String uriObj, final String uriReifPred);

	/**
	 * Retrieve the current map of prefixes and uris used in the <code>PieceOfKnowledge</code> contained by this <code>PoKHelperExtended</code>.
	 * 
	 * @return The current the map of prefixes/uris used.
	 */
	public Map<String, String> getNSPrefixes();

	/**
	 * @param uriSubj
	 *            The URI of the subject resource of the statement.
	 * @param uriPred
	 *            The URI of the predicate resource of the statement.
	 * @param idObj
	 *            The internal ID of the object resource of the statement.
	 * @return A <code>Set</code> containing URIs of every named resources that are predicate of any statement reifing the statement having <code>uriSubj</code>
	 *         as subject, <code>uriPred</code> as predicate and <code>idObj</code> as object in the <code>PieceOfKnowledge</code>.
	 */
	public Set<String> getPredsOnAnonResStatReif(final String uriSubj, final String uriPred, final String idObj);

	/**
	 * @param idSubj
	 *            The internal ID of the subject resource of the statement.
	 * @param uriPred
	 *            The URI of the predicate resource of the statement.
	 * @param idObj
	 *            The internal ID of the object resource of the statement.
	 * @return A <code>Set</code> containing URIs of every named resources that are predicate of any statement reifing the statement having <code>idSubj</code>
	 *         as subject, <code>uriPred</code> as predicate and <code>idObj</code> as object in the <code>PieceOfKnowledge</code>.
	 */
	public Set<String> getPredsOnAnonResStatReifOnAnonSubj(final String idSubj, final String uriPred, final String idObj);

	/**
	 * @param idSubj
	 *            The internal Id of the subject resource.
	 * @return A <code>Set</code> containing URIs of every named resources that are predicate of any statement having <code>idSubj</code> as subject in the
	 *         <code>PieceOfKnowledge</code>.
	 */
	public Set<String> getPredsOnAnonSubj(final String idSubj);

	/**
	 * @param uriSubj
	 *            The URI of the subject resource of the statement.
	 * @param uriPred
	 *            The URI of the predicate resource of the statement.
	 * @param litObj
	 *            The value of the object literal of the statement.
	 * @return A <code>Set</code> containing URIs of every named resources that are predicate of any statement reifing the statement having <code>uriSubj</code>
	 *         as subject, <code>uriPred</code> as predicate and <code>litObj</code> as object in the <code>PieceOfKnowledge</code>.
	 */
	public Set<String> getPredsOnLitStatReif(final String uriSubj, final String uriPred, final String litObj);

	/**
	 * @param idSubj
	 *            The internal ID of the subject resource of the statement.
	 * @param uriPred
	 *            The URI of the predicate resource of the statement.
	 * @param litObj
	 *            The value of the object literal of the statement.
	 * @return A <code>Set</code> containing URIs of every named resources that are predicate of any statement reifing the statement having <code>idSubj</code>
	 *         as subject, <code>uriPred</code> as predicate and <code>litObj</code> as object in the <code>PieceOfKnowledge</code>.
	 */
	public Set<String> getPredsOnLitStatReifOnAnonSubj(final String idSubj, final String uriPred, final String litObj);

	/**
	 * @param uriSubj
	 *            The URI of the subject resource of the statement.
	 * @param uriPred
	 *            The URI of the predicate resource of the statement.
	 * @param uriObj
	 *            The URI of the object resource of the statement.
	 * @return A <code>Set</code> containing URIs of every named resources that are predicate of any statement reifing the statement having <code>uriSubj</code>
	 *         as subject, <code>uriPred</code> as predicate and <code>uriObj</code> as object in the <code>PieceOfKnowledge</code>.
	 */
	public Set<String> getPredsOnResStatReif(final String uriSubj, final String uriPred, final String uriObj);

	/**
	 * @param idSubj
	 *            The internal ID of the subject resource of the statement.
	 * @param uriPred
	 *            The URI of the predicate resource of the statement.
	 * @param uriObj
	 *            The URI of the object resource of the statement.
	 * @return A <code>Set</code> containing URIs of every named resources that are predicate of any statement reifing the statement having <code>idSubj</code>
	 *         as subject, <code>uriPred</code> as predicate and <code>uriObj</code> as object in the <code>PieceOfKnowledge</code>.
	 */
	public Set<String> getPredsOnResStatReifOnAnonSubj(final String idSubj, final String uriPred, final String uriObj);

	/**
	 * @param uriSubj
	 *            The URI of the subject resource of the statement.
	 * @param uriPred
	 *            The URI of the predicate resource of the statement.
	 * @param idObj
	 *            The internal ID of the object resource of the statement.
	 * @param uriReifPred
	 *            The URI of the predicate reifing the statement
	 * @return A <code>List</code> containing URIs of every named resources that are object of statements having <code>uriReifPred</code> as predicate and the
	 *         reified statement (S:<code>uriSubj</code>, P:<code>uriPred</code>, O: <code>idObj</code>) as subject in the <code>PieceOfKnowledge</code>.
	 */
	public List<String> getRessOnPredAnonResStatReif(final String uriSubj, final String uriPred, final String idObj,
			final String uriReifPred);

	/**
	 * @param idSubj
	 *            The internal ID of the subject resource of the statement.
	 * @param uriPred
	 *            The URI of the predicate resource of the statement.
	 * @param idObj
	 *            The internal ID of the object resource of the statement.
	 * @param uriReifPred
	 *            The URI of the predicate reifing the statement
	 * @return A <code>List</code> containing URIs of every named resources that are object of statements having <code>uriReifPred</code> as predicate and the
	 *         reified statement (S:<code>idSubj</code>, P:<code>uriPred</code>, O: <code>idObj</code>) as subject in the <code>PieceOfKnowledge</code>.
	 */
	public List<String> getRessOnPredAnonResStatReifOnAnonSubj(final String idSubj, final String uriPred,
			final String idObj, final String uriReifPred);

	/**
	 * @param idSubj
	 *            The internal Id of the subject resource.
	 * @param uriPred
	 *            The URI of the predicate resource.
	 * @return A <code>List</code> containing URIs of every named resources that are object of statements having <code>uriPred</code> as predicate and
	 *         <code>idSubj</code> as subject in the <code>PieceOfKnowledge</code>.
	 */
	public List<String> getRessOnPredAnonSubj(final String idSubj, final String uriPred);

	/**
	 * @param uriSubj
	 *            The URI of the subject resource of the statement.
	 * @param uriPred
	 *            The URI of the predicate resource of the statement.
	 * @param litObj
	 *            The value of the object literal of the statement.
	 * @param uriReifPred
	 *            The URI of the predicate reifing the statement.
	 * @return A <code>List</code> containing URIs of every named resources that are object of statements having <code>uriReifPred</code> as predicate and the
	 *         reified statement (S:<code>uriSubj</code>, P:<code>uriPred</code>, O: <code>litObj</code>) as subject in the <code>PieceOfKnowledge</code>.
	 */
	public List<String> getRessOnPredLitStatReif(final String uriSubj, final String uriPred, final String litObj,
			final String uriReifPred);

	/**
	 * @param idSubj
	 *            The internal ID of the subject resource of the statement.
	 * @param uriPred
	 *            The URI of the predicate resource of the statement.
	 * @param litObj
	 *            The value of the object literal of the statement.
	 * @param uriReifPred
	 *            The URI of the predicate reifing the statement
	 * @return A <code>List</code> containing URIs of every named resources that are object of statements having <code>uriReifPred</code> as predicate and the
	 *         reified statement (S:<code>idSubj</code>, P:<code>uriPred</code>, O: <code>litObj</code>) as subject in the <code>PieceOfKnowledge</code>.
	 */
	public List<String> getRessOnPredLitStatReifOnAnonSubj(final String idSubj, final String uriPred,
			final String litObj, final String uriReifPred);

	/**
	 * @param uriSubj
	 *            The URI of the subject resource of the statement.
	 * @param uriPred
	 *            The URI of the predicate resource of the statement.
	 * @param uriObj
	 *            The URI of the object resource of the statement.
	 * @param uriReifPred
	 *            The URI of the predicate reifing the statement
	 * @return A <code>List</code> containing URIs of every named resources that are object of statements having <code>uriReifPred</code> as predicate and the
	 *         reified statement (S:<code>uriSubj</code>, P:<code>uriPred</code>, O: <code>uriObj</code>) as subject in the <code>PieceOfKnowledge</code>.
	 */
	public List<String> getRessOnPredResStatReif(final String uriSubj, final String uriPred, final String uriObj,
			final String uriReifPred);

	/**
	 * @param idSubj
	 *            The internal ID of the subject resource of the statement.
	 * @param uriPred
	 *            The URI of the predicate resource of the statement.
	 * @param uriObj
	 *            The URI of the object resource of the statement.
	 * @param uriReifPred
	 *            The URI of the predicate reifing the statement
	 * @return A <code>List</code> containing URIs of every named resources that are object of statements having <code>uriReifPred</code> as predicate and the
	 *         reified statement (S:<code>idSubj</code>, P:<code>uriPred</code>, O: <code>uriObj</code>) as subject in the <code>PieceOfKnowledge</code>.
	 */
	public List<String> getRessOnPredResStatReifOnAnonSubj(final String idSubj, final String uriPred,
			final String uriObj, final String uriReifPred);

	/**
	 * @param uriPred
	 *            The URI of the predicate resource.
	 * @param idObj
	 *            The internal ID of the object resource.
	 * @return A <code>Set</code> containing URIs of every named resources that are subject of statements having <code>uriPred</code> as predicate and
	 *         <code>idObj</code> as object resource URI in the <code>PieceOfKnowledge</code>.
	 */
	public Set<String> getSubjsOnPredAnonRes(final String uriPred, final String idObj);

	/**
	 * @param uriSubj
	 *            The URI of the subject resource.
	 * @param uriPred
	 *            The URI of the predicate resource.
	 * @param idObj
	 *            The internal Id of the object resource.
	 * @return Whether or not the <code>PieceOfKnowledge</code> contains at least one statement having <code>uriSubj</code> as subject, <code>uriPred</code> as
	 *         predicate and <code>idObj</code> as internal ID of the object.
	 */
	public boolean hasAnonResStat(final String uriSubj, final String uriPred, final String idObj);

	/**
	 * @param idSubj
	 *            The internal Id of the subject resource.
	 * @param uriPred
	 *            The URI of the predicate resource.
	 * @param idObj
	 *            The internal Id of the object resource.
	 * @return Whether or not the <code>PieceOfKnowledge</code> contains at least one statement having <code>idSubj</code> as subject, <code>uriPred</code> as
	 *         predicate and <code>idObj</code> as internal ID of the object.
	 */
	public boolean hasAnonResStatOnAnonSubj(final String idSubj, final String uriPred, final String idObj);

	/**
	 * @param idSubj
	 *            The internal Id of the subject resource.
	 * @param uriPred
	 *            The URI of the predicate resource.
	 * @param litObj
	 *            The literal value of the object.
	 * @return Whether or not the <code>PieceOfKnowledge</code> contains at least one statement having <code>idSubj</code> as subject, <code>uriPred</code> as
	 *         predicate and <code>litObj</code> as literal value of the object.
	 */
	public boolean hasLitStatOnAnonSubj(final String idSubj, final String uriPred, final String litObj);

	/**
	 * @param idSubj
	 *            The internal Id of the subject resource.
	 * @param uriPred
	 *            The URI of the predicate resource.
	 * @param uriObj
	 *            The URI of the object resource.
	 * @return Whether or not the <code>PieceOfKnowledge</code> contains at least one statement having <code>idSubj</code> as subject, <code>uriPred</code> as
	 *         predicate and <code>uriObj</code> as internal ID of the object.
	 */
	public boolean hasResStatOnAnonSubj(final String idSubj, final String uriPred, final String uriObj);

	/**
	 * @param uriSubj
	 *            The URI of the subject resource.
	 * @param uriPred
	 *            The URI of the predicate resource.
	 * @param idObj
	 *            The internal ID of the object resource.
	 * @return Whether or not the specified statement exists and is reified (i.e. subject of other statement).
	 */
	public boolean isAnonResStatReif(final String uriSubj, final String uriPred, final String idObj);

	/**
	 * @param idSubj
	 *            The internal ID of the subject resource.
	 * @param uriPred
	 *            The URI of the predicate resource.
	 * @param idObj
	 *            The internal ID of the object resource.
	 * @return Whether or not the specified statement exists and is reified (i.e. subject of other statement).
	 */
	public boolean isAnonResStatReifOnAnonSubj(final String idSubj, final String uriPred, final String idObj);

	/**
	 * Tests if the given internal ID is defined as an anonymous resource in the <code>PieceOfKnowledge</code>.
	 * 
	 * @param id
	 *            The internal Id to test existence in the <code>PieceOfKnowledge</code>.
	 * @return <code>true</code> if the URI is defined as an anonymous resource internal ID in the <code>PieceOfKnowledge</code>.
	 */
	public boolean isDefinedAnonRes(final String id);

	/**
	 * @param uriSubj
	 *            The URI of the subject resource.
	 * @param uriPred
	 *            The URI of the predicate resource.
	 * @param litObj
	 *            The value of the object literal.
	 * @return Whether or not the specified statement exists and is reified (i.e. subject of other statement).
	 */
	public boolean isLitStatReif(final String uriSubj, final String uriPred, final String litObj);

	/**
	 * @param idSubj
	 *            The internal ID of the subject resource.
	 * @param uriPred
	 *            The URI of the predicate resource.
	 * @param litObj
	 *            The value of the object literal.
	 * @return Whether or not the specified statement exists and is reified (i.e. subject of other statement).
	 */
	public boolean isLitStatReifOnAnonSubj(final String idSubj, final String uriPred, final String litObj);

	/**
	 * @param uriSubj
	 *            The URI of the subject resource.
	 * @param uriPred
	 *            The URI of the predicate resource.
	 * @param uriObj
	 *            The URI of the object resource.
	 * @return Whether or not the specified statement exists and is reified (i.e. subject of other statement).
	 */
	public boolean isResStatReif(final String uriSubj, final String uriPred, final String uriObj);

	/**
	 * @param idSubj
	 *            The internal ID of the subject resource.
	 * @param uriPred
	 *            The URI of the predicate resource.
	 * @param uriObj
	 *            The URI of the object resource.
	 * @return Whether or not the specified statement exists and is reified (i.e. subject of other statement).
	 */
	public boolean isResStatReifOnAnonSubj(final String idSubj, final String uriPred, final String uriObj);

	/**
	 * @param statId
	 *            The internal ID of the statement resource.
	 * @return The internal ID of the subject resource or <code>null</code> if the subject is not an anonymous resource.
	 */
	public String getAnonSubjOfAnonReifStat(final String statId);

	/**
	 * @param statUri
	 *            The URI of the statement resource.
	 * @return The internal ID of the subject resource or <code>null</code> if the subject is not an anonymous resource.
	 */
	public String getAnonSubjOfReifStat(final String statUri);

	/**
	 * @param statId
	 *            The internal ID of the statement resource.
	 * @return The URI of the subject resource or <code>null</code> if the subject is not a named resource.
	 */
	public String getSubjOfAnonReifStat(final String statId);

	/**
	 * @param statUri
	 *            The URI of the statement resource.
	 * @return The URI of the subject resource or <code>null</code> if the subject is not a named resource.
	 */
	public String getSubjOfReifStat(final String statUri);

	/**
	 * @param statId
	 *            The internal ID of the statement resource.
	 * @return The URI of the predicate resource.
	 */
	public String getPredOfAnonReifStat(final String statId);

	/**
	 * @param statUri
	 *            The URI of the statement resource.
	 * @return The URI of the predicate resource.
	 */
	public String getPredOfReifStat(final String statUri);

	/**
	 * @param statId
	 *            The internal ID of the statement resource.
	 * @return The internal ID of the object resource or <code>null</code> if the object is not an anonymous resource.
	 */
	public String getAnonResOfAnonReifStat(final String statId);

	/**
	 * @param statUri
	 *            The URI of the statement resource.
	 * @return The internal ID of the object resource or <code>null</code> if the object is not an anonymous resource.
	 */
	public String getAnonResOnReifStat(final String statUri);

	/**
	 * @param statId
	 *            The internal ID of the statement resource.
	 * @return The literal value of the object or <code>null</code> if the object is not a literal.
	 */
	public String getLitOfAnonReifStat(final String statId);

	/**
	 * @param statUri
	 *            The URI of the statement resource.
	 * @return The literal value of the object or <code>null</code> if the object is not a literal.
	 */
	public String getLitOfReifStat(final String statUri);

	/**
	 * @param statId
	 *            The internal ID of the statement resource.
	 * @return The URI of the object resource or <code>null</code> if the object is not a named resource.
	 */
	public String getResOfAnonReifStat(final String statId);

	/**
	 * @param statUri
	 *            The URI of the statement resource.
	 * @return The URI of the object resource or <code>null</code> if the object is not a named resource.
	 */
	public String getResOfReifStat(final String statUri);

	/**
	 * @return The <code>List</code> of internal IDs of all the anonymous reified statements.
	 */
	public List<String> getAnonReifStats();

	/**
	 * @return The <code>List</code> of URIs of all the named reified statements.
	 */
	public List<String> getReifStats();

	// TODO Comment after this !!!

	public Set<String> getPredsOnAnonStatReif(final String statId);

	public Set<String> getPredsOnStatReif(final String statUri);

	public List<String> getAnonRessOnPredAnonStatReif(final String statId, final String uriPred);

	public List<String> getRessOnPredAnonStatReif(final String statId, final String uriPred);

	public List<String> getLitsOnPredAnonStatReif(final String statId, final String uriPred);

	public List<String> getAnonRessOnPredStatReif(final String statUri, final String uriPred);

	public List<String> getRessOnPredStatReif(final String statUri, final String uriPred);

	public List<String> getLitsOnPredStatReif(final String statUri, final String uriPred);

}
