/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2009 EADS DEFENCE AND SECURITY SYSTEMS
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.core.helper;

import java.util.Map;

/**
 * <p>
 * This class is an extract from former <code>AnnotationHelperExtened</code>, in order to be separated from methods of <code>ResourceHelperExtended</code>.
 * </p>
 * More comments are done in {@link PoKHelperExtended}.
 * 
 * @author EADS WebLab Team
 * @date 2008-09-16
 * @see PoKHelperExtended
 */
interface RDFWriterExtended {

	/**
	 * Creates a statement having <code>uriSubj</code> as subject, <code>uriPred</code> as predicate and <code>uriObj</code> as URI of the object.<br />
	 * Changes are applied to the <code>PieceOfKnowledge</code> depending on {@link #setAutoCommitMode(boolean)}.
	 * 
	 * @param uriSubj
	 *            The URI of the subject resource.
	 * @param uriPred
	 *            The URI of the predicate resource.
	 * @param idObj
	 *            The internal ID of the object resource.
	 * @see #setAutoCommitMode(boolean)
	 * @see PoKHelper#commit()
	 */
	public void createAnonResStat(final String uriSubj, final String uriPred, final String idObj);

	/**
	 * Creates a statement having <code>idSubj</code> as subject, <code>uriPred</code> as predicate and <code>idObj</code> as internal ID of the object.<br />
	 * Changes are applied to the <code>PieceOfKnowledge</code> depending on {@link #setAutoCommitMode(boolean)}.
	 * 
	 * @param idSubj
	 *            The internal ID of the subject resource.
	 * @param uriPred
	 *            The URI of the predicate resource.
	 * @param idObj
	 *            The internal ID of the object resource.
	 * @see #setAutoCommitMode(boolean)
	 * @see PoKHelper#commit()
	 */
	public void createAnonResStatOnAnonSubj(final String idSubj, final String uriPred, final String idObj);

	/**
	 * If it does not exist, creates a statement having <code>uriSubj</code> as subject, <code>uriPred</code> as predicate and <code>idObj</code> as URI of the
	 * object.<br />
	 * Then, reified the statement using the three <code>Map</code>s.<br />
	 * Changes are applied to the <code>PieceOfKnowledge</code> depending on {@link #setAutoCommitMode(boolean)}.
	 * 
	 * @param uriSubj
	 *            The URI of the subject resource.
	 * @param uriPred
	 *            The URI of the predicate resource.
	 * @param idObj
	 *            The internal ID of the object resource.
	 * @param mapPredLit
	 *            <code>Map</code> of predicate URIs and literal values to be used to reified the statement. Might be <code>null</code> or empty.
	 * @param mapPredUri
	 *            <code>Map</code> of predicate URIs and named resource object URIs to be used to reified the statement. Might be <code>null</code> or empty.
	 * @param mapPredId
	 *            <code>Map</code> of predicate URIs and anonymous resource object internal IDs to be used to reified the statement. Might be <code>null</code>
	 *            or empty.
	 * @return The internal ID of the anonymous reified statement created.
	 */
	public String createAnonResStatReif(final String uriSubj, final String uriPred, final String idObj,
			final Map<String, String> mapPredLit, final Map<String, String> mapPredUri,
			final Map<String, String> mapPredId);

	/**
	 * If it does not exist, creates a statement having <code>uriSubj</code> as subject, <code>uriPred</code> as predicate and <code>idObj</code> as URI of the
	 * object.<br />
	 * Affects <code>statUri</code> as URI of the reified statement. Then, reified the statement using the three <code>Map</code>s.<br />
	 * Changes are applied to the <code>PieceOfKnowledge</code> depending on {@link #setAutoCommitMode(boolean)}.
	 * 
	 * @param uriSubj
	 *            The URI of the subject resource.
	 * @param uriPred
	 *            The URI of the predicate resource.
	 * @param idObj
	 *            The internal ID of the object resource.
	 * @param mapPredLit
	 *            <code>Map</code> of predicate URIs and literal values to be used to reified the statement. Might be <code>null</code> or empty.
	 * @param mapPredUri
	 *            <code>Map</code> of predicate URIs and named resource object URIs to be used to reified the statement. Might be <code>null</code> or empty.
	 * @param mapPredId
	 *            <code>Map</code> of predicate URIs and anonymous resource object internal IDs to be used to reified the statement. Might be <code>null</code>
	 *            or empty.
	 * @param statUri
	 *            The URI of the created reified statement.
	 * @return The uri of the created statement (i.e. <code>statUri</code>).
	 */
	public String createAnonResStatReif(final String uriSubj, final String uriPred, final String idObj,
			final Map<String, String> mapPredLit, final Map<String, String> mapPredUri,
			final Map<String, String> mapPredId, final String statUri);

	/**
	 * If it does not exist, creates a statement having <code>uriSubj</code> as subject, <code>uriPred</code> as predicate and <code>idObj</code> as URI of the
	 * object.<br />
	 * Then, reified the statement using the three <code>Map</code>s.<br />
	 * Changes are applied to the <code>PieceOfKnowledge</code> depending on {@link #setAutoCommitMode(boolean)}.
	 * 
	 * @param idSubj
	 *            The internal ID of the subject resource.
	 * @param uriPred
	 *            The URI of the predicate resource.
	 * @param idObj
	 *            The internal ID of the object resource.
	 * @param mapPredLit
	 *            <code>Map</code> of predicate URIs and literal values to be used to reified the statement. Might be <code>null</code> or empty.
	 * @param mapPredUri
	 *            <code>Map</code> of predicate URIs and named resource object URIs to be used to reified the statement. Might be <code>null</code> or empty.
	 * @param mapPredId
	 *            <code>Map</code> of predicate URIs and anonymous resource object internal IDs to be used to reified the statement. Might be <code>null</code>
	 *            or empty.
	 * @return The internal ID of the anonymous reified statement created.
	 */
	public String createAnonResStatReifOnAnonSubj(final String idSubj, final String uriPred, final String idObj,
			final Map<String, String> mapPredLit, final Map<String, String> mapPredUri,
			final Map<String, String> mapPredId);

	/**
	 * If it does not exist, creates a statement having <code>uriSubj</code> as subject, <code>uriPred</code> as predicate and <code>idObj</code> as URI of the
	 * object.<br />
	 * Affects <code>statUri</code> as URI of the reified statement. Then, reified the statement using the three <code>Map</code>s.<br />
	 * Changes are applied to the <code>PieceOfKnowledge</code> depending on {@link #setAutoCommitMode(boolean)}.
	 * 
	 * @param idSubj
	 *            The internal ID of the subject resource.
	 * @param uriPred
	 *            The URI of the predicate resource.
	 * @param idObj
	 *            The internal ID of the object resource.
	 * @param mapPredLit
	 *            <code>Map</code> of predicate URIs and literal values to be used to reified the statement. Might be <code>null</code> or empty.
	 * @param mapPredUri
	 *            <code>Map</code> of predicate URIs and named resource object URIs to be used to reified the statement. Might be <code>null</code> or empty.
	 * @param mapPredId
	 *            <code>Map</code> of predicate URIs and anonymous resource object internal IDs to be used to reified the statement. Might be <code>null</code>
	 *            or empty.
	 * @param statUri
	 *            The URI of the created reified statement.
	 * @return The uri of the created statement (i.e. <code>statUri</code>).
	 */
	public String createAnonResStatReifOnAnonSubj(final String idSubj, final String uriPred, final String idObj,
			final Map<String, String> mapPredLit, final Map<String, String> mapPredUri,
			final Map<String, String> mapPredId, final String statUri);

	/**
	 * Creates a statement having <code>idSubj</code> as subject, <code>uriPred</code> as predicate and <code>litObj</code> as literal value of the object.<br />
	 * Changes are applied to the <code>PieceOfKnowledge</code> depending on {@link #setAutoCommitMode(boolean)}.
	 * 
	 * @param idSubj
	 *            The internal Id of the subject resource.
	 * @param uriPred
	 *            The URI of the predicate resource.
	 * @param litObj
	 *            The literal value of the object.
	 */
	public void createLitStatOnAnonSubj(final String idSubj, final String uriPred, final String litObj);

	/**
	 * If it does not exist, creates a statement having <code>uriSubj</code> as subject, <code>uriPred</code> as predicate and <code>litObj</code> as URI of the
	 * object.<br />
	 * Affects <code>statUri</code> as URI of the reified statement. Then, reified the statement using the three <code>Map</code>s.<br />
	 * Changes are applied to the <code>PieceOfKnowledge</code> depending on {@link #setAutoCommitMode(boolean)}.
	 * 
	 * @param uriSubj
	 *            The URI of the subject resource.
	 * @param uriPred
	 *            The URI of the predicate resource.
	 * @param litObj
	 *            The literal value of the object.
	 * @param mapPredLit
	 *            <code>Map</code> of predicate URIs and literal values to be used to reified the statement. Might be <code>null</code> or empty.
	 * @param mapPredUri
	 *            <code>Map</code> of predicate URIs and named resource object URIs to be used to reified the statement. Might be <code>null</code> or empty.
	 * @param mapPredId
	 *            <code>Map</code> of predicate URIs and anonymous resource object internal IDs to be used to reified the statement. Might be <code>null</code>
	 *            or empty.
	 * @param statUri
	 *            The URI of the created reified statement.
	 * @return The uri of the created statement (i.e. <code>statUri</code>).
	 */
	public String createLitStatReif(final String uriSubj, final String uriPred, final String litObj,
			final Map<String, String> mapPredLit, final Map<String, String> mapPredUri,
			final Map<String, String> mapPredId, final String statUri);

	/**
	 * If it does not exist, creates a statement having <code>uriSubj</code> as subject, <code>uriPred</code> as predicate and <code>litObj</code> as URI of the
	 * object.<br />
	 * Then, reified the statement using the three <code>Map</code>s.<br />
	 * Changes are applied to the <code>PieceOfKnowledge</code> depending on {@link #setAutoCommitMode(boolean)}.
	 * 
	 * @param uriSubj
	 *            The URI of the subject resource.
	 * @param uriPred
	 *            The URI of the predicate resource.
	 * @param litObj
	 *            The literal value of the object.
	 * @param mapPredLit
	 *            <code>Map</code> of predicate URIs and literal values to be used to reified the statement. Might be <code>null</code> or empty.
	 * @param mapPredUri
	 *            <code>Map</code> of predicate URIs and named resource object URIs to be used to reified the statement. Might be <code>null</code> or empty.
	 * @param mapPredId
	 *            <code>Map</code> of predicate URIs and anonymous resource object internal IDs to be used to reified the statement. Might be <code>null</code>
	 *            or empty.
	 * @return The internal ID of the anonymous reified statement created.
	 */
	public String createLitStatReif(final String uriSubj, final String uriPred, final String litObj,
			final Map<String, String> mapPredLit, final Map<String, String> mapPredUri,
			final Map<String, String> mapPredId);

	/**
	 * If it does not exist, creates a statement having <code>idSubj</code> as subject, <code>uriPred</code> as predicate and <code>litObj</code> as URI of the
	 * object.<br />
	 * Then, reified the statement using the three <code>Map</code>s.<br />
	 * Changes are applied to the <code>PieceOfKnowledge</code> depending on {@link #setAutoCommitMode(boolean)}.
	 * 
	 * @param idSubj
	 *            The internal ID of the subject resource.
	 * @param uriPred
	 *            The URI of the predicate resource.
	 * @param litObj
	 *            The literal value of the object.
	 * @param mapPredLit
	 *            <code>Map</code> of predicate URIs and literal values to be used to reified the statement. Might be <code>null</code> or empty.
	 * @param mapPredUri
	 *            <code>Map</code> of predicate URIs and named resource object URIs to be used to reified the statement. Might be <code>null</code> or empty.
	 * @param mapPredId
	 *            <code>Map</code> of predicate URIs and anonymous resource object internal IDs to be used to reified the statement. Might be <code>null</code>
	 *            or empty.
	 * @return The internal ID of the anonymous reified statement created.
	 */
	public String createLitStatReifOnAnonSubj(final String idSubj, final String uriPred, final String litObj,
			final Map<String, String> mapPredLit, final Map<String, String> mapPredUri,
			final Map<String, String> mapPredId);

	/**
	 * If it does not exist, creates a statement having <code>idSubj</code> as subject, <code>uriPred</code> as predicate and <code>litObj</code> as URI of the
	 * object.<br />
	 * Affects <code>statUri</code> as URI of the reified statement. Then, reified the statement using the three <code>Map</code>s.<br />
	 * Changes are applied to the <code>PieceOfKnowledge</code> depending on {@link #setAutoCommitMode(boolean)}.
	 * 
	 * @param idSubj
	 *            The internal ID of the subject resource.
	 * @param uriPred
	 *            The URI of the predicate resource.
	 * @param litObj
	 *            The literal value of the object.
	 * @param mapPredLit
	 *            <code>Map</code> of predicate URIs and literal values to be used to reified the statement. Might be <code>null</code> or empty.
	 * @param mapPredUri
	 *            <code>Map</code> of predicate URIs and named resource object URIs to be used to reified the statement. Might be <code>null</code> or empty.
	 * @param mapPredId
	 *            <code>Map</code> of predicate URIs and anonymous resource object internal IDs to be used to reified the statement. Might be <code>null</code>
	 *            or empty.
	 * @param statUri
	 *            The URI of the created reified statement.
	 * @return The uri of the created statement (i.e. <code>statUri</code>).
	 */
	public String createLitStatReifOnAnonSubj(final String idSubj, final String uriPred, final String litObj,
			final Map<String, String> mapPredLit, final Map<String, String> mapPredUri,
			final Map<String, String> mapPredId, final String statUri);

	/**
	 * Creates a statement having <code>idSubj</code> as subject, <code>uriPred</code> as predicate and <code>uriObj</code> as URI of the object.<br />
	 * Changes are applied to the <code>PieceOfKnowledge</code> depending on {@link #setAutoCommitMode(boolean)}.
	 * 
	 * @param idSubj
	 *            The internal Id of the subject resource.
	 * @param uriPred
	 *            The URI of the predicate resource.
	 * @param uriObj
	 *            The URI of the object resource.
	 */
	public void createResStatOnAnonSubj(final String idSubj, final String uriPred, final String uriObj);

	/**
	 * If it does not exist, creates a statement having <code>uriSubj</code> as subject, <code>uriPred</code> as predicate and <code>uriObj</code> as URI of the
	 * object.<br />
	 * Then, reified the statement using the three <code>Map</code>s.<br />
	 * Changes are applied to the <code>PieceOfKnowledge</code> depending on {@link #setAutoCommitMode(boolean)}.
	 * 
	 * @param uriSubj
	 *            The URI of the subject resource.
	 * @param uriPred
	 *            The URI of the predicate resource.
	 * @param uriObj
	 *            The URI of the object resource.
	 * @param mapPredLit
	 *            <code>Map</code> of predicate URIs and literal values to be used to reified the statement. Might be <code>null</code> or empty.
	 * @param mapPredUri
	 *            <code>Map</code> of predicate URIs and named resource object URIs to be used to reified the statement. Might be <code>null</code> or empty.
	 * @param mapPredId
	 *            <code>Map</code> of predicate URIs and anonymous resource object internal IDs to be used to reified the statement. Might be <code>null</code>
	 *            or empty.
	 * @return The internal ID of the anonymous reified statement created.
	 */
	public String createResStatReif(final String uriSubj, final String uriPred, final String uriObj,
			final Map<String, String> mapPredLit, final Map<String, String> mapPredUri,
			final Map<String, String> mapPredId);

	/**
	 * If it does not exist, creates a statement having <code>uriSubj</code> as subject, <code>uriPred</code> as predicate and <code>uriObj</code> as URI of the
	 * object.<br />
	 * Affects <code>statUri</code> as URI of the reified statement. Then, reified the statement using the three <code>Map</code>s.<br />
	 * Changes are applied to the <code>PieceOfKnowledge</code> depending on {@link #setAutoCommitMode(boolean)}.
	 * 
	 * @param uriSubj
	 *            The URI of the subject resource.
	 * @param uriPred
	 *            The URI of the predicate resource.
	 * @param uriObj
	 *            The URI of the object resource.
	 * @param mapPredLit
	 *            <code>Map</code> of predicate URIs and literal values to be used to reified the statement. Might be <code>null</code> or empty.
	 * @param mapPredUri
	 *            <code>Map</code> of predicate URIs and named resource object URIs to be used to reified the statement. Might be <code>null</code> or empty.
	 * @param mapPredId
	 *            <code>Map</code> of predicate URIs and anonymous resource object internal IDs to be used to reified the statement. Might be <code>null</code>
	 *            or empty.
	 * @param statUri
	 *            The URI of the created reified statement.
	 * @return The uri of the created statement (i.e. <code>statUri</code>).
	 */
	public String createResStatReif(final String uriSubj, final String uriPred, final String uriObj,
			final Map<String, String> mapPredLit, final Map<String, String> mapPredUri,
			final Map<String, String> mapPredId, final String statUri);

	/**
	 * If it does not exist, creates a statement having <code>idSubj</code> as subject, <code>uriPred</code> as predicate and <code>uriObj</code> as URI of the
	 * object.<br />
	 * Then, reified the statement using the three <code>Map</code>s.<br />
	 * Changes are applied to the <code>PieceOfKnowledge</code> depending on {@link #setAutoCommitMode(boolean)}.
	 * 
	 * @param idSubj
	 *            The internal ID of the subject resource.
	 * @param uriPred
	 *            The URI of the predicate resource.
	 * @param uriObj
	 *            The URI of the object resource.
	 * @param mapPredLit
	 *            <code>Map</code> of predicate URIs and literal values to be used to reified the statement. Might be <code>null</code> or empty.
	 * @param mapPredUri
	 *            <code>Map</code> of predicate URIs and named resource object URIs to be used to reified the statement. Might be <code>null</code> or empty.
	 * @param mapPredId
	 *            <code>Map</code> of predicate URIs and anonymous resource object internal IDs to be used to reified the statement. Might be <code>null</code>
	 *            or empty.
	 * @return The internal ID of the anonymous reified statement created.
	 */
	public String createResStatReifOnAnonSubj(final String idSubj, final String uriPred, final String uriObj,
			final Map<String, String> mapPredLit, final Map<String, String> mapPredUri,
			final Map<String, String> mapPredId);

	/**
	 * If it does not exist, creates a statement having <code>idSubj</code> as subject, <code>uriPred</code> as predicate and <code>uriObj</code> as URI of the
	 * object.<br />
	 * Affects <code>statUri</code> as URI of the reified statement. Then, reified the statement using the three <code>Map</code>s.<br />
	 * Changes are applied to the <code>PieceOfKnowledge</code> depending on {@link #setAutoCommitMode(boolean)}.
	 * 
	 * @param idSubj
	 *            The internal ID of the subject resource.
	 * @param uriPred
	 *            The URI of the predicate resource.
	 * @param uriObj
	 *            The URI of the object resource.
	 * @param mapPredLit
	 *            <code>Map</code> of predicate URIs and literal values to be used to reified the statement. Might be <code>null</code> or empty.
	 * @param mapPredUri
	 *            <code>Map</code> of predicate URIs and named resource object URIs to be used to reified the statement. Might be <code>null</code> or empty.
	 * @param mapPredId
	 *            <code>Map</code> of predicate URIs and anonymous resource object internal IDs to be used to reified the statement. Might be <code>null</code>
	 *            or empty.
	 * @param statUri
	 *            The URI of the created reified statement.
	 * @return The uri of the created statement (i.e. <code>statUri</code>).
	 */
	public String createResStatReifOnAnonSubj(final String idSubj, final String uriPred, final String uriObj,
			final Map<String, String> mapPredLit, final Map<String, String> mapPredUri,
			final Map<String, String> mapPredId, final String statUri);

	/**
	 * Removes every statements having <code>uriSubj</code> as subject, <code>uriPred</code> as predicate and <code>idObj</code> as object.<br />
	 * Changes are applied to the <code>PieceOfKnowledge</code> depending on {@link #setAutoCommitMode(boolean)}.
	 * 
	 * @param uriSubj
	 *            The URI of the subject resource.
	 * @param uriPred
	 *            The URI of the predicate resource.
	 * @param idObj
	 *            The internal ID of the object resource.
	 * @return The number of removed statements.
	 */
	public void removeAnonResStat(final String uriSubj, final String uriPred, final String idObj);

	/**
	 * Removes every statements having <code>idSubj</code> as subject, <code>uriPred</code> as predicate and <code>idObj</code> as object.<br />
	 * Changes are applied to the <code>PieceOfKnowledge</code> depending on {@link #setAutoCommitMode(boolean)}.
	 * 
	 * @param idSubj
	 *            The internal ID of the subject resource.
	 * @param uriPred
	 *            The URI of the predicate resource.
	 * @param idObj
	 *            The internal ID of the object resource.
	 * @return The number of removed statements.
	 * @see PoKHelper#commit()
	 * @see #setAutoCommitMode(boolean)
	 */
	public void removeAnonResStatOnAnonSubj(final String idSubj, final String uriPred, final String idObj);

	/**
	 * Removes every statements having <code>idSubj</code> as subject, <code>uriPred</code> as predicate and <code>litObj</code> as object.<br />
	 * Changes are applied to the <code>PieceOfKnowledge</code> depending on {@link #setAutoCommitMode(boolean)}.
	 * 
	 * @param idSubj
	 *            The internal ID of the subject resource.
	 * @param uriPred
	 *            The URI of the predicate resource.
	 * @param litObj
	 *            The literal value of the object.
	 * @return The number of removed statements.
	 */
	public void removeLitStatOnAnonSubj(final String idSubj, final String uriPred, final String litObj);

	/**
	 * Removes every statements having <code>idSubj</code> as subject, <code>uriPred</code> as predicate and <code>uriObj</code> as object.<br />
	 * Changes are applied to the <code>PieceOfKnowledge</code> depending on {@link #setAutoCommitMode(boolean)}.
	 * 
	 * @param idSubj
	 *            The internal ID of the subject resource.
	 * @param uriPred
	 *            The URI of the predicate resource.
	 * @param uriObj
	 *            The URI of the object resource.
	 * @return The number of removed statements.
	 */
	public void removeResStatOnAnonSubj(final String idSubj, final String uriPred, final String uriObj);

	/**
	 * Removes every statements having <code>idSubj</code> as subject, whatever could be predicates and objects involved in statements.<br />
	 * Changes are applied to the <code>PieceOfKnowledge</code> depending on {@link #setAutoCommitMode(boolean)}.
	 * 
	 * @param idSubj
	 *            The internal ID of the subject resource.
	 * @return The number of removed statements.
	 * @see PoKHelper#commit()
	 * @see #setAutoCommitMode(boolean)
	 */
	public void removeStatsOnAnonSubj(final String idSubj);

	/**
	 * Removes every statements having <code>idObj</code> as object, whatever could be predicates and subject involved in statements.<br />
	 * Changes are applied to the <code>PieceOfKnowledge</code> depending on {@link #setAutoCommitMode(boolean)}.
	 * 
	 * @param idObj
	 *            The internal ID of the object resource.
	 * @return The number of removed statements.
	 */
	public void removeStatsWithAnonRes(final String idObj);

	/**
	 * Removes all the statements about the reified one and removes it itself.
	 * 
	 * @param statId
	 *            The internal ID of the reified statement to be removed.
	 */
	public void removeAnonReifStat(final String statId);

	/**
	 * Removes all the statements about the reified one and removes it itself.
	 * 
	 * @param statUri
	 *            The URI of the reified statement to be removed.
	 */
	public void removeReifStat(final String statUri);

	/**
	 * Sets a prefix to be used to represent the given <code>URI</code> in the RDF/XML view of the <code>PieceOfKnowledge</code> data.<br />
	 * The prefixes are used in other RDF representations such as Turtle. For instance, <code>skos</code> may represent
	 * <code>http://www.w3.org/2004/02/skos/core#</code>. If <code>prefix</code> or <code>uri</code> are <code>null</code> nothing is done. If
	 * <code>prefix</code> is <code>&quot;&quot;</code>, <code>uri</code> will be the default namespace. If <code>prefix</code> is <code>&quot;&quot;</code>
	 * while <code>uri</code> is not, nothing is done.
	 * 
	 * @param prefix
	 *            The <code>String</code> to be used as namespace prefix in the RDF/XML of the <code>PieceOfKnowledge</code>.
	 * @param uri
	 *            The <code>String</code> to be represented by <code>prefix</code>
	 * @return The current the map of prefixes/uris used.
	 */
	public Map<String, String> setNSPrefix(String prefix, String uri);

}
