/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2009 EADS DEFENCE AND SECURITY SYSTEMS
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.core.helper.impl;

import java.util.List;
import java.util.Set;

import org.apache.commons.logging.LogFactory;
import org.ow2.weblab.core.extended.exception.WebLabCheckedException;
import org.ow2.weblab.core.extended.exception.WebLabUncheckedException;
import org.ow2.weblab.core.extended.util.PoKUtil;
import org.ow2.weblab.core.helper.PoKHelper;
import org.ow2.weblab.core.model.PieceOfKnowledge;
import org.ow2.weblab.utils.BackEndJenaHelper;

/**
 * Implementation of <code>AnnotationHelper</code> based on <tt>Jena framework</tt>.<br />
 * You'd rather to load this class using the <code>HelperFactory</code>.<br />
 * 
 * @author EADS WebLab Team
 * @date 2008-01-18
 * @see JenaPoKHelperExtended
 * @todo Prevent the direct instantiation of this class.
 */
public class JenaPoKHelper implements PoKHelper {

	/**
	 * The current <code>Annotation</code> object loaded
	 */
	protected PieceOfKnowledge pok;

	/**
	 * The Jena model containing the RDF graph
	 */
	protected BackEndJenaHelper bejh;

	/**
	 * In auto commit mode, remove and create operations are writing into <code>pok</code> each time they are called. When performing a lot's of actions on the
	 * RDF, best practice is to set auto commit mode to <code>false</code> and to commit at the end of the process.
	 */
	protected boolean autoCommit = true;

	/*
	 * (non-Javadoc)
	 * @see org.weblab_project.core.helper.RDFReader#getSubjs()
	 */
	public Set<String> getSubjs() {
		return this.bejh.getSubjs();
	}

	/*
	 * (non-Javadoc)
	 * @see org.weblab_project.core.helper.RDFReader#getPredsOnSubj(java.lang.String)
	 */
	public Set<String> getPredsOnSubj(final String uriSubj) {
		return this.bejh.getPredsOnSubj(uriSubj);
	}

	/*
	 * (non-Javadoc)
	 * @see org.weblab_project.core.helper.RDFReader#getLitsOnPredSubj(java.lang.String, java.lang.String)
	 */
	public List<String> getLitsOnPredSubj(final String uriSubj, final String uriPred) {
		return this.bejh.getLitsOnPredSubj(uriSubj, uriPred);
	}

	/*
	 * (non-Javadoc)
	 * @see org.weblab_project.core.helper.RDFReader#getRessOnPredSubj(java.lang.String, java.lang.String)
	 */
	public List<String> getRessOnPredSubj(final String uriSubj, final String uriPred) {
		return this.bejh.getRessOnPredSubj(uriSubj, uriPred);
	}

	/*
	 * (non-Javadoc)
	 * @see org.weblab_project.core.helper.RDFReader#hasLitStat(java.lang.String, java.lang.String, java.lang.String)
	 */
	public boolean hasLitStat(final String uriSubj, final String uriPred, final String litObj) {
		return this.bejh.hasLitStat(uriSubj, uriPred, litObj);
	}

	/*
	 * (non-Javadoc)
	 * @see org.weblab_project.core.helper.RDFReader#hasResStat(java.lang.String, java.lang.String, java.lang.String)
	 */
	public boolean hasResStat(final String uriSubj, final String uriPred, final String uriObj) {
		return this.bejh.hasResStat(uriSubj, uriPred, uriObj);
	}

	
	/*
	 * (non-Javadoc)
	 * @see org.weblab_project.core.helper.RDFWriter#createLitStat(java.lang.String, java.lang.String, java.lang.String, java.lang.String)
	 */
	public void createLitStat(final String uriSubj, final String uriPred, final String litObj, final String language) {
		this.bejh.createLitStat(uriSubj, uriPred, litObj, language);	
		if (this.autoCommit) {
			this.commit();
		}
	}
	
	/*
	 * (non-Javadoc)
	 * @see org.weblab_project.core.helper.RDFWriter#createLitStat(java.lang.String, java.lang.String, java.lang.String)
	 */
	public void createLitStat(final String uriSubj, final String uriPred, final String litObj) {
		this.bejh.createLitStat(uriSubj, uriPred, litObj);
		if (this.autoCommit) {
			this.commit();
		}
	}

	/*
	 * (non-Javadoc)
	 * @see org.weblab_project.core.helper.RDFWriter#createResStat(java.lang.String, java.lang.String, java.lang.String)
	 */
	public void createResStat(final String uriSubj, final String uriPred, final String uriObj) {
		this.bejh.createResStat(uriSubj, uriPred, uriObj);
		if (this.autoCommit) {
			this.commit();
		}
	}

	/*
	 * (non-Javadoc)
	 * @see org.weblab_project.core.helper.RDFWriter#removeLitStat(java.lang.String, java.lang.String, java.lang.String)
	 */
	public void removeLitStat(final String uriSubj, final String uriPred, final String litObj) {
		this.bejh.removeLitStat(uriSubj, uriPred, litObj);
		if (this.autoCommit) {
			this.commit();
		}
	}

	/*
	 * (non-Javadoc)
	 * @see org.weblab_project.core.helper.RDFWriter#removeResStat(java.lang.String, java.lang.String, java.lang.String)
	 */
	public void removeResStat(final String uriSubj, final String uriPred, final String uriObj) {
		this.bejh.removeResStat(uriSubj, uriPred, uriObj);
		if (this.autoCommit) {
			this.commit();
		}
	}

	/*
	 * (non-Javadoc)
	 * @see org.weblab_project.core.helper.RDFWriter#removeStatsOnSubj(java.lang.String)
	 */
	public void removeStatsOnSubj(final String uriSubj) {
		this.bejh.removeStatsOnSubj(uriSubj);
		if (this.autoCommit) {
			this.commit();
		}
	}

	/*
	 * (non-Javadoc)
	 * @see org.weblab_project.core.helper.RDFWriter#removeStatsWithPred(java.lang.String)
	 */
	public void removeStatsWithPred(final String uriPred) {
		this.bejh.removeStatsWithPred(uriPred);
		if (this.autoCommit) {
			this.commit();
		}
	}

	/*
	 * (non-Javadoc)
	 * @see org.weblab_project.core.helper.RDFWriter#removeStatsWithLit(java.lang.String)
	 */
	public void removeStatsWithLit(final String litObj) {
		this.bejh.removeStatsWithLit(litObj);
		if (this.autoCommit) {
			this.commit();
		}
	}

	/*
	 * (non-Javadoc)
	 * @see org.weblab_project.core.helper.RDFWriter#removeStatsWithRes(java.lang.String)
	 */
	public void removeStatsWithRes(final String uriObj) {
		this.bejh.removeStatsWithRes(uriObj);
		if (this.autoCommit) {
			this.commit();
		}
	}

	/*
	 * (non-Javadoc)
	 * @see org.weblab_project.core.helper.RDFReader#getSubjsOnPred(java.lang.String)
	 */
	public Set<String> getSubjsOnPred(final String uriPred) {
		return this.bejh.getSubjsOnPred(uriPred);
	}

	/*
	 * (non-Javadoc)
	 * @see org.weblab_project.core.helper.RDFReader#getSubjsOnPredLit(java.lang.String, java.lang.String)
	 */
	public Set<String> getSubjsOnPredLit(final String uriPred, final String litObj) {
		return this.bejh.getSubjsOnPredLit(uriPred, litObj);
	}

	/*
	 * (non-Javadoc)
	 * @see org.weblab_project.core.helper.RDFReader#getSubjsOnPredRes(java.lang.String, java.lang.String)
	 */
	public Set<String> getSubjsOnPredRes(final String uriPred, final String uriObj) {
		return this.bejh.getSubjsOnPredRes(uriPred, uriObj);
	}

	/*
	 * (non-Javadoc)
	 * @see org.weblab_project.core.helper.AnnotationHelper#setAutoCommitMode(boolean)
	 */
	public void setAutoCommitMode(final boolean autoCommitMode) {
		this.autoCommit = autoCommitMode;
	}

	/*
	 * (non-Javadoc)
	 * @see org.weblab_project.core.helper.RDFReader#loadFromRdfXml(java.lang.String)
	 */
	public void loadFromRdfXml(final String rdfXml) throws WebLabCheckedException {
		this.bejh = new BackEndJenaHelper(rdfXml);
	}

	/*
	 * (non-Javadoc)
	 * @see org.weblab_project.core.helper.RDFWriter#getRdfXml()
	 */
	public String getRdfXml() {
		if (this.bejh != null && this.pok != null) {
			return this.bejh.getRdfXml();
		}
		return "";
	}

	/*
	 * (non-Javadoc)
	 * @see org.weblab_project.core.helper.RDFReader#isDefinedResource(java.lang.String)
	 */
	public boolean isDefinedResource(final String uri) {
		return this.bejh.isDefinedResource(uri);
	}

	/*
	 * (non-Javadoc)
	 * @see org.weblab_project.core.helper.AnnotationHelper#commit()
	 */
	public void commit() {
		if (this.bejh != null && this.pok != null) {
			final String data = this.getRdfXml();
			try {
				PoKUtil.setPoKData(this.pok, data);
			} catch (final WebLabCheckedException wce) {
				throw new WebLabUncheckedException("Serialisation of Jena "
						+ "model failed. RDF/XML string appears to be " + "invalid: " + data, wce);
			}
		} else {
			throw new WebLabUncheckedException("Annotation was not set.");
		}
	}

	/*
	 * (non-Javadoc)
	 * @see org.weblab_project.core.helper.PoKHelper#getPOK()
	 */
	public PieceOfKnowledge getPoK() {
		return this.pok;
	}

	/*
	 * (non-Javadoc)
	 * @see org.weblab_project.core.helper.AnnotationHelper#setAnnotation(org.weblab_project.core.model.Annotation)
	 */
	public void setPoK(PieceOfKnowledge pok) {
		this.pok = pok;
		if (this.pok == null) {
			throw new WebLabUncheckedException("No PieceOfKnowledge to load.");
		}
		if (this.bejh != null) {
			this.bejh.close();
		}
		if (this.pok.getData() != null) {
			final String data = PoKUtil.getPoKData(this.pok);
			try {
				this.loadFromRdfXml(data);
			} catch (final WebLabCheckedException wlce) {
				LogFactory.getLog(this.getClass()).warn("An error occurs loading the pok.");
				LogFactory.getLog(this.getClass()).trace(data);
				this.bejh = new BackEndJenaHelper();
			}
		} else {
			this.bejh = new BackEndJenaHelper();
		}
	}

	/*
	 * (non-Javadoc)
	 * @see org.weblab_project.core.helper.RDFReader#getPreds()
	 */
	public Set<String> getPreds() {
		return this.bejh.getPreds();
	}

	/*
	 * (non-Javadoc)
	 * @see org.weblab_project.core.helper.RDFReader#getPreds()
	 */
	public List<String> getLitsOnPredSubj(String uriSubj, String uriPred, String language) {
		return  this.bejh.getLitsOnPredSubj(uriSubj, uriPred, language);
	}

}
