/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2009 EADS DEFENCE AND SECURITY SYSTEMS
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.core.helper.impl;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.ow2.weblab.core.extended.exception.WebLabCheckedException;
import org.ow2.weblab.core.extended.util.PoKUtil;
import org.ow2.weblab.core.helper.ResourceHelper;
import org.ow2.weblab.core.model.Annotation;
import org.ow2.weblab.core.model.ComposedQuery;
import org.ow2.weblab.core.model.ComposedResource;
import org.ow2.weblab.core.model.Document;
import org.ow2.weblab.core.model.LowLevelDescriptor;
import org.ow2.weblab.core.model.MediaUnit;
import org.ow2.weblab.core.model.PieceOfKnowledge;
import org.ow2.weblab.core.model.Query;
import org.ow2.weblab.core.model.Resource;
import org.ow2.weblab.core.model.ResultSet;
import org.ow2.weblab.core.model.SimilarityQuery;
import org.ow2.weblab.utils.BackEndJenaHelper;

/**
 * This is an implementation of the <code>ResourceHelper</code>. It loads every <code>Annotations</code> that are on this object and in the inner
 * <code>Resources</code>. If the loaded <code>Resource</code> is an <code>Annotation</code>, its content won't be loaded.
 * 
 * @author EADS WebLab Team
 * @date 2008-11-25
 * @todo Prevent the direct instantiation of this class.
 */
public class JenaResourceHelper implements ResourceHelper {

	/**
	 * The current <code>Resource</code> object loaded.
	 */
	protected Resource resource;

	/**
	 * Map to remember the mapping between sub-resources and resource URIs
	 */
	protected Map<String, Resource> resourcesMap;

	/**
	 * An handler for the Jena model behind.
	 */
	protected BackEndJenaHelper bejh;

	/*
	 * (non-Javadoc)
	 * @see org.weblab_project.core.helper.ResourceHelper#getResource(java.lang.String )
	 */
	public Resource getResource(final String uri) throws WebLabCheckedException {
		if (!this.isFullyDefinedResource(uri)) {
			throw new WebLabCheckedException("Cannot retrieve type for " + "resource '" + uri
					+ "'. Resource is not defined.");
		}
		return this.resourcesMap.get(uri);
	}

	/*
	 * (non-Javadoc)
	 * @see org.weblab_project.core.helper.ResourceHelper#getSpecificResource(java .lang.String, java.lang.Class)
	 */
	public <T extends Resource> T getSpecificResource(final String uri, final Class<T> specificClass)
			throws WebLabCheckedException {
		final Resource res = this.getResource(uri);
		if (specificClass.isInstance(res)) {
			return specificClass.cast(res);
		}
		throw new WebLabCheckedException("Cannot retrieve Resource '" + uri + "'", new ClassCastException(
				"Unable to cast resource: " + res.getUri() + " into a " + specificClass.getName() + " since its a "
						+ res.getClass().getName()));
	}

	/*
	 * (non-Javadoc)
	 * @see org.weblab_project.core.helper.ResourceHelper#getType(java.lang.String)
	 */
	public Class<? extends Resource> getType(final String uri) throws WebLabCheckedException {
		final Resource res = this.getResource(uri);
		return res.getClass();
	}

	/*
	 * (non-Javadoc)
	 * @see org.weblab_project.core.helper.ResourceHelper#setResource(org.weblab_project .core.model.Resource)
	 */
	public void setResource(final Resource resource) {
		if (this.bejh != null) {
			this.bejh.close();
		}
		this.bejh = new BackEndJenaHelper();
		this.resourcesMap = new HashMap<String, Resource>();
		this.addResource(resource);
		this.resource = resource;
	}

	/**
	 * Method recursively called to add every resources in the model and in the map.
	 * 
	 * @param res
	 *            The resource to add.
	 */
	protected void addResource(final Resource res) {
		this.resourcesMap.put(res.getUri(), res);
		for (final Annotation annot : res.getAnnotation()) {
			this.addResource(annot);
		}
		for (final LowLevelDescriptor lld : res.getDescriptor()) {
			this.addResource(lld);
		}

		if (res instanceof MediaUnit) {
			this.addMediaUnit((MediaUnit) res);
		} else if (res instanceof PieceOfKnowledge) {
			this.addPieceOfKnowledge((PieceOfKnowledge) res);
		} else if (res instanceof ComposedResource) {
			this.addComposedResource((ComposedResource) res);
		} else if (res instanceof ResultSet) {
			this.addResultSet((ResultSet) res);
		} else if (res instanceof ComposedQuery) {
			this.addComposedQuery((ComposedQuery) res);
		} else if (res instanceof SimilarityQuery) {
			this.addSimilarityQuery((SimilarityQuery) res);
		}

	}

	/**
	 * @param sq
	 *            The similarity query to add.
	 */
	protected void addSimilarityQuery(final SimilarityQuery sq) {
		for (final Resource r : sq.getResource()) {
			this.addResource(r);
		}

	}

	/**
	 * @param cq
	 *            The composed unit to add.
	 */
	protected void addComposedQuery(final ComposedQuery cq) {
		for (final Query q : cq.getQuery()) {
			this.addResource(q);
		}
	}

	/**
	 * @param rc
	 *            The resourceCollection to add.
	 */
	protected void addComposedResource(final ComposedResource rc) {
		for (final Resource r : rc.getResource()) {
			this.addResource(r);
		}
	}

	/**
	 * @param rs
	 *            The resultSet to add.
	 */
	protected void addResultSet(final ResultSet rs) {
		for (final Resource r : rs.getResource()) {
			this.addResource(r);
		}
		final PieceOfKnowledge p = rs.getPok();
		if (p != null) {
			this.addResource(p);
		}
	}

	/**
	 * @param a
	 *            The pok to add.
	 */
	protected void addPieceOfKnowledge(final PieceOfKnowledge pok) {
		final String rdf = PoKUtil.getPoKData(pok);
		if (!rdf.trim().equals("")) {
			this.loadFromRdfXml(rdf, true);
		}
	}

	/**
	 * @param mu
	 *            The mediaUnit to add.
	 */
	protected void addMediaUnit(final MediaUnit mu) {
//		for (final Segment seg : (mu).getSegment()) {
//			this.addResource(seg);
//		}
		if (mu instanceof Document) {
			this.addDocument((Document) mu);
		}
	}
	
	/**
	 * @param cu
	 */
	protected void addDocument(final Document cu) {
		for (final MediaUnit mu : cu.getMediaUnit()) {
			this.addResource(mu);
		}
	}

	/*
	 * (non-Javadoc)
	 * @see org.weblab_project.core.helper.RDFReader#getLitsOnPredSubj(java.lang. String, java.lang.String)
	 */
	public List<String> getLitsOnPredSubj(String uriSubj, String uriPred) {
		return this.bejh.getLitsOnPredSubj(uriSubj, uriPred);
	}

	/*
	 * (non-Javadoc)
	 * @see org.weblab_project.core.helper.RDFReader#getPredsOnSubj(java.lang.String)
	 */
	public Set<String> getPredsOnSubj(String uriSubj) {
		return this.bejh.getPredsOnSubj(uriSubj);
	}

	/*
	 * (non-Javadoc)
	 * @see org.weblab_project.core.helper.RDFReader#getRessOnPredSubj(java.lang. String, java.lang.String)
	 */
	public List<String> getRessOnPredSubj(String uriSubj, String uriPred) {
		return this.bejh.getRessOnPredSubj(uriSubj, uriPred);
	}

	/*
	 * (non-Javadoc)
	 * @see org.weblab_project.core.helper.RDFReader#getSubjs()
	 */
	public Set<String> getSubjs() {
		return this.bejh.getSubjs();
	}

	/*
	 * (non-Javadoc)
	 * @see org.weblab_project.core.helper.AnnotationHelper#getSubjsOnPred(java.lang .String)
	 */
	public Set<String> getSubjsOnPred(final String uriPred) {
		return this.bejh.getSubjsOnPred(uriPred);
	}

	/*
	 * (non-Javadoc)
	 * @see org.weblab_project.core.helper.AnnotationHelper#getSubjsOnPredLit(java .lang.String, java.lang.String)
	 */
	public Set<String> getSubjsOnPredLit(final String uriPred, final String litObj) {
		return this.bejh.getSubjsOnPredLit(uriPred, litObj);
	}

	/*
	 * (non-Javadoc)
	 * @see org.weblab_project.core.helper.AnnotationHelper#getSubjsOnPredRes(java .lang.String, java.lang.String)
	 */
	public Set<String> getSubjsOnPredRes(final String uriPred, final String uriObj) {
		return this.bejh.getSubjsOnPredRes(uriPred, uriObj);
	}

	/*
	 * (non-Javadoc)
	 * @see org.weblab_project.core.helper.AnnotationHelper#hasLitStat(java.lang. String, java.lang.String, java.lang.String)
	 */
	public boolean hasLitStat(final String uriSubj, final String uriPred, final String litObj) {
		return this.bejh.hasLitStat(uriSubj, uriPred, litObj);
	}

	/*
	 * (non-Javadoc)
	 * @see org.weblab_project.core.helper.AnnotationHelper#hasResStat(java.lang. String, java.lang.String, java.lang.String)
	 */
	public boolean hasResStat(final String uriSubj, final String uriPred, final String uriObj) {
		return this.bejh.hasResStat(uriSubj, uriPred, uriObj);
	}

	/*
	 * (non-Javadoc)
	 * @see org.weblab_project.core.helper.AnnotationHelper#isDefinedResource(java .lang.String)
	 */
	public boolean isDefinedResource(final String uri) {
		return this.bejh.isDefinedResource(uri);
	}

	/*
	 * (non-Javadoc)
	 * @see org.weblab_project.core.helper.RDFReader#loadFromRdfXml(java.lang.String)
	 */
	public void loadFromRdfXml(final String rdfXml) throws WebLabCheckedException {
		this.loadFromRdfXml(rdfXml, false);
	}

	/**
	 * @param rdfXml
	 *            An RDF/XML String to be loaded or added to teh current model
	 * @param add
	 *            Whether to add <code>rdfXml</code> to current model or to create a new model from scratch.
	 */
	protected void loadFromRdfXml(final String rdfXml, boolean add) {
		if (add && this.bejh != null) {
			this.bejh.addToModel(rdfXml);
		} else {
			this.bejh = new BackEndJenaHelper(rdfXml);
		}
	}

	/*
	 * (non-Javadoc)
	 * @see org.weblab_project.core.helper.ResourceHelper#getResource()
	 */
	public Resource getResource() {
		return this.resource;
	}

	/*
	 * (non-Javadoc)
	 * @see org.weblab_project.core.helper.ResourceHelper#isFullyDefinedResource( java.lang.String)
	 */
	public boolean isFullyDefinedResource(final String uri) {
		return this.isDefinedResource(uri) && this.resourcesMap.containsKey(uri);
	}

	/*
	 * (non-Javadoc)
	 * @see org.weblab_project.core.helper.RDFReader#getPreds()
	 */
	public Set<String> getPreds() {
		return this.bejh.getPreds();
	}

	/*
	 * (non-Javadoc)
	 * @see org.weblab_project.core.helper.RDFWriter#getRdfXml()
	 */
	public String getRdfXml() {
		if (this.bejh != null && this.resource != null) {
			return this.bejh.getRdfXml();
		}
		return "";
	}

	
	public List<String> getLitsOnPredSubj(String uriSubj, String uriPred,
			String language) {
		
		return this.bejh.getLitsOnPredSubj(uriSubj, uriPred, language);
	}

}