/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2009 EADS DEFENCE AND SECURITY SYSTEMS
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.utils;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;
import java.util.Set;
import java.util.Map.Entry;

import org.apache.commons.logging.LogFactory;

import com.hp.hpl.jena.rdf.arp.ARP;
import com.hp.hpl.jena.rdf.model.AnonId;
import com.hp.hpl.jena.rdf.model.Literal;
import com.hp.hpl.jena.rdf.model.Model;
import com.hp.hpl.jena.rdf.model.ModelFactory;
import com.hp.hpl.jena.rdf.model.Property;
import com.hp.hpl.jena.rdf.model.RDFNode;
import com.hp.hpl.jena.rdf.model.RSIterator;
import com.hp.hpl.jena.rdf.model.ReifiedStatement;
import com.hp.hpl.jena.rdf.model.Resource;
import com.hp.hpl.jena.rdf.model.Statement;
import com.hp.hpl.jena.rdf.model.StmtIterator;
import com.hp.hpl.jena.vocabulary.RDF;

/**
 * This class is used behind the six implementation provided in this package. This way it prevents the use of Jena dependencies in our implementations; the only
 * dependencies to Jena are in this class.
 * 
 * @author ymombrun
 * @date 12 sept. 08
 */
public class BackEndJenaHelper {

	// FIXME Change comments since it's in util instead of in helpers.
	// TODO Javadoc every public methods
	// TODO Check if BASE = null is better than "". This is new in Jena 2.6.0. "" --> enables relative URIs, null does not.

	private static final String SYNTAX = "RDF/XML";

	private static final String BASE = "";

	{
		new ARP().getOptions().setStrictErrorMode();
	}

	public Set<String> getSubjs() {
		return this.getSubjects(ResourceTypes.NAMED);
	}

	public Set<String> getPredsOnSubj(final String uriSubj) {
		if (uriSubj == null) {
			throw new RDFHelperException("uriSubj was null but shouldn't.");
		}
		Resource subj = this.createTempRes(uriSubj, ResourceTypes.NAMED);
		return this.getPredicates(subj);
	}

	public List<String> getLitsOnPredSubj(final String uriSubj, final String uriPred, final String language) {
		if (uriSubj == null || uriPred == null) {
			throw new RDFHelperException("Some args were null but shouldn't. uriSubj:" + uriSubj + " uriPred:"
					+ uriPred);
		}
		Resource subj = this.createTempRes(uriSubj, ResourceTypes.NAMED);
		Property pred = this.createTempProp(uriPred);
		return this.getObjects(subj, pred, RDFNodeTypes.LITERAL, language);
	}
	
	public List<String> getLitsOnPredSubj(final String uriSubj, final String uriPred) {
		return getLitsOnPredSubj(uriSubj, uriPred, null);
	}

	public List<String> getRessOnPredSubj(final String uriSubj, final String uriPred) {
		if (uriSubj == null || uriPred == null) {
			throw new RDFHelperException("Some args were null but shouldn't. uriSubj:" + uriSubj + " uriPred:"
					+ uriPred);
		}
		Resource subj = this.createTempRes(uriSubj, ResourceTypes.NAMED);
		Property pred = this.createTempProp(uriPred);
		return this.getObjects(subj, pred, RDFNodeTypes.NAMED_R);
	}

	public boolean hasLitStat(final String uriSubj, final String uriPred, final String litObj) {
		if (uriSubj == null || uriPred == null || litObj == null) {
			throw new RDFHelperException("Some args were null but shouldn't. uriSubj:" + uriSubj + " uriPred:"
					+ uriPred + " litObj:" + litObj);
		}
		Resource subj = this.createTempRes(uriSubj, ResourceTypes.NAMED);
		Property pred = this.createTempProp(uriPred);
		Literal lit = this.createTempLit(litObj);
		return this.containsStatement(subj, pred, lit);
	}

	public boolean hasResStat(final String uriSubj, final String uriPred, final String uriObj) {
		if (uriSubj == null || uriPred == null || uriObj == null) {
			throw new RDFHelperException("Some args were null but shouldn't. uriSubj:" + uriSubj + " uriPred:"
					+ uriPred + " uriObj:" + uriObj);
		}
		Resource subj = this.createTempRes(uriSubj, ResourceTypes.NAMED);
		Property pred = this.createTempProp(uriPred);
		Resource obj = this.createTempRes(uriObj, ResourceTypes.NAMED);
		return this.containsStatement(subj, pred, obj);
	}

	public void createLitStat(final String uriSubj, final String uriPred, final String litObj) {
		createLitStat(uriSubj, uriPred, litObj, null);
	}
	
	public void createLitStat(final String uriSubj, final String uriPred, final String litObj, final String language) {
		if (uriSubj == null || uriPred == null || litObj == null) {
			throw new RDFHelperException("Some args were null but shouldn't. uriSubj:" + uriSubj + " uriPred:"
					+ uriPred + " litObj:" + litObj);
		}
		Resource subj = this.createTempRes(uriSubj, ResourceTypes.NAMED);
		Property pred = this.createTempProp(uriPred);
		Literal lit = this.createTempLit(litObj,language);
		this.createStatement(subj, pred, lit);
	}

	public void createResStat(final String uriSubj, final String uriPred, final String uriObj) {
		if (uriSubj == null || uriPred == null || uriObj == null) {
			throw new RDFHelperException("Some args were null but shouldn't. uriSubj:" + uriSubj + " uriPred:"
					+ uriPred + " uriObj:" + uriObj);
		}
		Resource subj = this.createTempRes(uriSubj, ResourceTypes.NAMED);
		Property pred = this.createTempProp(uriPred);
		Resource obj = this.createTempRes(uriObj, ResourceTypes.NAMED);
		this.createStatement(subj, pred, obj);
	}

	public void removeLitStat(final String uriSubj, final String uriPred, final String litObj) {
		if (uriSubj == null || uriPred == null || litObj == null) {
			throw new RDFHelperException("Some args were null but shouldn't. uriSubj:" + uriSubj + " uriPred:"
					+ uriPred + " litObj:" + litObj);
		}
		Resource subj = this.createTempRes(uriSubj, ResourceTypes.NAMED);
		Property pred = this.createTempProp(uriPred);
		Literal lit = this.createTempLit(litObj);
		this.removeStatement(subj, pred, lit);
	}

	public void removeResStat(final String uriSubj, final String uriPred, final String uriObj) {
		if (uriSubj == null || uriPred == null || uriObj == null) {
			throw new RDFHelperException("Some args were null but shouldn't. uriSubj:" + uriSubj + " uriPred:"
					+ uriPred + " uriObj:" + uriObj);
		}
		Resource subj = this.createTempRes(uriSubj, ResourceTypes.NAMED);
		Property pred = this.createTempProp(uriPred);
		Resource obj = this.createTempRes(uriObj, ResourceTypes.NAMED);
		this.removeStatement(subj, pred, obj);
	}

	public void removeStatsOnSubj(final String uriSubj) {
		if (uriSubj == null) {
			throw new RDFHelperException("uriSubj was null but shouldn't.");
		}
		Resource subj = this.createTempRes(uriSubj, ResourceTypes.NAMED);
		this.removeStatement(subj, null, null);
	}

	public void removeStatsWithPred(final String uriPred) {
		if (uriPred == null) {
			throw new RDFHelperException("uriPred was null but shouldn't.");
		}
		Property pred = this.createTempProp(uriPred);
		this.removeStatement(null, pred, null);
	}

	public void removeStatsWithLit(final String litObj) {
		if (litObj == null) {
			throw new RDFHelperException("litObj was null but shouldn't.");
		}
		Literal lit = this.createTempLit(litObj);
		this.removeStatement(null, null, lit);
	}

	public void removeStatsWithRes(final String uriObj) {
		if (uriObj == null) {
			throw new RDFHelperException("uriObj was null but shouldn't.");
		}
		Resource obj = this.createTempRes(uriObj, ResourceTypes.NAMED);
		this.removeStatement(null, null, obj);
	}

	public Set<String> getSubjsOnPred(final String uriPred) {
		if (uriPred == null) {
			throw new RDFHelperException("uriPred was null but shouldn't.");
		}
		Property pred = this.createTempProp(uriPred);
		return this.getSubjects(pred, null, ResourceTypes.NAMED);
	}

	public Set<String> getSubjsOnPredLit(final String uriPred, final String litObj) {
		if (uriPred == null || litObj == null) {
			throw new RDFHelperException("Some args were null but shouldn't. uriPred:" + uriPred + " litObj:" + litObj);
		}
		Property pred = this.createTempProp(uriPred);
		Literal obj = this.createTempLit(litObj);
		return this.getSubjects(pred, obj, ResourceTypes.NAMED);
	}

	public Set<String> getSubjsOnPredRes(final String uriPred, final String uriObj) {
		if (uriPred == null || uriObj == null) {
			throw new RDFHelperException("Some args were null but shouldn't. uriPred:" + uriPred + " uriObj:" + uriObj);
		}
		Property pred = this.createTempProp(uriPred);
		Resource obj = this.createTempRes(uriObj, ResourceTypes.NAMED);
		return this.getSubjects(pred, obj, ResourceTypes.NAMED);
	}

	public boolean isDefinedResource(final String uri) {
		if (uri == null) {
			throw new RDFHelperException("Uri args were null but shouldn't.");
		}
		final Resource res = this.createTempRes(uri, ResourceTypes.NAMED);
		return this.containsResource(res);
	}

	public void createAnonResStat(final String uriSubj, final String uriPred, final String idObj) {
		if (uriSubj == null || uriPred == null || idObj == null) {
			throw new RDFHelperException("Some args were null " + "but shouldn't. uriSubj:" + uriSubj + " uriPred:"
					+ uriPred + " idObj:" + idObj);
		}
		Resource s = this.createTempRes(uriSubj, ResourceTypes.NAMED);
		Property pred = this.createTempProp(uriPred);
		Resource obj = this.createTempRes(idObj, ResourceTypes.ANON);
		this.createStatement(s, pred, obj);
	}

	public void createAnonResStatOnAnonSubj(final String idSubj, final String uriPred, final String idObj) {
		if (idSubj == null || uriPred == null || idObj == null) {
			throw new RDFHelperException("Some args were null but " + "shouldn't. idSubj:" + idSubj + " uriPred:"
					+ uriPred + " idObj:" + idObj);
		}
		Resource subj = this.createTempRes(idSubj, ResourceTypes.ANON);
		Property pred = this.createTempProp(uriPred);
		Resource obj = this.createTempRes(idObj, ResourceTypes.ANON);
		this.createStatement(subj, pred, obj);
	}

	public String createAnonResStatReif(final String uriSubj, final String uriPred, final String idObj,
			final Map<String, String> mapPredLit, final Map<String, String> mapPredUri,
			final Map<String, String> mapPredId) {
		return this.createAnonResStatReif(uriSubj, uriPred, idObj, mapPredLit, mapPredUri, mapPredId, null);
	}

	public String createAnonResStatReifOnAnonSubj(final String idSubj, final String uriPred, final String idObj,
			final Map<String, String> mapPredLit, final Map<String, String> mapPredUri,
			final Map<String, String> mapPredId) {
		return this.createAnonResStatReifOnAnonSubj(idSubj, uriPred, idObj, mapPredLit, mapPredUri, mapPredId, null);
	}

	public void createLitStatOnAnonSubj(final String idSubj, final String uriPred, final String litObj) {
		if (idSubj == null || uriPred == null || litObj == null) {
			throw new RDFHelperException("Some args were null but " + "shouldn't. idSubj:" + idSubj + " uriPred:"
					+ uriPred + " litObj:" + litObj);
		}
		Resource subj = this.createTempRes(idSubj, ResourceTypes.ANON);
		Property pred = this.createTempProp(uriPred);
		Literal lit = this.createTempLit(litObj);
		this.createStatement(subj, pred, lit);
	}

	public String createLitStatReif(final String uriSubj, final String uriPred, final String litObj,
			final Map<String, String> mapPredLit, final Map<String, String> mapPredUri,
			final Map<String, String> mapPredId) {
		return this.createLitStatReif(uriSubj, uriPred, litObj, mapPredLit, mapPredUri, mapPredId, null);
	}

	public String createLitStatReifOnAnonSubj(final String idSubj, final String uriPred, final String litObj,
			final Map<String, String> mapPredLit, final Map<String, String> mapPredUri,
			final Map<String, String> mapPredId) {
		return this.createLitStatReifOnAnonSubj(idSubj, uriPred, litObj, mapPredLit, mapPredUri, mapPredId, null);
	}

	public void createResStatOnAnonSubj(final String idSubj, final String uriPred, final String uriObj) {
		if (idSubj == null || uriPred == null || uriObj == null) {
			throw new RDFHelperException("Some args were null " + "but shouldn't. idSubj:" + idSubj + " uriPred:"
					+ uriPred + " uriObj:" + uriObj);
		}
		Resource subj = this.createTempRes(idSubj, ResourceTypes.ANON);
		Property pred = this.createTempProp(uriPred);
		Resource obj = this.createTempRes(uriObj, ResourceTypes.NAMED);
		this.createStatement(subj, pred, obj);
	}

	public String createResStatReif(final String uriSubj, final String uriPred, final String uriObj,
			final Map<String, String> mapPredLit, final Map<String, String> mapPredUri,
			final Map<String, String> mapPredId) {
		return this.createResStatReif(uriSubj, uriPred, uriObj, mapPredLit, mapPredUri, mapPredId, null);
	}

	public String createResStatReifOnAnonSubj(final String idSubj, final String uriPred, final String uriObj,
			final Map<String, String> mapPredLit, final Map<String, String> mapPredUri,
			final Map<String, String> mapPredId) {
		return this.createResStatReifOnAnonSubj(idSubj, uriPred, uriObj, mapPredLit, mapPredUri, mapPredId, null);
	}

	public List<String> getAnonResOnPredAnonResStatReif(final String uriSubj, final String uriPred, final String idObj,
			final String uriReif) {
		if (uriSubj == null || uriPred == null || idObj == null || uriReif == null) {
			throw new RDFHelperException("Some args were null " + "but shouldn't. uriSubj:" + uriSubj + " uriPred:"
					+ uriPred + " uriObj:" + idObj + " uriReif:" + uriReif);
		}
		Resource subj = this.createTempRes(uriSubj, ResourceTypes.NAMED);
		Property pred = this.createTempProp(uriPred);
		Property p = this.createTempProp(uriReif);
		Resource obj = this.createTempRes(idObj, ResourceTypes.ANON);
		List<String> objects = new ArrayList<String>();
		for (ReifiedStatement rs : this.getReifStats(subj, pred, obj)) {
			objects.addAll(this.getObjects(rs, p, RDFNodeTypes.ANON_R));
		}
		return objects;
	}

	public List<String> getAnonResOnPredAnonResStatReifOnAnonSubj(final String idSubj, final String uriPred,
			final String idObj, final String uriReif) {
		if (idSubj == null || uriPred == null || idObj == null || uriReif == null) {
			throw new RDFHelperException("Some args were null " + "but shouldn't. idSubj:" + idSubj + " uriPred:"
					+ uriPred + " idObj:" + idObj + " uriReif:" + uriReif);
		}
		Resource subj = this.createTempRes(idSubj, ResourceTypes.ANON);
		Property pred = this.createTempProp(uriPred);
		Property p = this.createTempProp(uriReif);
		Resource obj = this.createTempRes(idObj, ResourceTypes.ANON);
		List<String> objects = new ArrayList<String>();
		for (ReifiedStatement rs : this.getReifStats(subj, pred, obj)) {
			objects.addAll(this.getObjects(rs, p, RDFNodeTypes.ANON_R));
		}
		return objects;
	}

	public List<String> getAnonResOnPredAnonSubj(final String idSubj, final String uriPred) {
		if (idSubj == null || uriPred == null) {
			throw new RDFHelperException("Some args were null " + "but shouldn't. idSubj:" + idSubj + " uriPred:"
					+ uriPred);
		}
		Resource subj = this.createTempRes(idSubj, ResourceTypes.ANON);
		Property pred = this.createTempProp(uriPred);
		return this.getObjects(subj, pred, RDFNodeTypes.ANON_R);
	}

	public List<String> getAnonResOnPredLitStatReif(final String uriSubj, final String uriPred, final String litObj,
			final String uriReif) {
		if (uriSubj == null || uriPred == null || litObj == null || uriReif == null) {
			throw new RDFHelperException("Some args were null " + "but shouldn't. uriSubj:" + uriSubj + " uriPred:"
					+ uriPred + " litObj:" + litObj + " uriReif:" + uriReif);
		}
		Resource subj = this.createTempRes(uriSubj, ResourceTypes.NAMED);
		Property pred = this.createTempProp(uriPred);
		Property p = this.createTempProp(uriReif);
		Literal obj = this.createTempLit(litObj);
		List<String> objects = new ArrayList<String>();
		for (ReifiedStatement rs : this.getReifStats(subj, pred, obj)) {
			objects.addAll(this.getObjects(rs, p, RDFNodeTypes.ANON_R));
		}
		return objects;
	}

	public List<String> getAnonResOnPredLitStatReifOnAnonSubj(final String idSubj, final String uriPred,
			final String litObj, final String uriReif) {
		if (idSubj == null || uriPred == null || litObj == null || uriReif == null) {
			throw new RDFHelperException("Some args were null " + "but shouldn't. idSubj:" + idSubj + " uriPred:"
					+ uriPred + " litObj:" + litObj + " uriReif:" + uriReif);
		}
		Resource subj = this.createTempRes(idSubj, ResourceTypes.ANON);
		Property pred = this.createTempProp(uriPred);
		Property p = this.createTempProp(uriReif);
		Literal obj = this.createTempLit(litObj);
		List<String> objects = new ArrayList<String>();
		for (ReifiedStatement rs : this.getReifStats(subj, pred, obj)) {
			objects.addAll(this.getObjects(rs, p, RDFNodeTypes.ANON_R));
		}
		return objects;
	}

	public List<String> getAnonResOnPredResStatReif(final String uriSubj, final String uriPred, final String uriObj,
			final String uriReif) {
		if (uriSubj == null || uriPred == null || uriObj == null || uriReif == null) {
			throw new RDFHelperException("Some args were null " + "but shouldn't. uriSubj:" + uriSubj + " uriPred:"
					+ uriPred + " uriObj:" + uriObj + " uriReif:" + uriReif);
		}
		Resource subj = this.createTempRes(uriSubj, ResourceTypes.NAMED);
		Property pred = this.createTempProp(uriPred);
		Property p = this.createTempProp(uriReif);
		Resource obj = this.createTempRes(uriObj, ResourceTypes.NAMED);
		List<String> objects = new ArrayList<String>();
		for (ReifiedStatement rs : this.getReifStats(subj, pred, obj)) {
			objects.addAll(this.getObjects(rs, p, RDFNodeTypes.ANON_R));
		}
		return objects;
	}

	public List<String> getAnonResOnPredResStatReifOnAnonSubj(final String idSubj, final String uriPred,
			final String uriObj, final String uriReif) {
		if (idSubj == null || uriPred == null || uriObj == null || uriReif == null) {
			throw new RDFHelperException("Some args were null " + "but shouldn't. idSubj:" + idSubj + " uriPred:"
					+ uriPred + " uriObj:" + uriObj + " uriReif:" + uriReif);
		}
		Resource subj = this.createTempRes(idSubj, ResourceTypes.ANON);
		Property pred = this.createTempProp(uriPred);
		Property p = this.createTempProp(uriReif);
		Resource obj = this.createTempRes(uriObj, ResourceTypes.NAMED);
		List<String> objects = new ArrayList<String>();
		for (ReifiedStatement rs : this.getReifStats(subj, pred, obj)) {
			objects.addAll(this.getObjects(rs, p, RDFNodeTypes.ANON_R));
		}
		return objects;
	}

	public List<String> getAnonResOnPredSubj(final String uriSubj, final String uriPred) {
		if (uriSubj == null || uriPred == null) {
			throw new RDFHelperException("Some args were null but " + "shouldn't. uriSubj:" + uriSubj + " uriPred:"
					+ uriPred);
		}
		Resource s = this.createTempRes(uriSubj, ResourceTypes.NAMED);
		Property pred = this.createTempProp(uriPred);
		return this.getObjects(s, pred, RDFNodeTypes.ANON_R);
	}

	public Set<String> getAnonSubj() {
		return this.getSubjects(ResourceTypes.ANON);
	}

	public Set<String> getAnonSubjOnPred(final String uriPred) {
		if (uriPred == null) {
			throw new RDFHelperException("uriPred was null but shouldn't");
		}
		Property pred = this.createTempProp(uriPred);
		return this.getSubjects(pred, null, ResourceTypes.ANON);
	}

	public Set<String> getAnonSubjOnPredAnonRes(final String uriPred, final String idObj) {
		if (uriPred == null || idObj == null) {
			throw new RDFHelperException("Some args were null " + "but shouldn't. uriPred:" + uriPred + " idObj:"
					+ idObj);
		}
		Property pred = this.createTempProp(uriPred);
		Resource obj = this.createTempRes(idObj, ResourceTypes.ANON);
		return this.getSubjects(pred, obj, ResourceTypes.ANON);
	}

	public Set<String> getAnonSubjOnPredLit(final String uriPred, final String litObj) {
		if (uriPred == null || litObj == null) {
			throw new RDFHelperException("Some args were null " + "but shouldn't. uriPred:" + uriPred + " litObj:"
					+ litObj);
		}
		Property pred = this.createTempProp(uriPred);
		Literal obj = this.createTempLit(litObj);
		return this.getSubjects(pred, obj, ResourceTypes.ANON);
	}

	public Set<String> getAnonSubjOnPredRes(final String uriPred, final String uriObj) {
		if (uriPred == null || uriObj == null) {
			throw new RDFHelperException("Some args were null " + "but shouldn't. uriPred:" + uriPred + " uriObj:"
					+ uriObj);
		}
		Property pred = this.createTempProp(uriPred);
		Resource obj = this.createTempRes(uriObj, ResourceTypes.NAMED);
		return this.getSubjects(pred, obj, ResourceTypes.ANON);
	}

	public List<String> getLitOnPredAnonResStatReif(final String uriSubj, final String uriPred, final String idObj,
			final String uriReif) {
		if (uriSubj == null || uriPred == null || idObj == null || uriReif == null) {
			throw new RDFHelperException("Some args were null " + "but shouldn't. uriSubj:" + uriSubj + " uriPred:"
					+ uriPred + " idObj:" + idObj + " uriReif:" + uriReif);
		}
		Resource subj = this.createTempRes(uriSubj, ResourceTypes.NAMED);
		Property pred = this.createTempProp(uriPred);
		Property p = this.createTempProp(uriReif);
		Resource obj = this.createTempRes(idObj, ResourceTypes.ANON);
		List<String> objects = new ArrayList<String>();
		for (ReifiedStatement rs : this.getReifStats(subj, pred, obj)) {
			objects.addAll(this.getObjects(rs, p, RDFNodeTypes.LITERAL));
		}
		return objects;
	}

	public List<String> getLitOnPredAnonResStatReifOnAnonSubj(final String idSubj, final String uriPred,
			final String idObj, final String uriReif) {
		if (idSubj == null || uriPred == null || idObj == null || uriReif == null) {
			throw new RDFHelperException("Some args were null " + "but shouldn't. uriSubj:" + idSubj + " uriPred:"
					+ uriPred + " idObj:" + idObj + " uriReif:" + uriReif);
		}
		Resource subj = this.createTempRes(idSubj, ResourceTypes.ANON);
		Property pred = this.createTempProp(uriPred);
		Property p = this.createTempProp(uriReif);
		Resource obj = this.createTempRes(idObj, ResourceTypes.ANON);
		List<String> objects = new ArrayList<String>();
		for (ReifiedStatement rs : this.getReifStats(subj, pred, obj)) {
			objects.addAll(this.getObjects(rs, p, RDFNodeTypes.LITERAL));
		}
		return objects;
	}

	public List<String> getLitOnPredAnonSubj(final String idSubj, final String uriPred) {
		if (idSubj == null || uriPred == null) {
			throw new RDFHelperException("Some args were null " + "but shouldn't. idSubj:" + idSubj + " uriPred:"
					+ uriPred);
		}
		Resource subj = this.createTempRes(idSubj, ResourceTypes.ANON);
		Property pred = this.createTempProp(uriPred);
		return this.getObjects(subj, pred, RDFNodeTypes.LITERAL);
	}

	public List<String> getLitOnPredLitStatReif(final String uriSubj, final String uriPred, final String litObj,
			final String uriReif) {
		if (uriSubj == null || uriPred == null || litObj == null || uriReif == null) {
			throw new RDFHelperException("Some args were null " + "but shouldn't. uriSubj:" + uriSubj + " uriPred:"
					+ uriPred + " litObj:" + litObj + " uriReif:" + uriReif);
		}
		Resource subj = this.createTempRes(uriSubj, ResourceTypes.NAMED);
		Property pred = this.createTempProp(uriPred);
		Property p = this.createTempProp(uriReif);
		Literal obj = this.createTempLit(litObj);
		List<String> objects = new ArrayList<String>();
		for (ReifiedStatement rs : this.getReifStats(subj, pred, obj)) {
			objects.addAll(this.getObjects(rs, p, RDFNodeTypes.LITERAL));
		}
		return objects;
	}

	public List<String> getLitOnPredLitStatReifOnAnonSubj(final String idSubj, final String uriPred,
			final String litObj, final String uriReif) {
		if (idSubj == null || uriPred == null || litObj == null || uriReif == null) {
			throw new RDFHelperException("Some args were null " + "but shouldn't. idSubj:" + idSubj + " uriPred:"
					+ uriPred + " litObj:" + litObj + " uriReif:" + uriReif);
		}
		Resource subj = this.createTempRes(idSubj, ResourceTypes.ANON);
		Property pred = this.createTempProp(uriPred);
		Property p = this.createTempProp(uriReif);
		Literal obj = this.createTempLit(litObj);
		List<String> objects = new ArrayList<String>();
		for (ReifiedStatement rs : this.getReifStats(subj, pred, obj)) {
			objects.addAll(this.getObjects(rs, p, RDFNodeTypes.LITERAL));
		}
		return objects;
	}

	public List<String> getLitOnPredResStatReif(final String uriSubj, final String uriPred, final String uriObj,
			final String uriReif) {
		if (uriSubj == null || uriPred == null || uriObj == null || uriReif == null) {
			throw new RDFHelperException("Some args were null " + "but shouldn't. uriSubj:" + uriSubj + " uriPred:"
					+ uriPred + " uriObj:" + uriObj + " uriReif:" + uriReif);
		}
		Resource subj = this.createTempRes(uriSubj, ResourceTypes.NAMED);
		Property pred = this.createTempProp(uriPred);
		Property p = this.createTempProp(uriReif);
		Resource obj = this.createTempRes(uriObj, ResourceTypes.NAMED);
		List<String> objects = new ArrayList<String>();
		for (ReifiedStatement rs : this.getReifStats(subj, pred, obj)) {
			objects.addAll(this.getObjects(rs, p, RDFNodeTypes.LITERAL));
		}
		return objects;
	}

	public List<String> getLitsOnPredResStatReifOnAnonSubj(final String idSubj, final String uriPred,
			final String uriObj, final String uriReif) {
		if (idSubj == null || uriPred == null || uriObj == null || uriReif == null) {
			throw new RDFHelperException("Some args were null " + "but shouldn't. idSubj:" + idSubj + " uriPred:"
					+ uriPred + " uriObj:" + uriObj + " uriReif:" + uriReif);
		}
		Resource subj = this.createTempRes(idSubj, ResourceTypes.ANON);
		Property pred = this.createTempProp(uriPred);
		Property p = this.createTempProp(uriReif);
		Resource obj = this.createTempRes(uriObj, ResourceTypes.NAMED);
		List<String> objects = new ArrayList<String>();
		for (ReifiedStatement rs : this.getReifStats(subj, pred, obj)) {
			objects.addAll(this.getObjects(rs, p, RDFNodeTypes.LITERAL));
		}
		return objects;
	}

	public Set<String> getPredOnAnonResStatReif(final String uriSubj, final String uriPred, final String idObj) {
		if (uriSubj == null || uriPred == null || idObj == null) {
			throw new RDFHelperException("Some args were null " + "but shouldn't. uriSubj:" + uriSubj + " uriPred:"
					+ uriPred + " idObj:" + idObj);
		}
		Resource subj = this.createTempRes(uriSubj, ResourceTypes.NAMED);
		Property pred = this.createTempProp(uriPred);
		Resource obj = this.createTempRes(idObj, ResourceTypes.ANON);
		Set<String> preds = new HashSet<String>();
		for (ReifiedStatement rs : this.getReifStats(subj, pred, obj)) {
			preds.addAll(this.getPredicates(rs));
		}
		return preds;
	}

	public Set<String> getPredOnAnonResStatReifOnAnonSubj(final String idSubj, final String uriPred, final String idObj) {
		if (idSubj == null || uriPred == null || idObj == null) {
			throw new RDFHelperException("Some args were null " + "but shouldn't. idSubj:" + idSubj + " uriPred:"
					+ uriPred + " idObj:" + idObj);
		}
		Resource subj = this.createTempRes(idSubj, ResourceTypes.ANON);
		Property pred = this.createTempProp(uriPred);
		Resource obj = this.createTempRes(idObj, ResourceTypes.ANON);
		Set<String> preds = new HashSet<String>();
		for (ReifiedStatement rs : this.getReifStats(subj, pred, obj)) {
			preds.addAll(this.getPredicates(rs));
		}
		return preds;
	}

	public Set<String> getPredOnAnonSubj(final String idSubj) {
		if (idSubj == null) {
			throw new RDFHelperException("idSubj was null but shouldn't.");
		}
		Resource subj = this.createTempRes(idSubj, ResourceTypes.ANON);
		return this.getPredicates(subj);
	}

	public Set<String> getPredOnLitStatReif(final String uriSubj, final String uriPred, final String litObj) {
		if (uriSubj == null || uriPred == null || litObj == null) {
			throw new RDFHelperException("Some args were null " + "but shouldn't. uriSubj:" + uriSubj + " uriPred:"
					+ uriPred + " litObj:" + litObj);
		}
		Resource subj = this.createTempRes(uriSubj, ResourceTypes.NAMED);
		Property pred = this.createTempProp(uriPred);
		Literal obj = this.createTempLit(litObj);
		Set<String> preds = new HashSet<String>();
		for (ReifiedStatement rs : this.getReifStats(subj, pred, obj)) {
			preds.addAll(this.getPredicates(rs));
		}
		return preds;
	}

	public Set<String> getPredOnLitStatReifOnAnonSubj(final String idSubj, final String uriPred, final String litObj) {
		if (idSubj == null || uriPred == null || litObj == null) {
			throw new RDFHelperException("Some args were null " + "but shouldn't. idSubj:" + idSubj + " uriPred:"
					+ uriPred + " litObj:" + litObj);
		}
		Resource subj = this.createTempRes(idSubj, ResourceTypes.ANON);
		Property pred = this.createTempProp(uriPred);
		Literal obj = this.createTempLit(litObj);
		Set<String> preds = new HashSet<String>();
		for (ReifiedStatement rs : this.getReifStats(subj, pred, obj)) {
			preds.addAll(this.getPredicates(rs));
		}
		return preds;
	}

	public Set<String> getPredOnResStatReif(final String uriSubj, final String uriPred, final String uriObj) {
		if (uriSubj == null || uriPred == null || uriObj == null) {
			throw new RDFHelperException("Some args were null " + "but shouldn't. uriSubj:" + uriSubj + " uriPred:"
					+ uriPred + " uriObj:" + uriObj);
		}
		Resource subj = this.createTempRes(uriSubj, ResourceTypes.NAMED);
		Property pred = this.createTempProp(uriPred);
		Resource obj = this.createTempRes(uriObj, ResourceTypes.NAMED);
		Set<String> preds = new HashSet<String>();
		for (ReifiedStatement rs : this.getReifStats(subj, pred, obj)) {
			preds.addAll(this.getPredicates(rs));
		}
		return preds;
	}

	public Set<String> getPredOnResStatReifOnAnonSubj(final String idSubj, final String uriPred, final String uriObj) {
		if (idSubj == null || uriPred == null || uriObj == null) {
			throw new RDFHelperException("Some args were null " + "but shouldn't. idSubj:" + idSubj + " uriPred:"
					+ uriPred + " uriObj:" + uriObj);
		}
		Resource subj = this.createTempRes(idSubj, ResourceTypes.ANON);
		Property pred = this.createTempProp(uriPred);
		Resource obj = this.createTempRes(uriObj, ResourceTypes.NAMED);
		Set<String> preds = new HashSet<String>();
		for (ReifiedStatement rs : this.getReifStats(subj, pred, obj)) {
			preds.addAll(this.getPredicates(rs));
		}
		return preds;
	}

	public List<String> getResOnPredAnonResStatReif(final String uriSubj, final String uriPred, final String idObj,
			final String uriReif) {
		if (uriSubj == null || uriPred == null || idObj == null || uriReif == null) {
			throw new RDFHelperException("Some args were null " + "but shouldn't. uriSubj:" + uriSubj + " uriPred:"
					+ uriPred + " idObj:" + idObj + " uriReif:" + uriReif);
		}
		Resource subj = this.createTempRes(uriSubj, ResourceTypes.NAMED);
		Property pred = this.createTempProp(uriPred);
		Property p = this.createTempProp(uriReif);
		Resource obj = this.createTempRes(idObj, ResourceTypes.ANON);
		List<String> objects = new ArrayList<String>();
		for (ReifiedStatement rs : this.getReifStats(subj, pred, obj)) {
			objects.addAll(this.getObjects(rs, p, RDFNodeTypes.NAMED_R));
		}
		return objects;
	}

	public List<String> getResOnPredAnonResStatReifOnAnonSubj(final String idSubj, final String uriPred,
			final String idObj, final String uriReif) {
		if (idSubj == null || uriPred == null || idObj == null || uriReif == null) {
			throw new RDFHelperException("Some args were null " + "but shouldn't. idSubj:" + idSubj + " uriPred:"
					+ uriPred + " idObj:" + idObj + " uriReif:" + uriReif);
		}
		Resource subj = this.createTempRes(idSubj, ResourceTypes.ANON);
		Property pred = this.createTempProp(uriPred);
		Property p = this.createTempProp(uriReif);
		Resource obj = this.createTempRes(idObj, ResourceTypes.ANON);
		List<String> objects = new ArrayList<String>();
		for (ReifiedStatement rs : this.getReifStats(subj, pred, obj)) {
			objects.addAll(this.getObjects(rs, p, RDFNodeTypes.NAMED_R));
		}
		return objects;
	}

	public List<String> getResOnPredAnonSubj(final String idSubj, final String uriPred) {
		if (idSubj == null || uriPred == null) {
			throw new RDFHelperException("Some args were null " + "but shouldn't. idSubj:" + idSubj + " uriPred:"
					+ uriPred);
		}
		Resource subj = this.createTempRes(idSubj, ResourceTypes.ANON);
		Property pred = this.createTempProp(uriPred);
		return this.getObjects(subj, pred, RDFNodeTypes.NAMED_R);
	}

	public List<String> getResOnPredLitStatReif(final String uriSubj, final String uriPred, final String litObj,
			final String uriReif) {
		if (uriSubj == null || uriPred == null || litObj == null || uriReif == null) {
			throw new RDFHelperException("Some args were null " + "but shouldn't. uriSubj:" + uriSubj + " uriPred:"
					+ uriPred + " litObj:" + litObj + " uriReif:" + uriReif);
		}
		Resource subj = this.createTempRes(uriSubj, ResourceTypes.NAMED);
		Property pred = this.createTempProp(uriPred);
		Property p = this.createTempProp(uriReif);
		Literal obj = this.createTempLit(litObj);
		List<String> objects = new ArrayList<String>();
		for (ReifiedStatement rs : this.getReifStats(subj, pred, obj)) {
			objects.addAll(this.getObjects(rs, p, RDFNodeTypes.NAMED_R));
		}
		return objects;
	}

	public List<String> getResOnPredLitStatReifOnAnonSubj(final String idSubj, final String uriPred,
			final String litObj, final String uriReif) {
		if (idSubj == null || uriPred == null || litObj == null || uriReif == null) {
			throw new RDFHelperException("Some args were null " + "but shouldn't. idSubj:" + idSubj + " uriPred:"
					+ uriPred + " litObj:" + litObj + " uriReif:" + uriReif);
		}
		Resource subj = this.createTempRes(idSubj, ResourceTypes.ANON);
		Property pred = this.createTempProp(uriPred);
		Property p = this.createTempProp(uriReif);
		Literal obj = this.createTempLit(litObj);
		List<String> objects = new ArrayList<String>();
		for (ReifiedStatement rs : this.getReifStats(subj, pred, obj)) {
			objects.addAll(this.getObjects(rs, p, RDFNodeTypes.NAMED_R));
		}
		return objects;
	}

	public List<String> getResOnPredResStatReif(final String uriSubj, final String uriPred, final String uriObj,
			final String uriReif) {
		if (uriSubj == null || uriPred == null || uriObj == null || uriReif == null) {
			throw new RDFHelperException("Some args were null " + "but shouldn't. uriSubj:" + uriSubj + " uriPred:"
					+ uriPred + " uriObj:" + uriObj + " uriReif:" + uriReif);
		}
		Resource subj = this.createTempRes(uriSubj, ResourceTypes.NAMED);
		Property pred = this.createTempProp(uriPred);
		Property p = this.createTempProp(uriReif);
		Resource obj = this.createTempRes(uriObj, ResourceTypes.NAMED);
		List<String> objects = new ArrayList<String>();
		for (ReifiedStatement rs : this.getReifStats(subj, pred, obj)) {
			objects.addAll(this.getObjects(rs, p, RDFNodeTypes.NAMED_R));
		}
		return objects;
	}

	public List<String> getResOnPredResStatReifOnAnonSubj(final String idSubj, final String uriPred,
			final String uriObj, final String uriReif) {
		if (idSubj == null || uriPred == null || uriObj == null || uriReif == null) {
			throw new RDFHelperException("Some args were null " + "but shouldn't. idSubj:" + idSubj + " uriPred:"
					+ uriPred + " uriObj:" + uriObj + " uriReif:" + uriReif);
		}
		Resource subj = this.createTempRes(idSubj, ResourceTypes.ANON);
		Property pred = this.createTempProp(uriPred);
		Property p = this.createTempProp(uriReif);
		Resource obj = this.createTempRes(uriObj, ResourceTypes.NAMED);
		List<String> objects = new ArrayList<String>();
		for (ReifiedStatement rs : this.getReifStats(subj, pred, obj)) {
			objects.addAll(this.getObjects(rs, p, RDFNodeTypes.NAMED_R));
		}
		return objects;
	}

	public Set<String> getSubjOnPredAnonRes(final String uriPred, final String idObj) {
		if (uriPred == null || idObj == null) {
			throw new RDFHelperException("Some args were null " + "but shouldn't. uriPred:" + uriPred + " idObj:"
					+ idObj);
		}
		Property pred = this.createTempProp(uriPred);
		Resource obj = this.createTempRes(idObj, ResourceTypes.ANON);
		return this.getSubjects(pred, obj, ResourceTypes.NAMED);
	}

	public boolean hasAnonResStat(final String uriSubj, final String uriPred, final String idObj) {
		if (uriSubj == null || uriPred == null || idObj == null) {
			throw new RDFHelperException("Some args were null but " + "shouldn't. uriSubj:" + uriSubj + " uriPred:"
					+ uriPred + " idObj:" + idObj);
		}
		Resource s = this.createTempRes(uriSubj, ResourceTypes.NAMED);
		Property pred = this.createTempProp(uriPred);
		Resource obj = this.createTempRes(idObj, ResourceTypes.ANON);
		return this.containsStatement(s, pred, obj);
	}

	public boolean hasAnonResStatOnAnonSubj(final String idSubj, final String uriPred, final String idObj) {
		if (idSubj == null || uriPred == null || idObj == null) {
			throw new RDFHelperException("Some args were null but " + "shouldn't. idSubj:" + idSubj + " uriPred:"
					+ uriPred + " idObj:" + idObj);
		}
		Resource subj = this.createTempRes(idSubj, ResourceTypes.ANON);
		Property pred = this.createTempProp(uriPred);
		Resource obj = this.createTempRes(idObj, ResourceTypes.ANON);
		return this.containsStatement(subj, pred, obj);
	}

	public boolean hasLitStatOnAnonSubj(final String idSubj, final String uriPred, final String litObj) {
		if (idSubj == null || uriPred == null || litObj == null) {
			throw new RDFHelperException("Some args were null but " + "shouldn't. idSubj:" + idSubj + " uriPred:"
					+ uriPred + " litObj:" + litObj);
		}
		Resource subj = this.createTempRes(idSubj, ResourceTypes.ANON);
		Property pred = this.createTempProp(uriPred);
		Literal lit = this.createTempLit(litObj);
		return this.containsStatement(subj, pred, lit);
	}

	public boolean hasResStatOnAnonSubj(final String idSubj, final String uriPred, final String uriObj) {
		if (idSubj == null || uriPred == null || uriObj == null) {
			throw new RDFHelperException("Some args were null but " + "shouldn't. idSubj:" + idSubj + " uriPred:"
					+ uriPred + " uriObj:" + uriObj);
		}
		Resource subj = this.createTempRes(idSubj, ResourceTypes.ANON);
		Property pred = this.createTempProp(uriPred);
		Resource obj = this.createTempRes(uriObj, ResourceTypes.NAMED);
		return this.containsStatement(subj, pred, obj);
	}

	public boolean isAnonResStatReif(final String uriSubj, final String uriPred, final String idObj) {
		if (uriSubj == null || uriPred == null || idObj == null) {
			throw new RDFHelperException("Some args were null " + "but shouldn't. uriSubj:" + uriSubj + " uriPred:"
					+ uriPred + " idObj:" + idObj);
		}
		Resource subj = this.createTempRes(uriSubj, ResourceTypes.NAMED);
		Property pred = this.createTempProp(uriPred);
		Resource obj = this.createTempRes(idObj, ResourceTypes.ANON);
		return this.isReified(subj, pred, obj);
	}

	public boolean isAnonResStatReifOnAnonSubj(final String idSubj, final String uriPred, final String idObj) {
		if (idSubj == null || uriPred == null || idObj == null) {
			throw new RDFHelperException("Some args were null " + "but shouldn't. idSubj:" + idSubj + " uriPred:"
					+ uriPred + " idObj:" + idObj);
		}
		Resource subj = this.createTempRes(idSubj, ResourceTypes.ANON);
		Property pred = this.createTempProp(uriPred);
		Resource obj = this.createTempRes(idObj, ResourceTypes.ANON);
		return this.isReified(subj, pred, obj);
	}

	public boolean isDefinedAnonRes(final String id) {
		final Resource res = this.createTempRes(id, ResourceTypes.ANON);
		return this.containsResource(res);
	}

	public boolean isLitStatReif(final String uriSubj, final String uriPred, final String litObj) {
		if (uriSubj == null || uriPred == null || litObj == null) {
			throw new RDFHelperException("Some args were null " + "but shouldn't. uriSubj:" + uriSubj + " uriPred:"
					+ uriPred + " litObj:" + litObj);
		}
		Resource subj = this.createTempRes(uriSubj, ResourceTypes.NAMED);
		Property pred = this.createTempProp(uriPred);
		Literal obj = this.createTempLit(litObj);
		return this.isReified(subj, pred, obj);
	}

	public boolean isLitStatReifOnAnonSubj(final String idSubj, final String uriPred, final String litObj) {
		if (idSubj == null || uriPred == null || litObj == null) {
			throw new RDFHelperException("Some args were null " + "but shouldn't. idSubj:" + idSubj + " uriPred:"
					+ uriPred + " litObj:" + litObj);
		}
		Resource subj = this.createTempRes(idSubj, ResourceTypes.ANON);
		Property pred = this.createTempProp(uriPred);
		Literal obj = this.createTempLit(litObj);
		return this.isReified(subj, pred, obj);
	}

	public boolean isResStatReif(final String uriSubj, final String uriPred, final String uriObj) {
		if (uriSubj == null || uriPred == null || uriObj == null) {
			throw new RDFHelperException("Some args were null " + "but shouldn't. uriSubj:" + uriSubj + " uriPred:"
					+ uriPred + " uriObj:" + uriObj);
		}
		Resource subj = this.createTempRes(uriSubj, ResourceTypes.NAMED);
		Property pred = this.createTempProp(uriPred);
		Resource obj = this.createTempRes(uriObj, ResourceTypes.NAMED);
		return this.isReified(subj, pred, obj);
	}

	public boolean isResStatReifOnAnonSubj(final String idSubj, final String uriPred, final String uriObj) {
		if (idSubj == null || uriPred == null || uriObj == null) {
			throw new RDFHelperException("Some args were null " + "but shouldn't. idSubj:" + idSubj + " uriPred:"
					+ uriPred + " uriObj:" + uriObj);
		}
		Resource subj = this.createTempRes(idSubj, ResourceTypes.ANON);
		Property pred = this.createTempProp(uriPred);
		Resource obj = this.createTempRes(uriObj, ResourceTypes.NAMED);
		return this.isReified(subj, pred, obj);
	}

	public void removeAnonResStat(final String uriSubj, final String uriPred, final String idObj) {
		if (uriSubj == null || uriPred == null || idObj == null) {
			throw new RDFHelperException("Some args were null but shouldn't. uriSubj:" + uriSubj + " uriPred:"
					+ uriPred + " idObj:" + idObj);
		}
		Resource subj = this.createTempRes(uriSubj, ResourceTypes.NAMED);
		Property pred = this.createTempProp(uriPred);
		Resource obj = this.createTempRes(idObj, ResourceTypes.ANON);
		this.removeStatement(subj, pred, obj);
	}

	public void removeAnonResStatOnAnonSubj(final String idSubj, final String uriPred, final String idObj) {
		if (idSubj == null || uriPred == null || idObj == null) {
			throw new RDFHelperException("Some args were null but shouldn't. idSubj:" + idSubj + " uriPred:" + uriPred
					+ " idObj:" + idObj);
		}
		Resource subj = this.createTempRes(idSubj, ResourceTypes.ANON);
		Property pred = this.createTempProp(uriPred);
		Resource obj = this.createTempRes(idObj, ResourceTypes.ANON);
		this.removeStatement(subj, pred, obj);
	}

	public void removeLitStatOnAnonSubj(final String idSubj, final String uriPred, final String litObj) {
		if (idSubj == null || uriPred == null || litObj == null) {
			throw new RDFHelperException("Some args were null but shouldn't. idSubj:" + idSubj + " uriPred:" + uriPred
					+ " litObj:" + litObj);
		}
		Resource subj = this.createTempRes(idSubj, ResourceTypes.ANON);
		Property pred = this.createTempProp(uriPred);
		Literal obj = this.createTempLit(litObj);
		this.removeStatement(subj, pred, obj);
	}

	public void removeResStatOnAnonSubj(final String idSubj, final String uriPred, final String uriObj) {
		if (idSubj == null || uriPred == null || uriObj == null) {
			throw new RDFHelperException("Some args were null but shouldn't. idSubj:" + idSubj + " uriPred:" + uriPred
					+ " uriObj:" + uriObj);
		}
		Resource subj = this.createTempRes(idSubj, ResourceTypes.ANON);
		Property pred = this.createTempProp(uriPred);
		Resource obj = this.createTempRes(uriObj, ResourceTypes.NAMED);
		this.removeStatement(subj, pred, obj);
	}

	public void removeStatsOnAnonSubj(final String idSubj) {
		if (idSubj == null) {
			throw new RDFHelperException("idSubj was null but shouldn't.");
		}
		Resource subj = this.createTempRes(idSubj, ResourceTypes.ANON);
		this.removeStatement(subj, null, null);
	}

	public void removeStatWithAnonRes(final String idObj) {
		if (idObj == null) {
			throw new RDFHelperException("idObj was null but shouldn't.");
		}
		Resource obj = this.createTempRes(idObj, ResourceTypes.ANON);
		this.removeStatement(null, null, obj);
	}

	public List<String> getAnonReifStats() {
		return this.listReifiedStatements(ResourceTypes.ANON);
	}

	public String getAnonResOfAnonReifStat(final String statId) {
		if (statId == null) {
			throw new RDFHelperException("statId was null but shouldn't.");
		}
		final ReifiedStatement rs = this.getSpecifiedReifStat(statId, ResourceTypes.ANON);
		if (rs != null) {
			return this.getReifiedStatementObject(rs, RDFNodeTypes.ANON_R);
		}
		return null;
	}

	public String getAnonResOnReifStat(final String statUri) {
		if (statUri == null) {
			throw new RDFHelperException("statUri was null but shouldn't.");
		}
		final ReifiedStatement rs = this.getSpecifiedReifStat(statUri, ResourceTypes.NAMED);
		if (rs != null) {
			return this.getReifiedStatementObject(rs, RDFNodeTypes.ANON_R);
		}
		return null;
	}

	public String getAnonSubjOfAnonReifStat(final String statId) {
		if (statId == null) {
			throw new RDFHelperException("statId was null but shouldn't.");
		}
		final ReifiedStatement rs = this.getSpecifiedReifStat(statId, ResourceTypes.ANON);
		if (rs != null) {
			return this.getReifiedStatementSubject(rs, ResourceTypes.ANON);
		}
		return null;
	}

	public String getAnonSubjOfReifStat(final String statUri) {
		if (statUri == null) {
			throw new RDFHelperException("statUri was null but shouldn't.");
		}
		final ReifiedStatement rs = this.getSpecifiedReifStat(statUri, ResourceTypes.NAMED);
		if (rs != null) {
			return this.getReifiedStatementSubject(rs, ResourceTypes.ANON);
		}
		return null;
	}

	public String getLitOfAnonReifStat(final String statId) {
		if (statId == null) {
			throw new RDFHelperException("statId was null but shouldn't.");
		}
		final ReifiedStatement rs = this.getSpecifiedReifStat(statId, ResourceTypes.ANON);
		if (rs != null) {
			return this.getReifiedStatementObject(rs, RDFNodeTypes.LITERAL);
		}
		return null;
	}

	public String getLitOfReifStat(final String statUri) {
		if (statUri == null) {
			throw new RDFHelperException("statUri was null but shouldn't.");
		}
		final ReifiedStatement rs = this.getSpecifiedReifStat(statUri, ResourceTypes.NAMED);
		if (rs != null) {
			return this.getReifiedStatementObject(rs, RDFNodeTypes.LITERAL);
		}
		return null;
	}

	public String getPredOfAnonReifStat(final String statId) {
		if (statId == null) {
			throw new RDFHelperException("statId was null but shouldn't.");
		}
		final ReifiedStatement rs = this.getSpecifiedReifStat(statId, ResourceTypes.ANON);
		if (rs != null) {
			return this.getReifiedStatementPredicate(rs);
		}
		return null;
	}

	public String getPredOfReifStat(final String statUri) {
		if (statUri == null) {
			throw new RDFHelperException("statUri was null but shouldn't.");
		}
		final ReifiedStatement rs = this.getSpecifiedReifStat(statUri, ResourceTypes.NAMED);
		if (rs != null) {
			return this.getReifiedStatementPredicate(rs);
		}
		return null;
	}

	public List<String> getReifStats() {
		return this.listReifiedStatements(ResourceTypes.NAMED);
	}

	public String getResOfAnonReifStat(final String statId) {
		if (statId == null) {
			throw new RDFHelperException("statId was null but shouldn't.");
		}
		final ReifiedStatement rs = this.getSpecifiedReifStat(statId, ResourceTypes.ANON);
		if (rs != null) {
			return this.getReifiedStatementObject(rs, RDFNodeTypes.NAMED_R);
		}
		return null;
	}

	public String getResOfReifStat(final String statUri) {
		if (statUri == null) {
			throw new RDFHelperException("statUri was null but shouldn't.");
		}
		final ReifiedStatement rs = this.getSpecifiedReifStat(statUri, ResourceTypes.NAMED);
		if (rs != null) {
			return this.getReifiedStatementObject(rs, RDFNodeTypes.NAMED_R);
		}
		return null;
	}

	public String getSubjOfAnonReifStat(final String statId) {
		if (statId == null) {
			throw new RDFHelperException("statId was null but shouldn't.");
		}
		final ReifiedStatement rs = this.getSpecifiedReifStat(statId, ResourceTypes.ANON);
		if (rs != null) {
			return this.getReifiedStatementSubject(rs, ResourceTypes.NAMED);
		}
		return null;
	}

	public String getSubjOfReifStat(final String statUri) {
		if (statUri == null) {
			throw new RDFHelperException("statUri was null but shouldn't.");
		}
		final ReifiedStatement rs = this.getSpecifiedReifStat(statUri, ResourceTypes.NAMED);
		if (rs != null) {
			return this.getReifiedStatementSubject(rs, ResourceTypes.NAMED);
		}
		return null;
	}

	public String createAnonResStatReif(final String uriSubj, final String uriPred, final String idObj,
			final Map<String, String> mapPredLit, final Map<String, String> mapPredUri,
			final Map<String, String> mapPredId, final String statUri) {
		if (uriSubj == null || uriPred == null || idObj == null) {
			throw new RDFHelperException("Some args were null " + "but shouldn't. uriSubj:" + uriSubj + " uriPred:"
					+ uriPred + " idObj:" + idObj);
		}
		Resource subj = this.createTempRes(uriSubj, ResourceTypes.NAMED);
		Property pred = this.createTempProp(uriPred);
		Resource obj = this.createTempRes(idObj, ResourceTypes.ANON);
		Map<Property, RDFNode> map = new HashMap<Property, RDFNode>();
		map.putAll(this.convertReifMap(mapPredLit, RDFNodeTypes.LITERAL));
		map.putAll(this.convertReifMap(mapPredUri, RDFNodeTypes.NAMED_R));
		map.putAll(this.convertReifMap(mapPredId, RDFNodeTypes.ANON_R));
		final String toRet = this.createReifiedStatement(subj, pred, obj, map, statUri);
		return toRet;
	}

	public String createAnonResStatReifOnAnonSubj(final String idSubj, final String uriPred, final String idObj,
			final Map<String, String> mapPredLit, final Map<String, String> mapPredUri,
			final Map<String, String> mapPredId, final String statUri) {

		if (idSubj == null || uriPred == null || idObj == null) {
			throw new RDFHelperException("Some args were null " + "but shouldn't. idSubj:" + idSubj + " uriPred:"
					+ uriPred + " idObj:" + idObj);
		}
		Resource subj = this.createTempRes(idSubj, ResourceTypes.ANON);
		Property pred = this.createTempProp(uriPred);
		Resource obj = this.createTempRes(idObj, ResourceTypes.ANON);
		Map<Property, RDFNode> map = new HashMap<Property, RDFNode>();
		map.putAll(this.convertReifMap(mapPredLit, RDFNodeTypes.LITERAL));
		map.putAll(this.convertReifMap(mapPredUri, RDFNodeTypes.NAMED_R));
		map.putAll(this.convertReifMap(mapPredId, RDFNodeTypes.ANON_R));
		final String toRet = this.createReifiedStatement(subj, pred, obj, map, statUri);
		return toRet;
	}

	public String createLitStatReif(final String uriSubj, final String uriPred, final String litObj,
			final Map<String, String> mapPredLit, final Map<String, String> mapPredUri,
			final Map<String, String> mapPredId, final String statUri) {
		if (uriSubj == null || uriPred == null || litObj == null) {
			throw new RDFHelperException("Some args were null " + "but shouldn't. uriSubj:" + uriSubj + " uriPred:"
					+ uriPred + " litObj:" + litObj);
		}
		Resource subj = this.createTempRes(uriSubj, ResourceTypes.NAMED);
		Property pred = this.createTempProp(uriPred);
		Literal lit = this.createTempLit(litObj);
		Map<Property, RDFNode> map = new HashMap<Property, RDFNode>();
		map.putAll(this.convertReifMap(mapPredLit, RDFNodeTypes.LITERAL));
		map.putAll(this.convertReifMap(mapPredUri, RDFNodeTypes.NAMED_R));
		map.putAll(this.convertReifMap(mapPredId, RDFNodeTypes.ANON_R));
		final String toRet = this.createReifiedStatement(subj, pred, lit, map, statUri);
		return toRet;
	}

	public String createLitStatReifOnAnonSubj(final String idSubj, final String uriPred, final String litObj,
			final Map<String, String> mapPredLit, final Map<String, String> mapPredUri,
			final Map<String, String> mapPredId, final String statUri) {
		if (idSubj == null || uriPred == null || litObj == null) {
			throw new RDFHelperException("Some args were null " + "but shouldn't. idSubj:" + idSubj + " uriPred:"
					+ uriPred + " litObj:" + litObj);
		}
		Resource subj = this.createTempRes(idSubj, ResourceTypes.ANON);
		Property pred = this.createTempProp(uriPred);
		Literal lit = this.createTempLit(litObj);
		Map<Property, RDFNode> map = new HashMap<Property, RDFNode>();
		map.putAll(this.convertReifMap(mapPredLit, RDFNodeTypes.LITERAL));
		map.putAll(this.convertReifMap(mapPredUri, RDFNodeTypes.NAMED_R));
		map.putAll(this.convertReifMap(mapPredId, RDFNodeTypes.ANON_R));
		final String toRet = this.createReifiedStatement(subj, pred, lit, map, statUri);
		return toRet;

	}

	public String createResStatReif(final String uriSubj, final String uriPred, final String uriObj,
			final Map<String, String> mapPredLit, final Map<String, String> mapPredUri,
			final Map<String, String> mapPredId, final String statUri) {
		if (uriSubj == null || uriPred == null || uriObj == null) {
			throw new RDFHelperException("Some args were null " + "but shouldn't. uriSubj:" + uriSubj + " uriPred:"
					+ uriPred + " uriObj:" + uriObj);
		}
		Resource subj = this.createTempRes(uriSubj, ResourceTypes.NAMED);
		Property pred = this.createTempProp(uriPred);
		Resource obj = this.createTempRes(uriObj, ResourceTypes.NAMED);
		Map<Property, RDFNode> map = new HashMap<Property, RDFNode>();
		map.putAll(this.convertReifMap(mapPredLit, RDFNodeTypes.LITERAL));
		map.putAll(this.convertReifMap(mapPredUri, RDFNodeTypes.NAMED_R));
		map.putAll(this.convertReifMap(mapPredId, RDFNodeTypes.ANON_R));
		final String toRet = this.createReifiedStatement(subj, pred, obj, map, statUri);
		return toRet;
	}

	public String createResStatReifOnAnonSubj(final String idSubj, final String uriPred, final String uriObj,
			final Map<String, String> mapPredLit, final Map<String, String> mapPredUri,
			final Map<String, String> mapPredId, final String statUri) {
		if (idSubj == null || uriPred == null || uriObj == null) {
			throw new RDFHelperException("Some args were null " + "but shouldn't. idSubj:" + idSubj + " uriPred:"
					+ uriPred + " uriObj:" + uriObj);
		}
		Resource subj = this.createTempRes(idSubj, ResourceTypes.ANON);
		Property pred = this.createTempProp(uriPred);
		Resource obj = this.createTempRes(uriObj, ResourceTypes.NAMED);
		Map<Property, RDFNode> map = new HashMap<Property, RDFNode>();
		map.putAll(this.convertReifMap(mapPredLit, RDFNodeTypes.LITERAL));
		map.putAll(this.convertReifMap(mapPredUri, RDFNodeTypes.NAMED_R));
		map.putAll(this.convertReifMap(mapPredId, RDFNodeTypes.ANON_R));
		final String toRet = this.createReifiedStatement(subj, pred, obj, map, statUri);
		return toRet;
	}

	public void removeAnonReifStat(final String statId) {
		ReifiedStatement rs = this.getSpecifiedReifStat(statId, ResourceTypes.ANON);
		if (rs != null) {
			this.removeReification(rs);
		}
	}

	public void removeReifStat(final String statUri) {
		ReifiedStatement rs = this.getSpecifiedReifStat(statUri, ResourceTypes.NAMED);
		if (rs != null) {
			this.removeReification(rs);
		}
	}

	public List<String> getAnonResOnPredAnonStatReif(final String statId, final String uriPred) {
		if (statId == null || uriPred == null) {
			throw new RDFHelperException("Some args were null " + "but shouldn't. statId:" + statId + " uriPred:"
					+ uriPred);
		}
		Property pred = this.createTempProp(uriPred);
		ReifiedStatement rs = this.getSpecifiedReifStat(statId, ResourceTypes.ANON);
		if (rs == null) {
			return new ArrayList<String>();
		}
		return this.getObjects(rs, pred, RDFNodeTypes.ANON_R);
	}

	public List<String> getAnonResOnPredStatReif(final String statUri, final String uriPred) {
		if (statUri == null || uriPred == null) {
			throw new RDFHelperException("Some args were null " + "but shouldn't. statUri:" + statUri + " uriPred:"
					+ uriPred);
		}
		Property pred = this.createTempProp(uriPred);
		ReifiedStatement rs = this.getSpecifiedReifStat(statUri, ResourceTypes.NAMED);
		if (rs == null) {
			return new ArrayList<String>();
		}
		return this.getObjects(rs, pred, RDFNodeTypes.ANON_R);
	}

	public List<String> getLitOnPredAnonStatReif(final String statId, final String uriPred) {
		if (statId == null || uriPred == null) {
			throw new RDFHelperException("Some args were null " + "but shouldn't. statId:" + statId + " uriPred:"
					+ uriPred);
		}
		Property pred = this.createTempProp(uriPred);
		ReifiedStatement rs = this.getSpecifiedReifStat(statId, ResourceTypes.ANON);
		if (rs == null) {
			return new ArrayList<String>();
		}
		return this.getObjects(rs, pred, RDFNodeTypes.LITERAL);
	}

	public List<String> getLitsOnPredStatReif(final String statUri, final String uriPred) {
		if (statUri == null || uriPred == null) {
			throw new RDFHelperException("Some args were null " + "but shouldn't. statUri:" + statUri + " uriPred:"
					+ uriPred);
		}
		Property pred = this.createTempProp(uriPred);
		ReifiedStatement rs = this.getSpecifiedReifStat(statUri, ResourceTypes.NAMED);
		if (rs == null) {
			return new ArrayList<String>();
		}
		return this.getObjects(rs, pred, RDFNodeTypes.LITERAL);
	}

	public Set<String> getPredOnAnonStatReif(final String statId) {
		if (statId == null) {
			throw new RDFHelperException("statId was null");
		}
		ReifiedStatement rs = this.getSpecifiedReifStat(statId, ResourceTypes.ANON);
		if (rs == null) {
			return new HashSet<String>();
		}
		return this.getPredicates(rs);
	}

	public Set<String> getPredOnStatReif(final String statUri) {
		if (statUri == null) {
			throw new RDFHelperException("statUri was null");
		}
		ReifiedStatement rs = this.getSpecifiedReifStat(statUri, ResourceTypes.NAMED);
		if (rs == null) {
			return new HashSet<String>();
		}
		return this.getPredicates(rs);
	}

	public List<String> getResOnPredAnonStatReif(final String statId, final String uriPred) {
		if (statId == null || uriPred == null) {
			throw new RDFHelperException("Some args were null " + "but shouldn't. statId:" + statId + " uriPred:"
					+ uriPred);
		}
		Property pred = this.createTempProp(uriPred);
		ReifiedStatement rs = this.getSpecifiedReifStat(statId, ResourceTypes.ANON);
		List<String> toRet = new ArrayList<String>();
		if (rs == null) {
			return toRet;
		}
		return this.getObjects(rs, pred, RDFNodeTypes.NAMED_R);
	}

	public List<String> getResOnPredStatReif(final String statUri, final String uriPred) {
		if (statUri == null || uriPred == null) {
			throw new RDFHelperException("Some args were null " + "but shouldn't. statUri:" + statUri + " uriPred:"
					+ uriPred);
		}
		Property pred = this.createTempProp(uriPred);
		ReifiedStatement rs = this.getSpecifiedReifStat(statUri, ResourceTypes.NAMED);
		List<String> toRet = new ArrayList<String>();
		if (rs == null) {
			return toRet;
		}
		return this.getObjects(rs, pred, RDFNodeTypes.NAMED_R);
	}

	public static enum ResourceTypes {
		NAMED, ANON
	}

	public static enum RDFNodeTypes {
		NAMED_R, ANON_R, LITERAL
	}

	private Model model;

	private Model tempModel;

	/**
	 * Loads a file into a Jena Model.
	 * 
	 * @param file
	 *            The file to load as a Jena Model.
	 * @param baseUri
	 *            the base URI to be used when converting relative URI's to absolute URI's. The base URI may be <code>null</code> if there are no relative URIs
	 *            to convert. A base URI of <code>""</code> may permit relative URIs to be used in the model. When handling many models and trying to add them
	 *            into a single model, it's recommended not to use <code>""</code> to prevent from relative URIs.
	 * @param lang
	 *            the language of the serialisation. <code>null</code> selects the default in the source of the input serialisation (RDF/XML). Predefined values
	 *            for <code>lang</code> are "RDF/XML", "N-TRIPLE", "TURTLE" (or "TTL") and "N3". <code>null</code> represents the default language, "RDF/XML".
	 *            "RDF/XML-ABBREV" is a synonym for "RDF/XML".
	 * @return The Jena model of the file.
	 * @throws IOException
	 *             If the file cannot be accessed
	 */
	public static Model modelFromFile(final File file, final String baseUri, final String lang) throws IOException {
		Model aModel = ModelFactory.createDefaultModel();
		FileInputStream fis;
		try {
			fis = new FileInputStream(file);
		} catch (final FileNotFoundException fnfe) {
			LogFactory.getLog(BackEndJenaHelper.class).warn("File " + file.getAbsolutePath() + " not found.", fnfe);
			throw fnfe;
		}
		try {
			aModel.read(fis, baseUri, lang);
		} finally {
			try {
				fis.close();
			} catch (final IOException ioe) {
				LogFactory.getLog(BackEndJenaHelper.class).warn(
						"Error wile closing stream on file: " + file.getAbsolutePath() + ".", ioe);
			}
		}
		return aModel;
	}

	/**
	 * 
	 */
	public BackEndJenaHelper() {
		this.model = ModelFactory.createDefaultModel();
		this.tempModel = ModelFactory.createDefaultModel();
	}

	/**
	 * @param model
	 *            The model to add
	 */
	public BackEndJenaHelper(Model model) {
		this.addToModel(model);
	}

	/**
	 * @param rdfXml
	 *            An RDF/XML String to be added to the model
	 */
	public BackEndJenaHelper(final String rdfXml) {
		this.addToModel(rdfXml);
	}

	/**
	 * @param modelToAdd
	 *            The <tt>Jena</tt> <code>Model</code> to add to the <code>model</code>
	 */
	public void addToModel(final Model modelToAdd) {
		if (this.model == null) {
			this.model = ModelFactory.createDefaultModel();
		}
		if (this.tempModel == null) {
			this.tempModel = ModelFactory.createDefaultModel();
		}
		this.model.add(modelToAdd);
	}

	/**
	 * @param rdfXml
	 *            An RDF/XML String to be added to the model
	 * @throws RDFHelperException
	 *             If <code>model</code> is <code>null</code> or empty.
	 */
	public void addToModel(final String rdfXml) {
		if (this.model == null) {
			this.model = ModelFactory.createDefaultModel();
		}
		if (this.tempModel == null) {
			this.tempModel = ModelFactory.createDefaultModel();
		}
		if (rdfXml != null && !rdfXml.equals("")) {
			StringReader sr = new StringReader(rdfXml);
			this.model.read(sr, BASE, SYNTAX);
			sr.close();
		} else {
			throw new RDFHelperException("Args was null or empty.");
		}
	}

	/**
	 * @return the model
	 */
	public Model getModel() {
		return this.model;
	}

	/**
	 * @param uriOrId
	 *            URI or internal ID of the resource to be created in <code>tempModel</code>.
	 * @param resType
	 *            The type of the resource to be created, i.e. named or anonymous one. In case of named, <code>uriOrId</code> will be used as an URI, in case of
	 *            anonymous, <code>uriOrId</code> will be used as the internal ID for anonymous resources.
	 * @return A Jena temporary <code>Resource</code>
	 */
	protected Resource createTempRes(final String uriOrId, final ResourceTypes resType) {
		if (resType.equals(ResourceTypes.NAMED)) {
			return this.tempModel.createResource(uriOrId);
		} else if (resType.equals(ResourceTypes.ANON)) {
			final AnonId a = AnonId.create(uriOrId);
			return this.tempModel.createResource(a);
		} else {
			throw new RDFHelperException("Unexpected error. Using a type that is not defined: " + resType.name() + "-"
					+ resType.ordinal());
		}
	}

	/**
	 * @param value
	 *            The value of the literal to be created in <code>tempModel</code>.
	 * @return A Jena temporary <code>Literal</code>
	 */
	protected Literal createTempLit(final String value) {
		return createTempLit(value, null);
	}
	
	/**
	 * @param value
	 *            The value of the literal to be created in <code>tempModel</code>.
	 * @param language 
	 * 			  The language of the typed value
	 * @return A Jena temporary <code>Literal</code>
	 */
	protected Literal createTempLit(final String value, final String language) {
		return this.tempModel.createLiteral(value,language);
	}

	/**
	 * @param uri
	 *            The uri of the property to create
	 * @return A Jena temporary <code>Property</code>
	 */
	protected Property createTempProp(final String uri) {
		return this.tempModel.createProperty(uri);
	}

	/**
	 * @param node
	 *            The <code>RDFNode</code> to test presence in <code>model</code>. It should have been created into <code>tempMode</code>.
	 * @return Whether or not <code>node</code> is defined in <code>model</code> .
	 */
	protected boolean containsResource(final RDFNode node) {
		return this.model.containsResource(node);
	}

	/**
	 * @param pred
	 *            The predicate.
	 * @param obj
	 *            The object. If <code>obj</code> is <code>null</code> only the predicate is used to search for subjects.
	 * @param resType
	 *            The type of the resources to retrieve, i.e. named or anonymous one.
	 * @return A <code>Set</code> of URIs or internal IDs resources that are subjects of statements having <code>pred</code> as predicate and <code>obj</code>
	 *         as object.
	 */
	protected Set<String> getSubjects(final Property pred, final RDFNode obj, final ResourceTypes resType) {
		List<Statement> stats = this.listStatements();
		if (obj != null) {
			ListIterator<Statement> it = stats.listIterator();
			while (it.hasNext()) {
				Statement stat = it.next();
				if (!stat.getObject().equals(obj)) {
					it.remove();
				}
			}
		}
		HashSet<String> subjs = new HashSet<String>();
		for (final Statement stat : stats) {
			if (pred.equals(stat.getPredicate())) {
				final Resource subj = stat.getSubject();
				if (resType.equals(ResourceTypes.NAMED) && subj.isURIResource()) {
					subjs.add(subj.getURI());
				} else if (resType.equals(ResourceTypes.ANON) && subj.isAnon()) {
					subjs.add(subj.getId().getLabelString());
				}
			}
		}
		return subjs;
	}

	/**
	 * @param subj
	 *            The subject. It can be either an anonymous resource, named resource or <code>null</code>. In case of <code>null</code>, only properties will
	 *            be used to search for objects.
	 * @param pred
	 *            The predicate.
	 * @param nodeType
	 *            The type of the objects to retrieve.
	 * @return A <code>List</code> of URIs, internal IDs or literal values that are object of statements having <code>pred</code> as predicate and
	 *         <code>subj</code> as subject.
	 */
	protected List<String> getObjects(final Resource subj, final Property pred, final RDFNodeTypes nodeType) {
		return getObjects(subj, pred, nodeType, null);
	}
	
	/**
	 * @param subj
	 *            The subject. It can be either an anonymous resource, named resource or <code>null</code>. In case of <code>null</code>, only properties will
	 *            be used to search for objects.
	 * @param pred
	 *            The predicate.
	 * @param nodeType
	 *            The type of the objects to retrieve.
	 * @return A <code>List</code> of URIs, internal IDs or literal values that are object of statements having <code>pred</code> as predicate and
	 *         <code>subj</code> as subject.
	 */
	protected List<String> getObjects(final Resource subj, final Property pred, final RDFNodeTypes nodeType, final String language) {
		List<Statement> stats = this.listStatements();
		if (subj != null) {
			ListIterator<Statement> it = stats.listIterator();
			while (it.hasNext()) {
				Statement stat = it.next();
				if (!stat.getSubject().equals(subj)) {
					it.remove();
				}
			}
		}
		List<String> objs = new ArrayList<String>();
		for (final Statement stat : stats) {
			if (pred.equals(stat.getPredicate())) {
				RDFNode obj = stat.getObject();
				if (nodeType.equals(RDFNodeTypes.ANON_R) && obj.isAnon()) {
					objs.add(((Resource) obj).getId().getLabelString());
				} else if (nodeType.equals(RDFNodeTypes.LITERAL) && obj.isLiteral()) {
					if (language!=null && ((Literal) obj).getLanguage().equals(language)) {
						objs.add(((Literal) obj).getLexicalForm());
					} else if (language == null){
						objs.add(((Literal) obj).getLexicalForm());
					}
				} else if (nodeType.equals(RDFNodeTypes.NAMED_R) && obj.isURIResource()) {
					objs.add(((Resource) obj).getURI());
				}
			}
		}
		return objs;
	}

	/**
	 * @param subj
	 *            The subject. It can be either an anonymous resource or named resource.
	 * @return A <code>Set</code> URIs of the predicates contained by the model an refining <code>subj</code> if not <code>null</code>.
	 */
	protected Set<String> getPredicates(final Resource subj) {
		List<Statement> stats = this.listStatements();
		HashSet<String> preds = new HashSet<String>();
		for (final Statement stat : stats) {
			if (subj.equals(stat.getSubject())) {
				preds.add(stat.getPredicate().getURI());
			}
		}
		return preds;
	}

	/**
	 * @param resType
	 *            The type of the resources to retrieve, i.e. named or anonymous one.
	 * @return A <code>Set</code> of URIs or internal IDs resources that are subjects of statements in this model.
	 */
	protected Set<String> getSubjects(final ResourceTypes resType) {
		List<Statement> stats = this.listStatements();
		HashSet<String> subjs = new HashSet<String>();
		for (Statement stat : stats) {
			Resource subj = stat.getSubject();
			if (resType.equals(ResourceTypes.NAMED) && subj.isURIResource()) {
				subjs.add(subj.getURI());
			} else if (resType.equals(ResourceTypes.ANON) && subj.isAnon()) {
				subjs.add(subj.getId().getLabelString());
			}
		}
		return subjs;
	}

	/**
	 * @return A list containing every statements including the one that are subject of reifications but not the reification implementation ones (subject,
	 *         object...).
	 */
	protected List<Statement> listStatements() {
		List<Statement> stats = new ArrayList<Statement>();
		StmtIterator sit = this.model.listStatements();
		while (sit.hasNext()) {
			Statement stat = sit.nextStatement();
			// Filtering reified statement
			
			
			if (!stat.getPredicate().equals(RDF.subject) && !stat.getPredicate().equals(RDF.predicate)
					&& !stat.getPredicate().equals(RDF.object)) {
				if (stat.getPredicate().equals(RDF.type) && stat.getObject().isResource()) {
					if (!((Resource) stat.getObject()).equals(RDF.Statement)) {
						stats.add(stat);
					}
				} else {
					stats.add(stat);
				}
			}
		}
		sit.close();
		RSIterator rsit = this.model.listReifiedStatements();
		while (rsit.hasNext()) {
			stats.add(rsit.nextRS().getStatement());
		}
		rsit.close();
		return stats;
	}

	/**
	 * @param subj
	 *            The subject of the statement to create; it might be either or named or not.
	 * @param pred
	 *            The predicate of the statement to create.
	 * @param obj
	 *            The object of the statement to create. It might either be a literal or a resource.
	 * @param add
	 *            Whether of not to add the created statement into the model.
	 */
	protected Statement createStatement(final Resource subj, final Property pred, final RDFNode obj, final boolean add) {
		if (!this.containsStatement(subj, pred, obj) && add) {
			this.model.add(this.model.createStatement(subj, pred, obj));
		}
		return this.model.createStatement(subj, pred, obj);
	}

	/**
	 * @param subj
	 *            The subject of the statement to create; it might be either or named or not.
	 * @param pred
	 *            The predicate of the statement to create.
	 * @param obj
	 *            The object of the statement to create. It might either be a literal or a resource.
	 */
	protected Statement createStatement(final Resource subj, final Property pred, final RDFNode obj) {
		if (!this.containsStatement(subj, pred, obj)) {
			this.model.add(this.model.createStatement(subj, pred, obj));
		}
		return this.model.createStatement(subj, pred, obj);
	}

	/**
	 * @param subj
	 *            The subject of the statement to be reified.
	 * @param pred
	 *            The predicate of the statement to be reified.
	 * @param obj
	 *            The object of the statement to be reified.
	 * @param reifs
	 *            Properties and values to be affected to the reified statement.
	 * @param uri
	 *            The uri (might be null) of the statement to create.
	 */
	protected String createReifiedStatement(final Resource subj, final Property pred, RDFNode obj,
			Map<Property, RDFNode> reifs, final String uri) {
		ReifiedStatement rst;
		if (uri == null) {
			rst = this.model.createReifiedStatement(this.createStatement(subj, pred, obj, false));
		} else {
			rst = this.model.createReifiedStatement(uri, this.createStatement(subj, pred, obj, false));
		}

		for (final Entry<Property, RDFNode> entry : reifs.entrySet()) {
			rst.addProperty(entry.getKey(), entry.getValue());
		}
		if (uri == null) {
			return rst.getId().getLabelString();
		}
		return rst.getURI();
	}

	/**
	 * Remove any statement matching <tt>(S,P,O)</tt> from the model. <code>null</code> values are working as wildcard.
	 * 
	 * @param subj
	 *            The subject of the statement to create; it might be either or named or not.
	 * @param pred
	 *            The predicate of the statement to create.
	 * @param obj
	 *            The object of the statement to create. It might either be a literal or a resource.
	 */
	protected void removeStatement(final Resource subj, final Property pred, final RDFNode obj) {
		if (subj == null || pred == null || obj == null) {
			StmtIterator sit = this.model.listStatements(subj, pred, obj);
			while (sit.hasNext()) {
				Statement s = sit.nextStatement();
				this.model.removeAllReifications(s);
			}
			sit.close();
			this.model.removeAll(subj, pred, obj);
		} else {
			Statement s = this.buildTempStatement(subj, pred, obj);
			this.model.removeAllReifications(s);
			this.model.remove(s);
		}
		for (Statement s : this.listStatements()) {
			if ((subj == null || s.getSubject().equals(subj)) && (pred == null || s.getPredicate().equals(pred))
					&& (obj == null || s.getObject().equals(obj))) {
				if (s.isReified()) {
					RSIterator rsit = this.model.listReifiedStatements(s);
					while (rsit.hasNext()) {
						ReifiedStatement rs = rsit.nextRS();
						this.model.removeAll(rs, null, null);
						this.model.removeAll(null, null, rs);
					}
					rsit.close();
				}
				this.model.removeAllReifications(s);
				this.model.remove(s);
			}
		}
	}

	/**
	 * Test whether any statement matching <tt>(S,P,O)</tt> is present in the model. <code>null</code> values are working as wildcard.
	 * 
	 * @param subj
	 *            The subject of the statement to create; it might be either or named or not.
	 * @param pred
	 *            The predicate of the statement to create.
	 * @param obj
	 *            The object of the statement to create. It might either be a literal or a resource.
	 */
	protected boolean containsStatement(Resource subj, Property pred, RDFNode obj) {
		if (!this.model.contains(subj, pred, obj)) {
			if (subj != null && pred != null && obj != null) {
				Statement s = this.buildTempStatement(subj, pred, obj);
				return this.model.isReified(s);
			}
			return false;
		}
		return true;
	}

	/**
	 * @return The RDF/XML String view of the model.
	 */
	public String getRdfXml() {
		final List<String> prefixToRemove = new LinkedList<String>();
		for (final Object prefix : this.model.getNsPrefixMap().keySet()) {
			if (prefix.toString().startsWith("j.")) {
				prefixToRemove.add(prefix.toString());
			}
		}
		for (final String prefix : prefixToRemove) {
			this.model.removeNsPrefix(prefix);
		}

		StringWriter writer = new StringWriter();
		try {
			this.model.write(writer, SYNTAX);
		} catch (final Exception e) {
			throw new RDFHelperException("Serialisation of Jena model " + "failed. " + SYNTAX
					+ "string appears to be invalid: " + writer.toString(), e);
		}
		return writer.toString().replace("\r", "");
	}

	/**
	 * @param reifs
	 *            A map of predicate URIs and objects (literal values, internal ID or URI of resources).
	 * @param objType
	 *            The type of the second string in the map (literal values, internal ID or URI of resources).
	 * @return The map to be used for reification.
	 */
	protected Map<Property, RDFNode> convertReifMap(final Map<String, String> reifs, final RDFNodeTypes objType) {
		if (reifs == null) {
			return new HashMap<Property, RDFNode>();
		}
		Map<Property, RDFNode> result = new HashMap<Property, RDFNode>(reifs.size());
		if (objType.equals(RDFNodeTypes.LITERAL)) {
			for (Entry<String, String> entry : reifs.entrySet()) {
				result.put(this.createTempProp(entry.getKey()), this.createTempLit(entry.getValue()));
			}
		} else if (objType.equals(RDFNodeTypes.NAMED_R)) {
			for (Entry<String, String> entry : reifs.entrySet()) {
				result.put(this.createTempProp(entry.getKey()), this.createTempRes(entry.getValue(),
						ResourceTypes.NAMED));
			}
		} else if (objType.equals(RDFNodeTypes.ANON_R)) {
			for (Entry<String, String> entry : reifs.entrySet()) {
				result.put(this.createTempProp(entry.getKey()), this
						.createTempRes(entry.getValue(), ResourceTypes.ANON));
			}
		}
		return result;
	}

	/**
	 * @param subj
	 *            The subject of the statement that might be reified.
	 * @param pred
	 *            The predicate of the statement that might be reified.
	 * @param obj
	 *            The object of the statement that might be reified.
	 * @return A list that might be empty of reified statement representing the statement (<code>subj</code>, <code>pred</code>, <code>obj</code> ).
	 */
	protected List<ReifiedStatement> getReifStats(final Resource subj, final Property pred, final RDFNode obj) {
		List<ReifiedStatement> reifs = new ArrayList<ReifiedStatement>();
		if (this.containsStatement(subj, pred, obj)) {
			Statement s = this.createStatement(subj, pred, obj, false);
			if (s.isReified()) {
				RSIterator rsit = s.listReifiedStatements();
				while (rsit.hasNext()) {
					reifs.add(rsit.nextRS());
				}
				rsit.close();
			}
		}
		return reifs;
	}

	/**
	 * @param subj
	 *            The subject of the statement to test reification.
	 * @param pred
	 *            The predicate of the statement to test reification.
	 * @param obj
	 *            The object of the statement to test reification.
	 * @return A temporary statement
	 */
	protected boolean isReified(final Resource subj, final Property pred, final RDFNode obj) {
		final Statement s = this.buildTempStatement(subj, pred, obj);
		return this.model.isReified(s);
	}

	/**
	 * @param subj
	 *            The subject of the statement to be build.
	 * @param pred
	 *            The predicate of the statement to be build.
	 * @param obj
	 *            The object of the statement to be build.
	 * @return A temporary statement
	 */
	private Statement buildTempStatement(final Resource subj, final Property pred, final RDFNode obj) {
		return this.tempModel.createStatement(subj, pred, obj);
	}

	/**
	 * To prevent from unchecked cast warning, method that convert from a non generic <code>Map</code> to a <code>String, String</code> one.
	 * 
	 * @param prefixMap
	 *            The <code>Map</code> of prefixes/uris used in <tt>Jena</tt>.
	 * @return The same <code>Map</code> but parameterized.
	 */
	private static Map<String, String> convertNSMap(Map<?, ?> prefixMap) {
		Map<String, String> ret = new HashMap<String, String>(prefixMap.size());
		for (final Entry<?, ?> entry : prefixMap.entrySet()) {
			if (entry.getKey() instanceof String && entry.getValue() instanceof String) {
				ret.put((String) entry.getKey(), (String) entry.getValue());
			}
		}
		return ret;
	}

	/**
	 * @param prefix
	 *            A prefix to be used in RDF/XML.
	 * @param uri
	 *            The uri associated with this prefix.
	 * @return The full map of prefixes after the addition of <code>prefix</code> and <code>uri</code>.
	 */
	public Map<String, String> setNSPrefix(final String prefix, final String uri) {
		if ((this.model != null) && (prefix != null) && (uri != null) && (!prefix.equals("")) && (!uri.equals(""))) {
			Map<?, ?> map = this.model.setNsPrefix(prefix, uri).getNsPrefixMap();
			return BackEndJenaHelper.convertNSMap(map);
		}
		return this.getNSPrefixes();

	}

	/**
	 * @return The full map of prefixes and uris to be used by the model.
	 */
	public Map<String, String> getNSPrefixes() {
		if (this.model != null) {
			return BackEndJenaHelper.convertNSMap(this.model.getNsPrefixMap());
		}
		return new HashMap<String, String>(0);
	}

	/**
	 * @param type
	 *            Whether statements to retrieve are anonymous or named.
	 * @return A list of internal ID or URI (depending on <code>type</code>) representing every reified statements in the model
	 */
	public List<String> listReifiedStatements(final ResourceTypes type) {
		List<String> idsOrUris = new ArrayList<String>();
		if (type.equals(ResourceTypes.ANON)) {
			for (final ReifiedStatement rs : this.listReifStats(type)) {
				idsOrUris.add(rs.getId().getLabelString());
			}
		} else if (type.equals(ResourceTypes.NAMED)) {
			for (ReifiedStatement rs : this.listReifStats(type)) {
				idsOrUris.add(rs.getURI());
			}
		} else {
			LogFactory.getLog(this.getClass()).warn(
					"Error in listReifiedStatement " + "ResourceType was neither an " + "anon nor a named resource...");
		}
		return idsOrUris;
	}

	/**
	 * @param type
	 *            The type of reified statements to retrieve, i.e. anonymous or named ones.
	 * @return Every reified statements of the model, having a type as <code>type</code>.
	 */
	private List<ReifiedStatement> listReifStats(final ResourceTypes type) {
		RSIterator rsi = this.model.listReifiedStatements();
		List<ReifiedStatement> rss = new ArrayList<ReifiedStatement>();
		if (type.equals(ResourceTypes.ANON)) {
			while (rsi.hasNext()) {
				ReifiedStatement rs = rsi.nextRS();
				if (rs.isAnon()) {
					rss.add(rs);
				}
			}
		} else if (type.equals(ResourceTypes.NAMED)) {
			while (rsi.hasNext()) {
				ReifiedStatement rs = rsi.nextRS();
				if (rs.isURIResource()) {
					rss.add(rs);
				}
			}
		} else {
			LogFactory.getLog(this.getClass()).warn(
					"Error in listReifiedStatement " + "ResourceType was neither an " + "anon nor a named resource...");
		}
		rsi.close();
		return rss;
	}

	/**
	 * @param idOrUri
	 *            The internal ID or uri of the reified statement to retrieve
	 * @param type
	 *            The type of reified statements to retrieve, i.e. anonymous or named.
	 * @return The statement or <code>null</code>.
	 */
	protected ReifiedStatement getSpecifiedReifStat(final String idOrUri, final ResourceTypes type) {
		ReifiedStatement toRet = null;
		if (type.equals(ResourceTypes.ANON)) {
			for (final ReifiedStatement rs : this.listReifStats(type)) {
				if (rs.getId().getLabelString().equals(idOrUri)) {
					toRet = rs;
					break;
				}
			}
		} else {
			for (final ReifiedStatement rs : this.listReifStats(type)) {
				if (rs.getURI().equals(idOrUri)) {
					toRet = rs;
					break;
				}
			}
		}
		return toRet;
	}

	/**
	 * @param rs
	 *            The reified statement to look for subject.
	 * @param type
	 *            The type of the subject.
	 * @return The subject (uri or id) or null if an error occurs
	 */
	protected String getReifiedStatementSubject(final ReifiedStatement rs, final ResourceTypes type) {
		String toRet = null;
		final Resource subj = rs.getStatement().getSubject();
		if (type.equals(ResourceTypes.ANON) && subj.isAnon()) {
			toRet = subj.getId().getLabelString();
		} else if (type.equals(ResourceTypes.NAMED) && subj.isURIResource()) {
			toRet = subj.getURI();
		}
		return toRet;
	}

	/**
	 * @param rs
	 *            The reified statement to look for object.
	 * @param type
	 *            The type of the object.
	 * @return The object (literal value, id or uri) or null if an error occurs
	 */
	protected String getReifiedStatementObject(final ReifiedStatement rs, final RDFNodeTypes type) {
		String toRet = null;
		final RDFNode obj = rs.getStatement().getObject();
		if (type.equals(RDFNodeTypes.ANON_R) && obj.isAnon()) {
			toRet = ((Resource) obj).getId().getLabelString();
		} else if (type.equals(RDFNodeTypes.LITERAL) && obj.isLiteral()) {
			toRet = ((Literal) obj).getLexicalForm();
		} else if (type.equals(RDFNodeTypes.NAMED_R) && obj.isURIResource()) {
			toRet = ((Resource) obj).getURI();
		}
		return toRet;
	}

	/**
	 * @param rs
	 *            The reified statement to look for predicate.
	 * @return The predicate uri or null if an error occurs
	 */
	protected String getReifiedStatementPredicate(final ReifiedStatement rs) {
		return rs.getStatement().getPredicate().getURI();
	}

	/**
	 * @param rs
	 *            The reified statement to look remove.
	 */
	protected void removeReification(ReifiedStatement rs) {
		this.removeStatement(rs, null, null);
		this.model.removeReification(rs);
	}

	/**
	 * @return A <code>Set</code> containing URIs of properties that are predicate of any statement in the <code>Annotation</code>.
	 */
	public Set<String> getPreds() {
		Set<String> preds = new HashSet<String>();
		for (final Statement stat : this.listStatements()) {
			preds.add(stat.getPredicate().getURI());
		}
		return preds;
	}

	/**
	 * Closes the model and the temporary model. Recreates a temporary model just in case.
	 */
	public void close() {
		this.model.close();
		this.tempModel.close();
		this.tempModel = ModelFactory.createDefaultModel();
	}

	/*
	 * (non-Javadoc)
	 * @see java.lang.Object#finalize()
	 */
	@Override
	protected void finalize() throws Throwable {
		super.finalize();
		this.tempModel.close();
		this.model.close();
	}
}
