package org.ow2.weblab.iterator;

import java.io.File;
import java.io.IOException;
import java.util.LinkedList;

import javax.annotation.PostConstruct;
import javax.jws.WebService;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.ow2.weblab.core.extended.exception.WebLabCheckedException;
import org.ow2.weblab.core.extended.jaxb.WebLabMarshaller;
import org.ow2.weblab.core.extended.properties.PropertiesLoader;
import org.ow2.weblab.core.model.Document;
import org.ow2.weblab.core.model.Resource;
import org.ow2.weblab.core.services.AccessDeniedException;
import org.ow2.weblab.core.services.ContentNotAvailableException;
import org.ow2.weblab.core.services.EmptyQueueException;
import org.ow2.weblab.core.services.InsufficientResourcesException;
import org.ow2.weblab.core.services.InvalidParameterException;
import org.ow2.weblab.core.services.QueueManager;
import org.ow2.weblab.core.services.ServiceNotConfiguredException;
import org.ow2.weblab.core.services.UnexpectedException;
import org.ow2.weblab.core.services.UnsupportedRequestException;
import org.ow2.weblab.core.services.queuemanager.NextResourceArgs;
import org.ow2.weblab.core.services.queuemanager.NextResourceReturn;

/**
 * This service is able to list resources from a folder. 
 * 
 * /!\ Only one folder is accessible at a time on a JVM (static variables to ensure thread safe execution)
 * 
 * @author CASSIDIAN WebLab team
 * @date
 */
@WebService(endpointInterface = "org.ow2.weblab.core.services.QueueManager")
public class FolderResourceIterator implements QueueManager {
	public static Log LOGGER = LogFactory.getLog(FolderResourceIterator.class);

	protected final static String PROPERTY_FILE = "resourceiterator.properties";
	protected final static String PROPERTY_NAME = "folderpath";

	/**
	 * folder containing WL resources
	 */
	protected WebLabMarshaller webLabMarshaller = new WebLabMarshaller();

	protected static File folder;
	protected static String folderPath;
	protected static LinkedList<String> resourceList = new LinkedList<String>();

	protected int loadResources() throws IOException {
		String[] filePaths = folder.list();
		if (filePaths == null) {
			throw new IOException("Folder " + folder.getAbsolutePath() + " is invalid.");
		}
		for (String filePath : filePaths) {
			FolderResourceIterator.resourceList.add(filePath);
		}
		if (FolderResourceIterator.resourceList.size() > 0) {
			LOGGER.info(FolderResourceIterator.resourceList.size() + " items added.");
		}else{
			LOGGER.warn("Reach the end of resources iterator list.");
		}
		return FolderResourceIterator.resourceList.size();
	}

	@PostConstruct
	public void init() throws IOException {
		if (FolderResourceIterator.folder == null) {
			/*
			 * try to load the property file to get the resources folder
			 */
			String folderPath = PropertiesLoader.loadProperties(PROPERTY_FILE).get(PROPERTY_NAME);

			/*
			 * unable to get the property key
			 */
			if (folderPath == null) {
				String errorMsg = "Unable to read property '" + PROPERTY_NAME + "' in property file: " + PROPERTY_FILE;
				LOGGER.error(errorMsg);
				throw new IOException(errorMsg);
			}

			FolderResourceIterator.folder = new File(folderPath);
			FolderResourceIterator.folderPath = FolderResourceIterator.folder.getAbsolutePath();
			if (!FolderResourceIterator.folderPath.endsWith(File.separator)) {
				FolderResourceIterator.folderPath += File.separator;
			}

			if (!folder.exists()) {
				if (!folder.mkdirs()) {
					throw new IOException("Unable to create folder " + folder.getAbsolutePath());
				}
			}
			if (!folder.isDirectory()) {
				throw new IOException("File " + folder.getAbsolutePath() + " is invalid, not a directory.");
			}
			loadResources();
		}
	}

	/**
	 * Load the next resource in the list. If we face an error during loading,
	 * we try the next one until we find a valid resource or reach the end of
	 * the list.
	 * 
	 * @return the loaded Resource or null if we reached the end of the list.
	 */
	protected Resource loadNextResource() {
		Resource res = null;
		while (res == null && !FolderResourceIterator.resourceList.isEmpty()) {
			File file = new File(folderPath + FolderResourceIterator.resourceList.getFirst());
			FolderResourceIterator.resourceList.removeFirst();
			try {
				res = webLabMarshaller.unmarshal(file, Document.class);
			} catch (WebLabCheckedException e) {
				LOGGER.warn("Unable to unmarshal resource : " + e.getMessage());
			} finally {
				file.delete();
			}
		}
		return res;
	}

	protected boolean hasNext() {
		if (FolderResourceIterator.resourceList.isEmpty()) {
			try {
				loadResources();
			} catch (IOException e) {
				LOGGER.error(e);
			}
		}
		return FolderResourceIterator.resourceList.size() > 0;
	}

	@Override
	public NextResourceReturn nextResource(NextResourceArgs args) throws AccessDeniedException, ContentNotAvailableException, EmptyQueueException,
			InsufficientResourcesException, InvalidParameterException, ServiceNotConfiguredException, UnexpectedException, UnsupportedRequestException {
		synchronized (FolderResourceIterator.resourceList) {
			if (hasNext()) {
				Resource res = loadNextResource();

				if (res == null) {
					throw new EmptyQueueException("No more resources.", "Insufficient resources.");
				}
				LOGGER.trace("Resource " + res.getUri() + " loaded.");

				NextResourceReturn ret = new NextResourceReturn();
				ret.setResource(res);
				return ret;
			} else {
				throw new EmptyQueueException("No more resources.", "Insufficient resources.");
			}
		}
	}

	public static String getFolderPath() {
		return folderPath;
	}
	
	public static int getNbResources() {
		return resourceList.size();
	}

}
