/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2010 EADS - CASSIDIAN
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.portlet;

import java.io.File;
import java.io.IOException;
import java.io.StringWriter;
import java.net.MalformedURLException;
import java.net.URL;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.ResourceBundle;
import java.util.Map.Entry;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.portlet.ActionRequest;
import javax.portlet.ActionResponse;
import javax.portlet.EventRequest;
import javax.portlet.EventResponse;
import javax.portlet.PortletException;
import javax.portlet.PortletRequest;
import javax.portlet.PortletRequestDispatcher;
import javax.portlet.PortletResponse;
import javax.portlet.RenderRequest;
import javax.portlet.RenderResponse;
import javax.portlet.ResourceRequest;
import javax.portlet.ResourceResponse;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.ow2.weblab.core.extended.exception.WebLabCheckedException;
import org.ow2.weblab.core.extended.factory.AnnotationFactory;
import org.ow2.weblab.core.extended.factory.ResourceFactory;
import org.ow2.weblab.core.extended.jaxb.WebLabMarshaller;
import org.ow2.weblab.core.extended.ontologies.WebLabRetrieval;
import org.ow2.weblab.core.extended.util.ResourceUtil;
import org.ow2.weblab.core.extended.util.ServiceUtil;
import org.ow2.weblab.core.helper.PoKHelper;
import org.ow2.weblab.core.helper.impl.AdvancedSelector;
import org.ow2.weblab.core.helper.impl.JenaPoKHelper;
import org.ow2.weblab.core.helper.impl.RDFSelectorFactory;
import org.ow2.weblab.core.helper.impl.Statements;
import org.ow2.weblab.core.model.Annotation;
import org.ow2.weblab.core.model.PieceOfKnowledge;
import org.ow2.weblab.core.model.ResultSet;
import org.ow2.weblab.core.model.StringQuery;
import org.ow2.weblab.core.services.Analyser;
import org.ow2.weblab.core.services.InvalidParameterException;
import org.ow2.weblab.core.services.Searcher;
import org.ow2.weblab.core.services.analyser.ProcessArgs;
import org.ow2.weblab.core.services.analyser.ProcessReturn;
import org.ow2.weblab.core.services.searcher.SearchArgs;
import org.ow2.weblab.core.services.searcher.SearchReturn;
import org.ow2.weblab.portlet.tool.AdvancedSearchConfigBean;
import org.ow2.weblab.portlet.tool.AdvancedSearchFieldBean;
import org.ow2.weblab.portlet.tool.DateFieldBean;
import org.ow2.weblab.portlet.tool.SearchServiceBean;
import org.ow2.weblab.portlet.tool.TextFieldBean;
import org.springframework.beans.factory.BeanFactory;
import org.springframework.beans.factory.xml.XmlBeanFactory;
import org.springframework.core.io.ClassPathResource;

/**
 * Portlet used to generate text requests. When request is submitted, a
 * RequestEvent (request and service definition) is sent. Portlet allow
 * modification by user of the searcher service URL to call.
 * 
 * @author emilien, nbureau
 * 
 */
public class AdvancedSearchPortlet extends WebLabPortlet {
	private static final String DO_SAVE_QUERY = "doSaveQuery";

	private static final String DO_SEARCH = "doSearch";

	private static Log logger = LogFactory.getLog(AdvancedSearchPortlet.class);

	protected static final String DEFAULT_SYNTAX_NAME = "lucene";
	private static final String DEFAULT_SEARCH = "def_mess";

	private static String SEARCH_ACTION = "search";
	private static String NEXT_DOCUMENTS_ACTION = "nextDocuments";
	private static String RESET_ACTION = "resetSearch";
	private static String SAVE_QUERY_ACTION = "saveQuery";
	
	private static String NEXT_DOCUMENTS_REACTION = "sendNextDocuments";
	private static String FACET_SELECTION_REACTION = "facetSelection";
	private static String FACET_UNSELECTION_REACTION = "facetUnselection";

	public static String ENABLE_ADVANCED_SEARCH = "enableAdvancedSearch";
	public static String ENABLE_SAVE = "enableSave";
	public static String ENABLE_GENERATE = "enableGenerate";

	public static String TEXT_KEYWORDS_INPUT = "textInput";
	public static String LAST_TEXT_KEYWORDS_INPUT = "lastTextInput";
	public static String QUERY_INPUT = "queryInput";
	public static String SUBMIT_INPUT = "submitInput";
	public static String USER_LAST_SEARCH = "user_last_search";
	public static String USER_ESB_CONF = "user_esb_conf";
	public static String SEARCH_ERROR = "searcher_error";
	public static String ESB_CONFIG_NAME = "esb_config_bean";
	public static String SEARCH_SERVICES_MAP = "search_services_map";
	public static String DISPLAY_CHECKING_POPUP = "display_checking_popup";
	public static String SEARCH_SERVICE = "searchServiceInput";
	public static String LAST_SEARCH_SERVICE = "last_search_service";
	public static String NATIVE_SYNTAX = "native_syntax";
	public static String NATIVE_SYNTAX_DISPLAYED = "native_syntax_displayed";

	private static String OFFSET_PROPERTY = WebLabRetrieval.HAS_EXPECTED_OFFSET;
	private static String LIMIT_PROPERTY = WebLabRetrieval.HAS_EXPECTED_LIMIT;
	private static String HTML_CONTENT_TYPE = "text/html; charset=utf-8";
	private static String TEXT_CONTENT_TYPE = "text/plain; charset=utf-8";
	
	protected static AdvancedSearchConfigBean advanced_search_conf;
	
	protected static URL SEARCHER_WSDL_LOCATION;
	protected static Map<String, Searcher> searchers;
	protected static Map<String, Analyser> translators;
	protected static Map<String, String> usageContexts;

	protected static int distinct_submit_query;
	protected static boolean USE_REDIRECTION;

	private static final SimpleDateFormat DATE_FORMAT = new SimpleDateFormat("yyyy-MM-dd");
	private static final Date OLDEST_DATE = new Date(0);

	@Override
	public void init() throws PortletException {
		/*
		 * init WebLab portlet for event mapping
		 */
		super.init();

		ClassPathResource resource = new ClassPathResource(AdvancedSearchConfigBean.DEFAULT_CONF_FILE);
		BeanFactory factory = new XmlBeanFactory(resource);

		advanced_search_conf = factory.getBean(AdvancedSearchConfigBean.DEFAULT_BEAN_NAME, AdvancedSearchConfigBean.class);

		/*
		 * initialize default configuration from portlet.xml
		 */

		try {
			/*
			 * getting searcher and analyser WSDL
			 */
			try {
				SEARCHER_WSDL_LOCATION = new File(getPortletContext().getRealPath("WEB-INF/classes/services/WebLab.wsdl")).toURI().toURL();
			} catch (MalformedURLException e) {
				logger.error(e);
			}

			/*
			 * getting searchers and translators defined in
			 * AdvancedSearchConfigBean
			 */
			searchers = new HashMap<String, Searcher>();
			translators = new HashMap<String, Analyser>();
			usageContexts = new HashMap<String, String>();
			for (String serviceId : advanced_search_conf.getSearchServices().keySet()) {
				SearchServiceBean serviceBean = advanced_search_conf.getSearchServices().get(serviceId);
				// Add searcher
				Searcher searcher = ServiceUtil.getSearcherService(SEARCHER_WSDL_LOCATION, new URL(serviceBean.getURI()));
				searchers.put(serviceId, searcher);
				// Add usageContext if exists
				if (serviceBean.getUsageContext() != null && !serviceBean.getUsageContext().isEmpty()) {
					usageContexts.put(serviceId, serviceBean.getUsageContext());
					logger.info("add uc: " + serviceBean.getUsageContext() + " for searcher " + serviceId);
				}

				// Add translator if exists
				if (serviceBean.getTranslatorClass() != null && !serviceBean.getTranslatorClass().isEmpty()) {
					Class<?> clazz;
					try {
						clazz = Class.forName(serviceBean.getTranslatorClass());

						if (Analyser.class.isAssignableFrom(clazz)) {
							Analyser queryAnalyser = Analyser.class.cast(clazz.newInstance());
							translators.put(serviceId, queryAnalyser);
						}
					} catch (ClassNotFoundException e) {
						logger.error("Unbale to load QueryTranslator class for: " + serviceId);
					} catch (InstantiationException e) {
						logger.error("Unbale to instantiate QueryTranslator class for: " + serviceId);
					} catch (IllegalAccessException e) {
						logger.error("Access exception when trying to load QueryTranslator class for: " + serviceId);
					}
				}

			}
		} catch (Exception e) {
			logger.error("Unbale to get service URI");
		}

		/*
		 * Action/reaction name definition (default names are used as init-param names)
		 */
		if (getInitParameter(SEARCH_ACTION) != null){
			SEARCH_ACTION = getInitParameter(SEARCH_ACTION);
			logger.info("SEARCH_ACTION is fixed to "+SEARCH_ACTION);
		}
		if (getInitParameter(NEXT_DOCUMENTS_ACTION) != null){
			NEXT_DOCUMENTS_ACTION = getInitParameter(NEXT_DOCUMENTS_ACTION);
			logger.info("NEXT_DOCUMENTS_ACTION is fixed to "+NEXT_DOCUMENTS_ACTION);
		}
		if (getInitParameter(RESET_ACTION) != null){
			RESET_ACTION = getInitParameter(RESET_ACTION);
			logger.info("RESET_ACTION is fixed to "+RESET_ACTION);
		}
		if (getInitParameter(SAVE_QUERY_ACTION) != null){
			SAVE_QUERY_ACTION = getInitParameter(SAVE_QUERY_ACTION);
			logger.info("SAVE_QUERY_ACTION is fixed to "+SAVE_QUERY_ACTION);
		}
		
		if (getInitParameter(NEXT_DOCUMENTS_REACTION) != null){
			NEXT_DOCUMENTS_REACTION = getInitParameter(NEXT_DOCUMENTS_REACTION);
			logger.info("NEXT_DOCUMENTS_REACTION is fixed to "+NEXT_DOCUMENTS_REACTION);
		}
		if (getInitParameter(FACET_SELECTION_REACTION) != null){
			FACET_SELECTION_REACTION = getInitParameter(FACET_SELECTION_REACTION);
			logger.info("FACET_SELECTION_REACTION is fixed to "+FACET_SELECTION_REACTION);
		}
		if (getInitParameter(FACET_UNSELECTION_REACTION) != null){
			FACET_UNSELECTION_REACTION = getInitParameter(FACET_UNSELECTION_REACTION);
			logger.info("FACET_UNSELECTION_REACTION is fixed to "+FACET_UNSELECTION_REACTION);
		}

		/*
		 * setting the redirect parameter
		 */
		if (getInitParameter("use_redirection") != null) {
			USE_REDIRECTION = Boolean.parseBoolean(getInitParameter("use_redirection"));
		} else {
			USE_REDIRECTION = false;
		}
		distinct_submit_query = 0;
	}

	@Override
	public void destroy() {
		super.destroy();
	}

	/**
	 * Render HTML corresponding to the view mode
	 */
	@Override
	public void doView(RenderRequest req, RenderResponse res) throws IOException, PortletException {

		if (req.getParameter(SEARCH_ERROR) != null) {
			/*
			 * dispatch to jsp
			 */
			req.setAttribute(SEARCH_ERROR, req.getParameter(SEARCH_ERROR));
			logger.info("error in searcher");
			PortletRequestDispatcher rd = getPortletContext().getRequestDispatcher(getInitParameter("error_page_url"));
			rd.include(req, res);
		}

		if (req.getPortletSession().getAttribute(LAST_TEXT_KEYWORDS_INPUT) == null) {
			req.setAttribute(LAST_TEXT_KEYWORDS_INPUT, ResourceBundle.getBundle("search_portlet", req.getLocale()).getString("search.defaultText"));
			req.setAttribute(DEFAULT_SEARCH, true);
		} else {
			req.setAttribute(LAST_TEXT_KEYWORDS_INPUT, req.getPortletSession().getAttribute(LAST_TEXT_KEYWORDS_INPUT));
			req.setAttribute(DEFAULT_SEARCH, false);
		}

		req.setAttribute(SEARCH_SERVICES_MAP, advanced_search_conf.getSearchServices());
		req.setAttribute(DISPLAY_CHECKING_POPUP, advanced_search_conf.isDisplayCheckingPopup());

		if (req.getPortletSession().getAttribute(LAST_SEARCH_SERVICE) == null) {
			if (advanced_search_conf.getSearchServices().keySet().toArray().length > 0) {
				req.setAttribute(LAST_SEARCH_SERVICE, advanced_search_conf.getSearchServices().keySet().toArray()[0]);
			}
		} else {
			req.setAttribute(LAST_SEARCH_SERVICE, req.getPortletSession().getAttribute(LAST_SEARCH_SERVICE));
		}

		if (req.getPortletSession().getAttribute(NATIVE_SYNTAX) == null) {
			req.setAttribute(NATIVE_SYNTAX, false);
		} else {
			req.setAttribute(NATIVE_SYNTAX, req.getPortletSession().getAttribute(NATIVE_SYNTAX));
		}

		req.setAttribute(ENABLE_ADVANCED_SEARCH, advanced_search_conf.isEnableAdvancedSearch());
		req.setAttribute(ENABLE_SAVE, advanced_search_conf.isEnableSaveQuery());
		req.setAttribute(ENABLE_GENERATE, advanced_search_conf.isEnableGenerate());

		req.setAttribute(NATIVE_SYNTAX_DISPLAYED, advanced_search_conf.isNativeSyntaxOptionDisplayed());

		/*
		 * dispatch to jsp
		 */
		PortletRequestDispatcher rd = getPortletContext().getRequestDispatcher(getInitParameter("search_page_url"));
		rd.include(req, res);

	}

	@Override
	public void doEdit(RenderRequest req, RenderResponse res) throws PortletException, IOException {
		/*
		 * dispatch to jsp
		 */
		PortletRequestDispatcher rd = getPortletContext().getRequestDispatcher(getInitParameter("edit_page_url"));
		rd.include(req, res);
	}

	@Override
	public void processAction(ActionRequest req, ActionResponse res) throws IOException, PortletException {
		/*
		 * get the form name
		 */
		String form_name = req.getParameter("form_name");
		/*
		 * get the form action
		 */
		String form_action = req.getParameter("form_action");

		/*
		 * mapping
		 */
		if (form_name != null) {
			if (form_name.equals("search_query")) {
				logger.debug("Do 'search' action.");
				List<AdvancedSearchFieldBean> advancedSearchFieldList = AdvancedSearchPortlet.getFieldListFromParameters(req.getParameterMap());

				req.getPortletSession().setAttribute("AdvancedSearchFieldBeanList", advancedSearchFieldList);
				req.getPortletSession().setAttribute("AdvancedSearchTextSearch", req.getParameter(TEXT_KEYWORDS_INPUT));

				String query = AdvancedSearchPortlet.generateQuery(req.getParameter(TEXT_KEYWORDS_INPUT), advancedSearchFieldList);
				/*
				 * create query
				 */
				StringQuery q = ResourceFactory.createResource(getDefaultNamespace() + getPortletName(), "queryID_" + distinct_submit_query++,
						StringQuery.class);

				q.setRequest(query);

				/*
				 * setting nativeSyntax
				 */
				boolean isExpressedInNativeSyntax = false;
				if (req.getParameter("searchBox_nativeSyntax") != null && req.getParameter("searchBox_nativeSyntax").equals("on")) {
					logger.debug("Request will be parsed as native");
					isExpressedInNativeSyntax = true;
				}

				req.getPortletSession().setAttribute(NATIVE_SYNTAX, isExpressedInNativeSyntax);

				/*
				 * Add isExpressedWith property
				 */
				Annotation annot = AnnotationFactory.createAndLinkAnnotation(q);
				PoKHelper h = new JenaPoKHelper(annot);
				if (isExpressedInNativeSyntax) {
					h.createLitStat(q.getUri(), WebLabRetrieval.IS_EXPRESSED_WITH, req.getParameter(SEARCH_SERVICE));
					logger.debug("Request parsed for " + req.getParameter(SEARCH_SERVICE) + " engine");
				} else {
					h.createLitStat(q.getUri(), WebLabRetrieval.IS_EXPRESSED_WITH, DEFAULT_SYNTAX_NAME);
					logger.debug("Request parsed with default syntax");
				}

				if (form_action.compareToIgnoreCase(DO_SEARCH) == 0) {
					logger.info("Query sent to index : " + q.getRequest());
					doSearch(req, res, q);
				} else if (form_action.compareToIgnoreCase(DO_SAVE_QUERY) == 0) {
					doSave(req, res, q);
				}
				
			} else if (form_name.equals("reset")) {
				logger.debug("Do 'reset' action.");
				req.getPortletSession().removeAttribute(LAST_TEXT_KEYWORDS_INPUT);
				req.getPortletSession().removeAttribute("AdvancedSearchFieldBeanList");
				ResultSet emptyRS = ResourceFactory.createResource("AdvancedSearchPortlet_ResultSet", "" + System.currentTimeMillis(), ResultSet.class);
				sendEventForAction(RESET_ACTION, emptyRS, res);
			}
		}
	}

	@SuppressWarnings("unchecked")
	@Override
	public void serveResource(ResourceRequest request, ResourceResponse response) throws PortletException, IOException {

		/*
		 * serving advanced search field
		 */
		if (request.getParameter("page_part").equals("advancedSearchField")) {
			List<AdvancedSearchFieldBean> fieldList = (List<AdvancedSearchFieldBean>) request.getPortletSession().getAttribute("AdvancedSearchFieldBeanList");

			if (fieldList == null) {
				fieldList = new LinkedList<AdvancedSearchFieldBean>();
			}

			/*
			 * setting advanced search field list
			 */
			request.setAttribute("advancedSearchStructure", fieldList);
			/*
			 * setting boolean conditions
			 */
			request.setAttribute("boolean_conditions", advanced_search_conf.getBooleanConditions());
			/*
			 * setting field list
			 */
			request.setAttribute("advanced_search_fields", advanced_search_conf.getAdvancedSearchFields());
			/*
			 * dispatch to jsp
			 */
			response.setContentType(HTML_CONTENT_TYPE);
			PortletRequestDispatcher rd = getPortletContext().getRequestDispatcher(getInitParameter("advanced_search_field_page_url"));
			rd.include(request, response);

		} else if (request.getParameter("page_part").equals("nativeSyntax")) {
			Analyser translatorToUse = translators.get(request.getParameter("searchServiceInput"));

			logger.info(request.getParameter("searchServiceInput") + " translator is used.");

			List<AdvancedSearchFieldBean> fieldList = AdvancedSearchPortlet.getFieldListFromParameters(request.getParameterMap());
			String query = AdvancedSearchPortlet.generateQuery(request.getParameter(TEXT_KEYWORDS_INPUT), fieldList);

			/*
			 * create args for query translator
			 */
			StringQuery q = ResourceFactory.createResource(getDefaultNamespace() + getPortletName(), "queryID_" + distinct_submit_query++, StringQuery.class);
			q.setRequest(query);

			ProcessArgs translatorArgs = new ProcessArgs();
			translatorArgs.setResource(q);
			ProcessReturn translatorResult;
			if (translatorToUse != null) {
				try {
					translatorResult = translatorToUse.process(translatorArgs);
					StringQuery resultQuery = (StringQuery) translatorResult.getResource();
					logger.info("query translated : " + resultQuery.getRequest());
					response.setContentType(TEXT_CONTENT_TYPE);
					response.getWriter().print(resultQuery.getRequest());
				} catch (Exception pe) {
					logger.error(pe);
					response.setProperty(SEARCH_ERROR, pe.getMessage());
				}
			} else {
				response.setContentType(TEXT_CONTENT_TYPE);
				response.getWriter().print(query);
			}
			response.getWriter().flush();
		} else if (request.getParameter("page_part").equals("advancedSearchNewLine")) {
			AdvancedSearchFieldBean newLine;

			boolean launchFormAfter = false;
			if (request.getParameter("launchFormAfter") != null)
				launchFormAfter = Boolean.parseBoolean(request.getParameter("launchFormAfter"));

			int position = 0;
			if (request.getParameter("position") != null)
				position = Integer.parseInt(request.getParameter("position"));

			if (request.getParameter("booleanCondition") != null && request.getParameter("fieldName") != null && request.getParameter("fieldValue") != null
					&& request.getParameter("fieldListValue") != null && request.getParameter("fieldBeforeValue") != null
					&& request.getParameter("fieldAfterValue") != null) {
				String booleanCondition = request.getParameter("booleanCondition");

				// FieldName is composed by field type and name, try to retrieve
				// the name :
				String[] fieldTypeAndName = request.getParameter("fieldName").split(":");
				String fieldType = fieldTypeAndName[0];
				String fieldName = fieldTypeAndName[1];

				if (fieldType.equals("date")) {
					String beforeValue = "";
					if (request.getParameter("fieldBeforeValue") != null && !request.getParameter("fieldBeforeValue").isEmpty()) {
						beforeValue = request.getParameter("fieldBeforeValue");
					}
					String afterValue = "";
					if (request.getParameter("fieldAfterValue") != null && !request.getParameter("fieldAfterValue").isEmpty())
						afterValue = request.getParameter("fieldAfterValue");

					newLine = new DateFieldBean(position, booleanCondition, fieldName, beforeValue, afterValue);
				} else if (fieldType.equals("list")) {
					String value = "";
					if (request.getParameter("fieldListValue") != null && !request.getParameter("fieldListValue").isEmpty()) {
						value = request.getParameter("fieldListValue");
					}
					newLine = new TextFieldBean(position, AdvancedSearchFieldBean.Types.LIST_TYPE, booleanCondition, fieldName, value);
				} else {
					String value = "";
					if (request.getParameter("fieldValue") != null && !request.getParameter("fieldValue").isEmpty()) {
						value = request.getParameter("fieldValue");
					}
					newLine = new TextFieldBean(position, AdvancedSearchFieldBean.Types.TEXT_TYPE, booleanCondition, fieldName, value);
				}

				logger.info("New field added : " + newLine);

				/*
				 * setting new field
				 */
				request.setAttribute("advancedSearchNewLine", newLine);

				if (request.getPortletSession().getAttribute("AdvancedSearchFieldBeanList") != null) {
					((List<AdvancedSearchFieldBean>) request.getPortletSession().getAttribute("AdvancedSearchFieldBeanList")).add(newLine);
				} else {
					List<AdvancedSearchFieldBean> fieldList = new LinkedList<AdvancedSearchFieldBean>();
					fieldList.add(newLine);
					request.getPortletSession().setAttribute("AdvancedSearchFieldBeanList", fieldList);
				}
			}
			/*
			 * setting boolean conditions
			 */
			request.setAttribute("boolean_conditions", advanced_search_conf.getBooleanConditions());
			/*
			 * setting field list
			 */
			request.setAttribute("advanced_search_fields", advanced_search_conf.getAdvancedSearchFields());
			request.setAttribute("launchFormAfter", launchFormAfter);

			/*
			 * dispatch to jsp
			 */
			response.setContentType(HTML_CONTENT_TYPE);
			PortletRequestDispatcher rd = getPortletContext().getRequestDispatcher(getInitParameter("advanced_search_new_field_page_url"));
			rd.include(request, response);
		} else if (request.getParameter("page_part").equals("advancedSearchRemoveLine")) {
			response.setContentType(HTML_CONTENT_TYPE);
			int position = Integer.parseInt(request.getParameter("position"));

			logger.info("Removing field at position :" + position);
			((List<AdvancedSearchFieldBean>) request.getPortletSession().getAttribute("AdvancedSearchFieldBeanList")).remove(position);

			/*
			 * setting advanced search field list
			 */
			request.setAttribute("advancedSearchStructure",
					request.getPortletSession().getAttribute("AdvancedSearchFieldBeanList"));
			/*
			 * setting boolean conditions
			 */
			request.setAttribute("boolean_conditions", advanced_search_conf.getBooleanConditions());
			/*
			 * setting field list
			 */
			request.setAttribute("advanced_search_fields", advanced_search_conf.getAdvancedSearchFields());

			/*
			 * dispatch to jsp
			 */
			response.setContentType(HTML_CONTENT_TYPE);
			PortletRequestDispatcher rd = getPortletContext().getRequestDispatcher(getInitParameter("advanced_search_field_page_url"));
			rd.include(request, response);

		} else if (request.getParameter("page_part").equals("generateQuery")) {
			List<AdvancedSearchFieldBean> fieldList = AdvancedSearchPortlet.getFieldListFromParameters(request.getParameterMap());
			String query = AdvancedSearchPortlet.generateQuery(request.getParameter(TEXT_KEYWORDS_INPUT), fieldList);
			response.setContentType(TEXT_CONTENT_TYPE);
			response.getWriter().print(query);
			response.getWriter().flush();
		} else if (request.getParameter("page_part").equals("getSelectValues")) {
			response.setContentType(TEXT_CONTENT_TYPE);

			String id = request.getParameter("id");
			Map<String, String> valuesMap;
			if (id.equals("format")) {
				valuesMap = advanced_search_conf.getFormatValues();
			} else if (id.equals("language")) {
				valuesMap = advanced_search_conf.getLanguageValues();
			} else {
				valuesMap = new HashMap<String, String>();
			}

			StringBuilder responseSB = new StringBuilder();
			responseSB.append("[{\"optionValue\": \"\", \"optionDisplay\": \"-\"},");
			for (Entry<String, String> entry : valuesMap.entrySet()) {
				responseSB.append('{');
				responseSB.append("\"optionValue\": ");
				responseSB.append("\"").append(entry.getKey()).append("\"");
				responseSB.append(", \"optionDisplay\": ");
				responseSB.append("\"").append(ResourceBundle.getBundle("search_portlet", request.getLocale()).getString(entry.getValue())).append("\"");
				responseSB.append("},");
			}
			responseSB.deleteCharAt(responseSB.length() - 1);
			responseSB.append(']');

			response.getWriter().print(responseSB.toString());
			response.getWriter().flush();
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * javax.portlet.GenericPortlet#processEvent(javax.portlet.EventRequest,
	 * javax.portlet.EventResponse)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public void processEvent(EventRequest request, EventResponse response) throws PortletException, IOException {
		long start = System.currentTimeMillis();
		String reaction = getReaction(request.getEvent().getQName()).getLocalPart();
		if (reaction.equals(NEXT_DOCUMENTS_REACTION)) {

			/*
			 * checking if the portlet is the query producer
			 */

			if (request.getEvent().getValue() != null && request.getEvent().getValue() instanceof PieceOfKnowledge) {
				PieceOfKnowledge pok = (PieceOfKnowledge) request.getEvent().getValue();
				AdvancedSelector rdfSelector = RDFSelectorFactory.getSelector(true);
				Statements map = rdfSelector.searchFor(pok);

				String queryURI = null;
				queryURI = map.getTypedValue(pok.getUri(), WebLabRetrieval.HAS_ORDERED_QUERY, String.class);

				logger.debug("Map of pok : " + map);
				logger.debug("QueryID : " + queryURI);

				SearchArgs userSearchArg = (SearchArgs) request.getPortletSession().getAttribute(USER_LAST_SEARCH);

				if (userSearchArg != null && queryURI != null && userSearchArg.getQuery().getUri().equals(queryURI)) {
					/*
					 * this portlet has generated the query, redo search new
					 * offset and limit
					 */
					userSearchArg.setOffset(Integer.parseInt((String) map.entrySet().iterator().next().getValue().getValue(OFFSET_PROPERTY)));
					userSearchArg.setLimit(Integer.parseInt((String) map.entrySet().iterator().next().getValue().getValue(LIMIT_PROPERTY)));

					logger.info(request.getPortletSession().getAttribute(LAST_SEARCH_SERVICE) + " searcher is used : updating a resultSet, offset:"
							+ userSearchArg.getOffset() + " limit:" + userSearchArg.getLimit());

					// process query to the searcher
					try {
						SearchReturn searchResult = launchQuery(request, response, userSearchArg);
						logger.info(searchResult.getResultSet());
						sendEventForAction(NEXT_DOCUMENTS_ACTION, searchResult.getResultSet(), response);
					} catch (Exception e) {
						logger.error(e);
						response.setRenderParameter(SEARCH_ERROR, e.getMessage());
					}
				}
			}
		} else if (reaction.equals(FACET_SELECTION_REACTION) || reaction.equals(FACET_UNSELECTION_REACTION)) {
			logger.info("Facet selection event.");
			if (!(request.getEvent().getValue() != null && request.getEvent().getValue() instanceof PieceOfKnowledge)) {
				logger.warn("No value in facet selection event, or it's not a valid PoK.");
			}else{
				PieceOfKnowledge pok = (PieceOfKnowledge) request.getEvent().getValue();
				final AdvancedSelector rdfSelector = RDFSelectorFactory.getSelector(true);
				final Statements map = rdfSelector.searchFor(pok);
				
				final String queryURI = map.getTypedValue(pok.getUri(), WebLabRetrieval.HAS_ORDERED_QUERY, String.class);

				logger.debug("Map of pok : " + map);
				logger.debug("QueryID : " + queryURI);

				final SearchArgs userSearchArg = (SearchArgs) request.getPortletSession().getAttribute(USER_LAST_SEARCH);

				if (!(userSearchArg != null &&  queryURI != null && userSearchArg.getQuery().getUri().equals(queryURI))) {
					logger.warn("userSearchArg is null or problem with queryURI.");
					logger.debug("userSearchArg="+userSearchArg+"\nQueryID=" + queryURI);
				}else{
					logger.debug("User queryID: " + userSearchArg.getQuery().getUri());
					/*
					 * this portlet has generated the query, redo search with
					 * filter
					 */
					if (!(userSearchArg.getQuery() instanceof StringQuery)) {
						logger.error("Could not filter on non StringQuery requests.");
					}else{
						String filterValue = map.getTypedValue(pok.getUri(), getInitParameter("facetFilterQueryProperty"), String.class);
						StringQuery lastSearchQuery = (StringQuery) userSearchArg.getQuery();
						String lastSearch = lastSearchQuery.getRequest();

						String booleanOperator = filterValue.contains("AND") ? "AND" : "NOT";
						String fieldName = filterValue.replaceFirst(booleanOperator, "").split(":")[0].trim();
						String fieldValue = filterValue.replaceFirst(booleanOperator, "").split(":")[1].trim();

						List<AdvancedSearchFieldBean> fieldList;
						if (request.getPortletSession().getAttribute("AdvancedSearchFieldBeanList") != null) {
							fieldList = (List<AdvancedSearchFieldBean>) request.getPortletSession().getAttribute("AdvancedSearchFieldBeanList");
						} else {
							fieldList = new LinkedList<AdvancedSearchFieldBean>();
						}

						TextFieldBean facetFilterField = new TextFieldBean(-1, AdvancedSearchFieldBean.Types.TEXT_TYPE, booleanOperator, fieldName, fieldValue);

						if (reaction.equals(FACET_SELECTION_REACTION)) {
							lastSearchQuery.setRequest("(" + lastSearch + ") " + filterValue);
							fieldList.add(facetFilterField);
						} else if (reaction.equals(FACET_UNSELECTION_REACTION)) {
							lastSearchQuery.setRequest(lastSearch.replaceFirst("\\(", "").replaceFirst("\\)\\s*" + Pattern.quote(filterValue) + "($|\\s*\\))",
									"$1"));

							Iterator<AdvancedSearchFieldBean> fieldsIterator = fieldList.iterator();
							while (fieldsIterator.hasNext()) {
								AdvancedSearchFieldBean field = fieldsIterator.next();
								if (field instanceof TextFieldBean && field.getBooleanCondition().equals(booleanOperator)
										&& field.getFieldName().equals(fieldName) && ((TextFieldBean) field).getValue().equals(fieldValue)) {
									fieldsIterator.remove();
								}
							}
						}

						request.getPortletSession().setAttribute("AdvancedSearchFieldBeanList", fieldList);
						userSearchArg.setOffset(0);
						userSearchArg.setLimit(10);
						userSearchArg.setQuery(lastSearchQuery);

						// Searcher searcherToUse =
						// searchers.get(request.getPortletSession().getAttribute(LAST_SEARCH_SERVICE));
						logger.info(request.getPortletSession().getAttribute(LAST_SEARCH_SERVICE) + " searcher is used, create a facet filter: "
								+ lastSearchQuery.getRequest());
						/*
						 * process query to the searcher
						 */
						// final SearchReturn searchResult;
						try {
							// if (searcherToUse != null) {
							SearchReturn searchResult = launchQuery(request, response, userSearchArg);
							sendEventForAction(SEARCH_ACTION, searchResult.getResultSet(), response);
							// }
						} catch (final Exception e) {
							logger.error(e);
							response.setRenderParameter(SEARCH_ERROR, e.getMessage());
						}
					}
				}
			}
		}
		logger.info("processEvent response time :" + Long.toString(System.currentTimeMillis() - start));
	}

	/**
	 * Send the received query in a "saveQuery" event.
	 * 
	 * @param req
	 * @param res
	 * @param q
	 */
	protected void doSave(ActionRequest req, ActionResponse res, StringQuery q) {
		logger.info("Saving query...");
		sendEventForAction(SAVE_QUERY_ACTION, q, res);
		logger.debug("Save Query Event sent.");
	}

	/**
	 * Sending query to search engine
	 * 
	 * @param req
	 * @param res
	 * @param q
	 */
	protected void doSearch(ActionRequest req, ActionResponse res, StringQuery q) {
		logger.info("Sending query to search engine...");
		try {
			// create args for search service
			SearchArgs args = new SearchArgs();
			args.setLimit(10);
			args.setOffset(0);
			args.setQuery(q);

			if (req.getParameter(SEARCH_SERVICE) != null) {
				logger.debug("Sending to search service " + req.getParameter(SEARCH_SERVICE));

				try {
					WebLabMarshaller marsh_l = new WebLabMarshaller();
					StringWriter sw_l = new StringWriter();
					marsh_l.marshalResource(args.getQuery(), sw_l);
					logger.debug("with request :" + sw_l.toString());
					marsh_l = null;
					sw_l = null;					
				} catch (WebLabCheckedException e) {
					logger.debug("Debug marshalling for request failed");
				}
				
				req.getPortletSession().setAttribute(LAST_SEARCH_SERVICE, req.getParameter(SEARCH_SERVICE));
			}

			SearchReturn searchResult = launchQuery(req, res, args);
			
			logger.debug("Results received.");
			
			if (searchResult != null) {
				if (USE_REDIRECTION) {
					logger.debug("sendEventForActionAndRedirect "+SEARCH_ACTION+" to share results.");
					sendEventForActionAndRedirect(SEARCH_ACTION, searchResult.getResultSet(), res);
				} else {
					logger.debug("sendEventForAction "+SEARCH_ACTION+" to share results.");
					sendEventForAction(SEARCH_ACTION, searchResult.getResultSet(), res);
				}
			}else{
				logger.error("Search results are nempty or null.");
			}
		} catch (PortletException e) {
			logger.error(e);
			res.setRenderParameter(SEARCH_ERROR, e.getMessage());
		} catch (IOException e) {
			logger.error(e);
			res.setRenderParameter(SEARCH_ERROR, e.getMessage());
		}
	}

	/**
	 * Launch the query to the selected Searcher
	 * 
	 * @param req
	 * @param res
	 * @param q
	 */
	protected SearchReturn launchQuery(PortletRequest req, PortletResponse res, SearchArgs args) throws PortletException {
		SearchReturn searchResult = null;
		// setting search args in user session
		req.getPortletSession().setAttribute(USER_LAST_SEARCH, args);

		StringQuery q = (StringQuery) args.getQuery();

		if (q.getRequest() != null && q.getRequest() != "") {
			try {
				// Use searcher selected by the user
				Searcher searcherToUse = searchers.get(req.getPortletSession().getAttribute(LAST_SEARCH_SERVICE));

				if (searcherToUse == null)
					searcherToUse = searchers.get(searchers.keySet().iterator().next());

				// process query to the searcher
				if (searcherToUse == null)
					throw new InvalidParameterException("Searcher is null.");

				String usageContext = usageContexts.get(req.getParameter(SEARCH_SERVICE));
				if (usageContext != null && !usageContext.isEmpty()) {
					args.setUsageContext(usageContext);
					logger.info("Usage context: " + usageContext + " is used.");
				}
				logger.trace("searcherToUse=" + searcherToUse);
				logger.trace("Before querying searcher");
				if(logger.isTraceEnabled()){
					logger.trace("SOAP query sent: "+ResourceUtil.saveToXMLString(args.getQuery()));
					logger.trace("Usage context: "+args.getUsageContext());
				}

				// do the search
				searchResult = searcherToUse.search(args);

				// setting text query in user session
				req.getPortletSession().setAttribute(LAST_TEXT_KEYWORDS_INPUT, req.getParameter(TEXT_KEYWORDS_INPUT));

				// setting search args in user session
				req.getPortletSession().setAttribute(USER_LAST_SEARCH, args);

				// // setting service used in user session
				// req.getPortletSession().setAttribute(LAST_SEARCH_SERVICE,
				// req.getParameter(SEARCH_SERVICE));

			} catch (Exception e) {
				logger.error("Error during search: " + e.getMessage());
				e.printStackTrace();
				throw new PortletException("Error during search.", e);
			}
		} else {
			logger.error("Empty query?!");
			throw new PortletException("Empty query?!");
		}
		return searchResult;
	}

	/**
	 * to get the search fields
	 */
	protected static List<AdvancedSearchFieldBean> getFieldListFromParameters(Map<String, String[]> parameters) {
		List<AdvancedSearchFieldBean> advancedSearchFieldList = new LinkedList<AdvancedSearchFieldBean>();
		Pattern p = Pattern.compile("advanced_booleanCondition_(\\d+)");

		for (Entry<String, String[]> entry : parameters.entrySet()) {
			if (entry.getKey().startsWith("advanced_booleanCondition_")) {
				Matcher m = p.matcher(entry.getKey());
				m.matches();
				int position = Integer.parseInt(m.group(1));

				if (parameters.get("advanced_fieldName_" + position) != null && parameters.get("advanced_fieldName_" + position)[0] != null
						&& !parameters.get("advanced_fieldName_" + position)[0].isEmpty()) {
					String[] fieldTypeAndName = parameters.get("advanced_fieldName_" + position)[0].split(":");
					String fieldType = fieldTypeAndName[0];
					String fieldName = fieldTypeAndName[1];

					boolean addField = true;
					AdvancedSearchFieldBean field;
					if (fieldType.equals("date")) {
						// By default, set before date to 01/01/1970
						String beforeValue = DATE_FORMAT.format(OLDEST_DATE);
						if (parameters.get("advanced_fieldBeforeValue_" + position) != null
								&& !parameters.get("advanced_fieldBeforeValue_" + position)[0].isEmpty()) {
							beforeValue = parameters.get("advanced_fieldBeforeValue_" + position)[0];
						}

						// By default, set after date to today
						String afterValue = DATE_FORMAT.format(Calendar.getInstance().getTime());
						if (parameters.get("advanced_fieldAfterValue_" + position) != null
								&& !parameters.get("advanced_fieldAfterValue_" + position)[0].isEmpty()) {
							afterValue = parameters.get("advanced_fieldAfterValue_" + position)[0];
						}

						field = new DateFieldBean(position, entry.getValue()[0], fieldName, beforeValue, afterValue);
					} else if (fieldType.equals("list")) {
						String value = "";
						if (parameters.get("advanced_fieldListValue_" + position) != null
								&& !parameters.get("advanced_fieldListValue_" + position)[0].isEmpty()) {
							value = parameters.get("advanced_fieldListValue_" + position)[0];
						} else {
							addField = false;
						}

						field = new TextFieldBean(position, AdvancedSearchFieldBean.Types.LIST_TYPE, entry.getValue()[0], fieldName, value);
					} else {
						String value = "";
						if (parameters.get("advanced_fieldValue_" + position) != null && !parameters.get("advanced_fieldValue_" + position)[0].isEmpty()) {
							value = parameters.get("advanced_fieldValue_" + position)[0];
						} else {
							addField = false;
						}
						field = new TextFieldBean(position, AdvancedSearchFieldBean.Types.TEXT_TYPE, entry.getValue()[0], fieldName, value);
					}

					if (addField) {
						advancedSearchFieldList.add(field);
					}
				} else {
					logger.error("Name field was null for element " + position);
				}
			}
		}
		Collections.sort(advancedSearchFieldList);
		return advancedSearchFieldList;
	}

	/**
	 * To generate the query from a search field list
	 * 
	 * @param textQuery
	 * @param advancedSearchFieldList
	 * @return
	 */
	protected static String generateQuery(String textQuery, List<AdvancedSearchFieldBean> advancedSearchFieldList) {
		// If text query is empty, remove the first boolean operator to make
		// correct syntax
		if (textQuery.isEmpty() && advancedSearchFieldList.size() > 0)
			advancedSearchFieldList.get(0).setBooleanCondition("");

		StringBuilder generatedQuery = new StringBuilder();
		for (AdvancedSearchFieldBean advancedSearchFieldBean : advancedSearchFieldList) {
			if (advancedSearchFieldBean.getBooleanCondition() != null && !advancedSearchFieldBean.getBooleanCondition().isEmpty())
				generatedQuery.append(advancedSearchFieldBean.getBooleanCondition()).append(" ");

			generatedQuery.append(advancedSearchFieldBean.getFieldName()).append(":");
			if (advancedSearchFieldBean instanceof TextFieldBean) {
				generatedQuery.append('(');
				generatedQuery.append(((TextFieldBean) advancedSearchFieldBean).getValue());
				generatedQuery.append(')');
			} else if (advancedSearchFieldBean instanceof DateFieldBean) {
				generatedQuery.append('[');
				generatedQuery.append(((DateFieldBean) advancedSearchFieldBean).getBeforeValue());
				generatedQuery.append(' ');
				generatedQuery.append(((DateFieldBean) advancedSearchFieldBean).getAfterValue());
				generatedQuery.append(']');
			}
			generatedQuery.append(' ');
		}
		if (generatedQuery.length() > 0)
			generatedQuery.deleteCharAt(generatedQuery.length() - 1);

		// return query
		if (generatedQuery.length() == 0)
			return textQuery;
		else if (textQuery.length() == 0)
			return generatedQuery.toString();
		else
			return '(' + textQuery + ')' + ' ' + generatedQuery.toString();
	}
}
