/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 *
 * Copyright (C) 2004 - 2009 EADS DEFENCE AND SECURITY SYSTEMS
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of the
 * GNU Lesser General Public License as published by the Free Software Foundation; either version
 * 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See
 * the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.portlet.tool;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Random;
import java.util.Set;

import org.apache.commons.logging.LogFactory;
import org.ow2.weblab.core.extended.ontologies.RDF;
import org.ow2.weblab.core.extended.ontologies.WebLab;
import org.ow2.weblab.core.helper.RDFHelperFactory;
import org.ow2.weblab.core.helper.ResourceHelper;
import org.ow2.weblab.core.model.Document;
import org.ow2.weblab.core.model.MediaUnit;
import org.ow2.weblab.core.model.Segment;
import org.ow2.weblab.core.model.Text;
import org.ow2.weblab.portlet.bean.LegendField;

/**
 * Factory to make the legend of a weblab media unit. Media unit is analised to
 * add all distinct annotations as fields. Each field corresponding a random
 * color and a name (corresponding the annotation type).
 * 
 * @author emilien
 */
public class LegendFactory {
	private static Random random = new Random();

	/**
	 * Constructor without parameters.
	 */
	private LegendFactory() {
		//
	}

	/**
	 * To get the MediaUnitLegend of the media unit.
	 * 
	 * @param mediaUnnit
	 *            the media unit used to make the legend.
	 * @return the legend of media unit.
	 */
	public static MediaUnitLegend getLegend(MediaUnit mediaUnit) {	
		LogFactory.getLog(LegendFactory.class).info(
				"Creating a new legend for media unit :" + mediaUnit.getUri());
		/*
		 * Creating MediaUnitLegend
		 */
		MediaUnitLegend leg = new MediaUnitLegend();
		Set<String> predicatSet = new HashSet<String>();
	
		List<LegendField> listField = new ArrayList<LegendField>();

		if (mediaUnit instanceof Text) {
			/*
			 * media is a Text
			 */
			matchEntities(mediaUnit, predicatSet);
		} else if (mediaUnit instanceof Document) {
			/*
			 * unit is a composed unit
			 */
			
			List<Text> mediaUnits = new ArrayList<Text>();
			for (MediaUnit subMU: ((Document)mediaUnit).getMediaUnit()) {
				if (subMU instanceof Text) {
					mediaUnits.add((Text) subMU);
				}
			}
			/*
			 * get all distinct entities
			 */
			for (MediaUnit unit : mediaUnits) {
				matchEntities(unit, predicatSet);
			}
		}

		/*
		 * build legend fields corresponding to distinct predicates
		 */
		for (String predicat : predicatSet) {
			/*
			 * processing predicate without values
			 */
			LegendField f = new LegendField();
			f.setDisplayName(predicat);
			f.setEntityType(predicat);
			f.setMatchedValue(null);
			f.setShow(true);
			String styleClassName = predicat.replace(".", "_");
			styleClassName = styleClassName.replace("http://", "http_");
			styleClassName = styleClassName.replace("/", "_");
			styleClassName = styleClassName.replace("#", "_");
			f.setStyleClassName(styleClassName);
			int r = random.nextInt(255);
			int g = random.nextInt(255);
			int b = random.nextInt(255);
			f.setStyleColor(r + "," + g + "," + b);
			listField.add(f);
				
		}
		leg.setFields(listField);
		return leg;
	}

	/**
	 * match all named entities of the unit and add its in the predicateSet map
	 * and its values in predicatValue map.
	 * 
	 * @param mediaUnit
	 * @param predicatSet
	 * @param predicatValueMap
	 */
	private static void matchEntities(MediaUnit mediaUnit,
			Set<String> predicatSet) {
		
		List<Segment> listSeg;
		listSeg = mediaUnit.getSegment();
		/*
		 * check distinct annotation for all linear segment of the mediaUnit.
		 */
		for (Segment seg : listSeg) {
			ResourceHelper rh = RDFHelperFactory.getResourceHelper(mediaUnit);
			/*
			 * get all "refersTo" properties
			 */
			List<String> refers = rh.getRessOnPredSubj(seg.getUri(), WebLab.REFERS_TO);
			for (final String uri : refers) {
				predicatSet.addAll(rh.getRessOnPredSubj(uri, RDF.TYPE));
			}
		}
	}
}