/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 *
 * Copyright (C) 2004 - 2009 EADS DEFENCE AND SECURITY SYSTEMS
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of the
 * GNU Lesser General Public License as published by the Free Software Foundation; either version
 * 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See
 * the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.portlet.tool;

import java.util.ArrayList;
import java.util.List;

import org.ow2.weblab.core.model.Document;
import org.ow2.weblab.core.model.MediaUnit;
import org.ow2.weblab.core.model.Text;

/**
 * MediaUnitSplitter manage splitted Text media units for pagination. It split a
 * given media unit in pages of given size. It manage page navigation to get
 * page offset of the current part.
 * 
 * @author emilien
 * 
 */
public class MediaUnitSplitter {

	private MediaUnit mu;
	private int textSize;
	private int current_page;
	private int total_text_size;
	private int pageCount;

	public MediaUnitSplitter() {
		this.current_page = 1;
	}

	/**
	 * Constructor
	 * 
	 * @param unit
	 *            the unit to split
	 * @param textSize
	 *            the size of each parts
	 */
	public MediaUnitSplitter(MediaUnit unit, int textSize) {
		this.mu = unit;
		this.textSize = textSize;
		this.current_page = 1;
		this.total_text_size = 0;
		/*
		 * get the final text size of the media unit
		 */
		getUnitTextSize();
		/*
		 * page count
		 */
		this.pageCount = this.total_text_size / textSize;
		if (this.total_text_size % textSize > 0) {
			this.pageCount++;
		}
	}

	/**
	 * The start offset of text in the current page
	 * 
	 * @return the text offset of current page
	 */
	public int getCurrentPageStartOffset() {
//		WebLabLogFactory.getStandardLog().info(
//				"start offset in splitter : "
//						+ ((this.current_page - 1) * this.textSize));
		return (this.current_page - 1) * this.textSize;
	}

	/**
	 * The end offset of text in the current page
	 * 
	 * @return the text offset of current page
	 */
	public int getCurrentPageEndOffset() {
		if (this.current_page == this.pageCount) {

//			WebLabLogFactory
//					.getStandardLog()
//					.info(
//							"end offset in splitter : "
//									+ ((this.current_page) * this.textSize
//											+ this.total_text_size - ((this.current_page) * this.textSize)));
			return (this.current_page) * this.textSize + this.total_text_size
					- ((this.current_page) * this.textSize);
		}
//		WebLabLogFactory.getStandardLog().info(
//				"end offset in splitter : "
//						+ ((this.current_page) * this.textSize));
		return (this.current_page) * this.textSize;
	}

	/**
	 * change current page to the next page, if exist.
	 * 
	 * @return the new current page number
	 */
	public int nextPage() {
		if (this.current_page < this.pageCount) {
			this.current_page++;
		}
		return this.current_page;
	}

	/**
	 * change current page to the previous page, if exist.
	 * 
	 * @return the new current page number
	 */
	public int prevPage() {
		if (this.current_page > 0) {
			this.current_page--;
		}
		return this.current_page;
	}

	/**
	 * Number of page for this splitter
	 * 
	 * @return the number of page of this splitter
	 */
	public int getAvaiblePageCount() {
		return this.pageCount;
	}

	/**
	 * Current page number
	 * 
	 * @return Current page number
	 */
	public int getCurrentPageIndex() {
		return this.current_page;
	}

	/**
	 * Set the current page for this splitter
	 * 
	 * @param currentPage
	 *            the page number to set
	 * @return the new current page number
	 */
	public int setCurrentPageIndex(int currentPage) {
		if (currentPage > 0 && currentPage <= this.pageCount) {
			this.current_page = currentPage;
		}

//		WebLabLogFactory.getStandardLog().info(
//				"current page in splitter : " + currentPage);
		return this.current_page;
	}

	/**
	 * Count the total text size of the splitter media unit. If the media unit
	 * is a composed media unit, sum all text parts
	 */
	private void getUnitTextSize() {
		if (this.mu instanceof Document) {
			
			List<Text> subText = new ArrayList<Text>();
			for (MediaUnit subMU : ((Document)mu).getMediaUnit()) {
				if (subMU instanceof Text) {
					subText.add((Text) subMU);
				}
			}
			
			
			
			for (Text text : subText) {
				this.total_text_size += text.getContent().length();
			}
		} else if (this.mu instanceof Text) {
			this.total_text_size = ((Text) this.mu).getContent().length();
		}
	}

	/*
	 * Getters and setters
	 */
	public MediaUnit getMediaUnit() {
		return this.mu;
	}

	public void setMediaUnit(MediaUnit mu) {
		this.mu = mu;
	}

	public int getTextSize() {
		return this.textSize;
	}

	public void setTextSize(int textSize) {
		this.textSize = textSize;
	}
}