/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 *
 * Copyright (C) 2004 - 2009 EADS DEFENCE AND SECURITY SYSTEMS
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of the
 * GNU Lesser General Public License as published by the Free Software Foundation; either version
 * 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See
 * the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.portlet.tool;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.ResourceBundle;
import java.util.Set;
import java.util.TreeSet;

import javax.portlet.PortletURL;

import org.apache.commons.logging.LogFactory;
import org.ow2.weblab.core.extended.comparator.SegmentComparator;
import org.ow2.weblab.core.extended.ontologies.RDF;
import org.ow2.weblab.core.extended.ontologies.RDFS;
import org.ow2.weblab.core.extended.ontologies.WebLabProcessing;
import org.ow2.weblab.core.extended.util.SegmentUtil;
import org.ow2.weblab.core.helper.impl.IPredicateValuePairs;
import org.ow2.weblab.core.helper.impl.Statements;
import org.ow2.weblab.core.model.Document;
import org.ow2.weblab.core.model.LinearSegment;
import org.ow2.weblab.core.model.MediaUnit;
import org.ow2.weblab.core.model.Segment;
import org.ow2.weblab.core.model.Text;
import org.ow2.weblab.portlet.bean.LegendField;
import org.ow2.weblab.portlet.bean.NamedEntity;
import org.ow2.weblab.portlet.bean.TextZone;



/**
 * A MediaUnitHighlighter render a highlighted Weblab media unit. Text of the
 * media unit is highlighted with elements of the legend. Render is done in HTML
 * representation using jQuery for overlapping. HTML text generated is
 * highlighted like specified in the legend
 * 
 * @author emilien
 * 
 */
public class MediaUnitHighLighter {

	private static boolean SHOW_INSTANCES_LINK = true;
	private static int HIGHLIGHTING_TIMEOUT = 1000;
	private static String UNIT_START_TAG_SEPARATOR = "<div class='log_data text_media_unit'>";
	private static String UNIT_END_TAG_SEPARATOR = "</div>";

	/**
	 * Public constructor without parameters.
	 */
	private MediaUnitHighLighter() {
		// do not use this constructor
	}

	/**
	 * get a HTML representation text parts of the media unit where annotations
	 * about entities are located inside span tag. span class corresponding
	 * context and the entity type. Each span tag has a background color, define
	 * in the legend.
	 * 
	 * Possibility to get parts of text with offsets.<br />
	 * 
	 * For Example &lt;div id="com_eads_weblab_div_content"
	 * class="com_eads_weblab_div_content"&gt; &lt;div&gt; a part of text with
	 * an &lt;span class='com_eads_weblab_annotation'&gt; annotation&lt/span&gt;
	 * &lt;/div&gt; &lt;/div&gt;
	 * 
	 * @return HTML code
	 */
	public static String getHtmlRepresentation(MediaUnit mediaUnit,
			String context, MediaUnitLegend legend, int startOffset,
			int endOffset, PortletURL actionURL, ResourceBundle resourceBundle, boolean showInstanceLink, boolean showToolTips, Statements mediaUnitWTMap) {

		int unitCharAdded = UNIT_START_TAG_SEPARATOR.length();
		int totalCharAdded = 0;
		SHOW_INSTANCES_LINK = showInstanceLink;
		StringBuilder strBuild = new StringBuilder();
		
		
		
		
		
		
		//System.out.println(mediaUnitWTMap);
		
		/*
		 * starting content tag div
		 */
		strBuild.append("<div id='");
		strBuild.append(context);
		strBuild.append("_div_content' class='log_data ");
		strBuild.append(context);
		strBuild.append("_div_content'>");
		/*
		 * Parts of text
		 */
		StringBuilder mergedTextUnits = new StringBuilder();
		/*
		 * Entities found list
		 */
		List<NamedEntity> listEN = new ArrayList<NamedEntity>();
		/*
		 * check the media unit type
		 */
		if (mediaUnit instanceof Document) {
			/*
			 * unit is a ComposedResource, processing each units
			 */

			int textPosition = 0;
			boolean canAddChar = true;
			int iunit = 0;
			List<Text> textSubMU = new ArrayList<Text>();
			
			for ( MediaUnit subMU: ((Document) mediaUnit).getMediaUnit()) {
				if(subMU instanceof Text) {
					textSubMU.add((Text) subMU);
				}
			}
		
			while (canAddChar && iunit < textSubMU.size() && textSubMU.get(iunit).getContent() != null) {

				Text text = textSubMU.get(iunit);
				int unitSize = text.getContent().length();
				if (textPosition < startOffset) {
					/*
					 * earlier position
					 */
					if (textPosition + unitSize > startOffset) {
						/*
						 * need to remove beginning
						 */
						if (startOffset
								+ (unitSize - (startOffset - textPosition)) > endOffset) {
							/*
							 * remove beginning and end
							 */
							mergedTextUnits.append(UNIT_START_TAG_SEPARATOR);
							mergedTextUnits.append(text.getContent().substring(
									startOffset - textPosition,
									endOffset - textPosition));
							matchEntitiesInUnit(text, legend, listEN,
									totalCharAdded, mediaUnitWTMap, startOffset - textPosition,
									endOffset - textPosition);
							mergedTextUnits.append(UNIT_END_TAG_SEPARATOR);
							totalCharAdded += unitCharAdded
									+ (endOffset - startOffset)
									+ UNIT_END_TAG_SEPARATOR.length();
							iunit++;
							textPosition += endOffset - startOffset;
							canAddChar = false;

						} else {
							/*
							 * only remove beginning
							 */
							mergedTextUnits.append(UNIT_START_TAG_SEPARATOR);
							mergedTextUnits.append(text.getContent().substring(
									(startOffset - textPosition), unitSize));
							matchEntitiesInUnit(text, legend, listEN,
									totalCharAdded, mediaUnitWTMap, startOffset - textPosition,
									unitSize);
							mergedTextUnits.append(UNIT_END_TAG_SEPARATOR);
							totalCharAdded += unitCharAdded + unitSize
									- (startOffset - textPosition)
									+ UNIT_END_TAG_SEPARATOR.length();
							iunit++;
							textPosition += unitSize;
						}
					} else {
						/*
						 * change for next unit
						 */
						iunit++;
						textPosition += unitSize;
					}
				} else {
					/*
					 * unit need to be added
					 */
					if (textPosition + unitSize < endOffset) {

						/*
						 * adding all the unit
						 */
						mergedTextUnits.append(UNIT_START_TAG_SEPARATOR);
						mergedTextUnits.append(text.getContent());
						matchEntitiesInUnit(text, legend, listEN,
								totalCharAdded, mediaUnitWTMap, 0, unitSize);
						mergedTextUnits.append(UNIT_END_TAG_SEPARATOR);
						totalCharAdded += unitCharAdded + unitSize
								+ UNIT_END_TAG_SEPARATOR.length();
						iunit++;
						textPosition += unitSize;

					} else {
						/*
						 * adding part of the last unit
						 */
						mergedTextUnits.append(UNIT_START_TAG_SEPARATOR);
						mergedTextUnits.append(text.getContent().substring(0,
								endOffset - textPosition));
						matchEntitiesInUnit(text, legend, listEN,
								totalCharAdded, mediaUnitWTMap, 0, endOffset - textPosition);
						mergedTextUnits.append(UNIT_END_TAG_SEPARATOR);
						totalCharAdded += unitCharAdded + endOffset
								- textPosition
								+ UNIT_END_TAG_SEPARATOR.length();

						/*
						 * end adding
						 */
						canAddChar = false;
						iunit++;
						textPosition += endOffset - textPosition;
					}
				}
			}
		} else if (mediaUnit instanceof Text) {
			mergedTextUnits.append(UNIT_START_TAG_SEPARATOR);
			/*
			 * getting entities on unit
			 */
			matchEntitiesInUnit(mediaUnit, legend, listEN, totalCharAdded,
					mediaUnitWTMap, startOffset, endOffset);
			/*
			 * media unit content is tagged with entities list
			 */
			String content = ((Text) mediaUnit).getContent();
			mergedTextUnits.append(content.substring(startOffset, endOffset));
			mergedTextUnits.append(UNIT_END_TAG_SEPARATOR);
			totalCharAdded += unitCharAdded + (endOffset - startOffset)
					+ UNIT_END_TAG_SEPARATOR.length();
		}
		/*
		 * adding span tag in merged text
		 */
		strBuild.append(tagContent(listEN, mergedTextUnits, legend, actionURL, resourceBundle, showToolTips));

		/*
		 * ending content tag div
		 */
		strBuild.append("</div>");
		return strBuild.toString();
		
	}

	/**
	 * Put in span tag every matched entities(of the list) in content . Class of
	 * a span corresponding entity type define in the legend.
	 * 
	 * @param listEN
	 *            list of entities to match
	 * @param content
	 *            string to analyze
	 * @return content where every named entities matched are in a span tag For
	 *         example "a text about <span
	 *         class="com_eads_weblab_annoation_gate_organization>EADS</span>"
	 */
	private static StringBuilder tagContent(List<NamedEntity> listEN,
			StringBuilder strBuf, MediaUnitLegend legend, PortletURL actionURL, ResourceBundle resourceBundle, boolean showToolTips) {
		StringBuilder strScript = new StringBuilder();
		/*
		 * TextZone building for intersections and overlaps
		 */
		List<TextZone> zoneList = buildZones(listEN);

		int overlap = 0;
		List<List<String>> overlapStyles = new ArrayList<List<String>>();
		Set<String> distinctOverlapStyles = new TreeSet<String>();
		int charAdded = 0;
		boolean containOverLap = false;

		/*
		 * Make <span> tag from TextZones
		 */
		for (TextZone tz : zoneList) {
			/*
			 * reset of action URL
			 */
			
			if (actionURL!=null) {
				actionURL.getParameterMap().remove("candidate_instance_uri");
				actionURL.getParameterMap().remove("instance_uri");
				actionURL.getParameterMap().remove("class_uri");
			}
			StringBuilder spanStart = new StringBuilder(
					"<span class='log_data ");
			/*
			 * check zone references entities
			 */
			if (tz.referenceEntities()) {
				if (!tz.isOverLap()) {
					/*
					 * no overlaped zone
					 */
					NamedEntity ne = tz.getEntityReference();
					spanStart.append(legend.getStyleContext());
					spanStart.append(legend.getStyle(ne));
					spanStart.append("'");
					if (legend.show(ne)) {
						spanStart.append(" style='background-color:rgb(");
						spanStart.append(legend.getColor(ne));
						spanStart.append("); color:");
						spanStart.append(getFontColor(legend.getColor(ne)));
						spanStart.append("'>");
					} else {
						spanStart.append("'>");
					}
					
					/*
					 * adding WebLabDocURI
					 */
					//actionURL.setParameter("document_uri", webLabDocURI);
					
					/*
					 * adding instance uri
					 */
					
					if (ne.isCandidate()) {
						if (actionURL != null) 
						actionURL.setParameter("candidate_instance_uri", ne.getEntityURI());
					} else {
						if (actionURL != null) 
						actionURL.setParameter("instance_uri", ne.getEntityURI());
					}
					
					if (actionURL != null) 
					actionURL.setParameter("class_uri", ne.getEntityClassURI());
					
					if (getFontColor(legend.getColor(ne))=="black") {
						if (ne.isCandidate()) {
							spanStart.append("<span class='tt black candidate_instance'>" );
						} else {
							spanStart.append("<span class='tt black instance'>" );
						}
					} else {
						if (ne.isCandidate()) {
							spanStart.append("<span class='tt white candidate_instance'>" );
						} else {
							spanStart.append("<span class='tt white instance'>" );
						}
					}
					
					/*
					 * insert start tags
					 */
					strBuf.insert(tz.getTextStartOffset() + charAdded,
							spanStart);
					charAdded += spanStart.length();
					/*
					 * insert end tags
					 */
					
					
					if (showToolTips) {
					 
						StringBuilder tooltip = new StringBuilder();
						if (ne.isCandidate()) {
							tooltip.append("<span class='tooltip_candidate'><span class='top'></span><span class='middle'>");
							tooltip.append(resourceBundle.getString("portlet.tooltip.candidate_instance"));
							// display label
							if (!(ne.getEntityLabel() == null || ne.getEntityLabel().equals(""))) {
								tooltip.append(resourceBundle.getString("portlet.tooltip.candidate_instance.about"));
								if (SHOW_INSTANCES_LINK) {
									tooltip.append("<a href='");
									tooltip.append(actionURL);
									tooltip.append("'>");
								}
								tooltip.append(ne.getEntityLabel()+" ");
								if (SHOW_INSTANCES_LINK) {
									tooltip.append("</a>");
								}
							}
							String typeName = legend.getDisplayName(ne,resourceBundle);
							if (typeName != null) {
								tooltip.append(resourceBundle.getString("portlet.tooltip.candidate_instance.type"));
								tooltip.append(legend.getDisplayName(ne,resourceBundle)+")");
							}
							
							
							tooltip.append("</span><span class='bottom'></span></span>");
							strBuf.insert(tz.getTextEndOffset() + charAdded, tooltip);
							charAdded += tooltip.length();
						} else {
							tooltip.append("<span class='tooltip'><span class='top'></span><span class='middle'>");
							tooltip.append(resourceBundle.getString("portlet.tooltip.instance"));
							if (SHOW_INSTANCES_LINK) {
								tooltip.append("<a href='");
								tooltip.append(actionURL);
								tooltip.append("'>");
							}
							tooltip.append(ne.getEntityLabel());
							if (SHOW_INSTANCES_LINK) {
								tooltip.append("</a>");
							}
							
							
							tooltip.append("</span><span class='bottom'></span></span>");
							strBuf.insert(tz.getTextEndOffset() + charAdded, tooltip);
							charAdded += tooltip.length();
						}
					}
					
					strBuf.insert(tz.getTextEndOffset() + charAdded, "</span></span>");
					charAdded += "</span></span>".length();
					
					
					
				} else {
					/*
					 * Overlap zone
					 */
					//System.out.println("overlaped zones...");
					containOverLap = true;
					ArrayList<String> ioverlapStyle = new ArrayList<String>();
					for (NamedEntity ne : tz.getEntitiesReferences()) {
						spanStart.append(legend.getStyleContext());
						spanStart.append(legend.getStyle(ne));
						spanStart.append(" ");
						ioverlapStyle.add(legend.getStyleContext()
								+ legend.getStyle(ne));
						distinctOverlapStyles.add(legend.getStyleContext()
								+ legend.getStyle(ne));
					}
					spanStart.append("' id='");
					spanStart.append(legend.getStyleContext());
					spanStart.append("_overlap_");
					spanStart.append(overlap);
					/*
					 * adding style for i overlap
					 */
					overlapStyles.add(overlap, ioverlapStyle);
					/*
					 * add an overlap count
					 */
					overlap++;
					spanStart.append("'>");

					
					/*
					 * adding WebLabDocURI
					 */
					//actionURL.setParameter("document_uri", webLabDocURI);
					
					/*
					 * tooltip
					 */
					
					spanStart.append("<span class='tt black instance'>" );
					/*
					 * insert start tags
					 */
					strBuf.insert(tz.getTextStartOffset() + charAdded,
							spanStart);
					charAdded += spanStart.length();
					
					if (showToolTips) {
					
						StringBuilder tooltip = new StringBuilder("<span class='tooltip_multiple'><span class='top'></span><span class='middle'>");
						tooltip.append(resourceBundle.getString("portlet.tooltip.multiple_instances"));
							
							
						for (NamedEntity ne : tz.getEntitiesReferences()) {
							
							if (actionURL != null ){
								/*
								 * reset of action URL
								 */
								actionURL.getParameterMap().remove("candidate_instance_uri");
								actionURL.getParameterMap().remove("instance_uri");
								actionURL.getParameterMap().remove("class_uri");
							}
							if (ne.isCandidate()) {
								tooltip.append("<li>");
								
								if (SHOW_INSTANCES_LINK) {
									if (actionURL != null) {
										actionURL.setParameter("candidate_instance_uri", ne.getEntityURI());
										actionURL.setParameter("class_uri", ne.getEntityClassURI());
									}
									tooltip.append("<a href='");
									tooltip.append(actionURL);
									tooltip.append("'>");
								}
								
								if (SHOW_INSTANCES_LINK) {
									if (!(ne.getEntityLabel()==null || ne.getEntityLabel().equals(""))) {
										tooltip.append(ne.getEntityLabel());
									} else {
										tooltip.append(ne.getEntityURI());
									}
									tooltip.append("</a>&nbsp;");
								} else {
									if (!(ne.getEntityLabel()==null || ne.getEntityLabel().equals(""))) {
										tooltip.append(ne.getEntityLabel());
									} else {
										tooltip.append(ne.getEntityURI());
									}
								}
								
								
								//tooltip.append(resourceBundle.getString(ne.getEntityClassURI()));
								
								String typeName = legend.getDisplayName(ne,resourceBundle);
								if (typeName != null) {
									tooltip.append(resourceBundle.getString("portlet.tooltip.candidate_instance.type"));
									tooltip.append(legend.getDisplayName(ne,resourceBundle)+")");
								}
								
								//tooltip.append(resourceBundle.getString("portlet.tooltip.multiple_candidate_instance"));
								tooltip.append("</li>");
							} else {
								tooltip.append("<li>");
								if (SHOW_INSTANCES_LINK) {
									if (actionURL != null) {
										actionURL.setParameter("instance_uri", ne.getEntityURI());
										actionURL.setParameter("class_uri", ne.getEntityClassURI());
									}
									tooltip.append("<a href='");
									tooltip.append(actionURL);
									tooltip.append("'>");
								}
								tooltip.append(ne.getEntityLabel());
								
								if (SHOW_INSTANCES_LINK) {
									if (!(ne.getEntityLabel()==null || ne.getEntityLabel().equals(""))) {
										tooltip.append(ne.getEntityLabel());
									} else {
										tooltip.append(ne.getEntityURI());
									}
									tooltip.append("</a>&nbsp;");
								} else {
									if (!(ne.getEntityLabel()==null || ne.getEntityLabel().equals(""))) {
										tooltip.append(ne.getEntityLabel());
									} else {
										tooltip.append(ne.getEntityURI());
									}
								}
								tooltip.append(resourceBundle.getString(ne.getEntityClassURI()));
								tooltip.append(resourceBundle.getString("portlet.tooltip.multiple_instance"));
								tooltip.append("</li>");
							}
						}
							
						tooltip.append("</span><span class='bottom'></span></span>");
							
						strBuf.insert(tz.getTextEndOffset() + charAdded, tooltip);	
						charAdded += tooltip.length();
					}
					
					/*
					 * insert end tags
					 */
					strBuf.insert(tz.getTextEndOffset() + charAdded, "</span></span></span>");
					charAdded += "</span></span></span>".length();
				}
			}
		}
		/*
		 * Make Javascript in overlap case, using jQuery framework
		 */
		if (containOverLap) {
			LogFactory.getLog(MediaUnitHighLighter.class).info("generating Javascript...");

			/*
			 * adding INLINE tag
			 */
			strScript.append("<SCRIPT language='JavaScript'>");

			Map<String, String> colors = legend.getColorMap();
			Map<String, String> styles = legend.getStyleMap();
			Map<String, Boolean> check = legend.getCheckedMap();
			String styleContext = legend.getStyleContext();

			/*
			 * make color and check tabs correponding the legend
			 */
			strScript
					.append("jQuery.colorsTab = new Array();\n");
			strScript
					.append("jQuery.colorsCheckTab = new Array();\n");

			/*
			 * push color and check in tabs
			 */
			for (Object key : styles.keySet()) {
				strScript.append("jQuery.colorsTab['");
				strScript.append(styleContext);
				strScript.append(styles.get(key));
				strScript.append("']=[");
				strScript.append(colors.get(key));
				strScript.append("];\n");
				strScript.append("jQuery.colorsCheckTab['");
				strScript.append(styleContext);
				strScript.append(styles.get(key));
				strScript.append("']=");
				strScript.append(check.get(key));
				strScript.append(";\n");
			}

			/*
			 * make overlap styles tab
			 */
			strScript
					.append("jQuery.overlapStyles = new Array();\n");
			int ilapstyle = 0;
			for (List<String> overlapStyle : overlapStyles) {
				strScript.append("jQuery.overlap_");
				strScript.append(ilapstyle);
				strScript.append("= new Array();\n");
				int jlapstyle = 0;
				for (String style : overlapStyle) {
					strScript.append("jQuery.overlap_");
					strScript.append(ilapstyle);
					strScript.append("[");
					strScript.append(jlapstyle);
					strScript.append("]='");
					strScript.append(style);
					strScript.append("';\n");
					jlapstyle++;
				}
				strScript.append("jQuery.overlapStyles[");
				strScript.append(ilapstyle);
				strScript.append("]=jQuery.overlap_");
				strScript.append(ilapstyle);
				strScript.append(";\n");
				ilapstyle++;
			}
			/*
			 * creating style list corresponding overlap styles
			 */
			strScript
					.append("jQuery.distinctOverlapStyles = new Array();\n");
			strScript
					.append("jQuery.distinctOverlapStylesCount=0;\n");
			for (String distinctStyle : distinctOverlapStyles) {
				strScript
						.append("jQuery.distinctOverlapStyles[jQuery.distinctOverlapStylesCount]='");
				strScript.append(distinctStyle);
				strScript.append("';\n");
				strScript
						.append("jQuery.distinctOverlapStylesCount++;\n");
			}

			/*
			 * make script to change overlaps background color at interval
			 */
			strScript.append("var curCount = 0;\n");
			strScript.append("function changeOverlapColor() {\n");
			strScript
					.append("var curOverLapStyle=jQuery.distinctOverlapStyles[curCount];\n");
			strScript.append("for (var i=0;i<");
			strScript.append(overlap);
			strScript.append(";i++) {\n");
			strScript.append("var curLapStyles = new Array();\n");
			strScript
					.append("curLapStyles = jQuery.overlapStyles[i];\n");
			strScript.append("var noOverlapChecked=true;\n");
			strScript
					.append("for (var k=0;(k<curLapStyles.length && noOverlapChecked);k++) {\n");
			strScript.append("kOverLapStyle=curLapStyles[k];");
			strScript
					.append("if (jQuery.colorsCheckTab[kOverLapStyle]) {\n");
			strScript.append("noOverlapChecked=false;\n");
			strScript.append("}");
			strScript.append("}\n");
			strScript.append("if (noOverlapChecked) {\n");
			strScript.append("var e = jQuery(\"");
			strScript.append(styleContext);
			strScript.append("_overlap_\"+i);\n");
			strScript.append("e.css('background','transparent');\n");
			strScript.append("e.css('color',document.fgColor);\n");
			strScript.append("} else {\n");
			strScript.append("var containOverlap = false;");
			strScript
					.append("for (var j=0; (j<curLapStyles.length && !containOverlap); j++) {");
			strScript.append("if (curLapStyles[j]==curOverLapStyle) {");
			strScript.append("containOverlap=true;");
			strScript.append("}");
			strScript.append("}");
			strScript.append("if(containOverlap){\n");
			strScript
					.append("if(jQuery.colorsCheckTab[curOverLapStyle]) {\n");
			strScript
					.append("var overcolor =  jQuery.colorsTab[curOverLapStyle];\n");
			strScript.append("var e = jQuery(\"#");
			strScript.append(styleContext);
			strScript.append("_overlap_\"+i);\n");
			strScript.append("e.css('background','rgb('+overcolor+')');\n");
			strScript
					.append("e.css('color',jQuery.ColorPickerManager.getFontColor(overcolor));\n");
			strScript.append("}\n");
			strScript.append("}\n");
			strScript.append("}\n");
			strScript.append("}\n");
			strScript
					.append("if(curCount==jQuery.distinctOverlapStylesCount-1) {\n");
			strScript.append("curCount = 0;\n");
			strScript.append("} else {\n");
			strScript.append("curCount++;\n");
			strScript.append("}");
			strScript.append("}\n;");
			strScript.append("var changeoverlaptimer;");
			strScript.append("clearInterval(changeoverlaptimer);");
			strScript.append("changeoverlaptimer = setInterval(\"changeOverlapColor()\",");
			strScript.append(HIGHLIGHTING_TIMEOUT);
			strScript.append(");\n");
			strScript.append("</SCRIPT>");
			LogFactory.getLog(MediaUnitHighLighter.class).debug("generating Javascript ok");
		}
		/*
		 * append document text to script text, replacing backslash to <br> tag
		 */
		strScript.append(strBuf.toString().replace("\n", "<br>"));
		return strScript;
	}

	/**
	 * Check for segment intersection or inclusion and create new entities to
	 * decompose or fuse intersection/inclusion
	 * 
	 * @param listeEN
	 *            sorted entity List.
	 * @return
	 */
	private static List<TextZone> buildZones(List<NamedEntity> listeEN) {
		Set<Integer> listOffset = new TreeSet<Integer>();
		ArrayList<TextZone> zoneList = new ArrayList<TextZone>();

		for (NamedEntity ne : listeEN) {
			listOffset.add(ne.getStartOffset());
			listOffset.add(ne.getEndOffset());
		}

		int start = -1;
		int end = -1;
		Iterator<Integer> it = listOffset.iterator();

		if (it.hasNext()) {
			start = it.next();
			while (it.hasNext()) {
				end = it.next();
				TextZone tz = new TextZone(start, end);
				zoneList.add(tz);
				start = end;
			}
		}

		for (TextZone tz : zoneList) {
			for (NamedEntity ne : listeEN) {
				if (tz.reference(ne)) {
					tz.getEntitiesReferences().add(ne);
				}
			}
		}
		return zoneList;
	}

	/**
	 * return the font color for the selectedColor
	 * 
	 * @param selectedColor
	 *            Color in r,g,b representation
	 * @return the font color corresponding "black" or "white"
	 */
	private static String getFontColor(String selectedColor) {

		String[] tabColor = selectedColor.split(",");
		int color = Integer.parseInt(tabColor[0])
				+ Integer.parseInt(tabColor[1]) + Integer.parseInt(tabColor[2]);
		if (color > 3 * 128) {
			return "black";
		}
		return "white";
	}

	/**
	 * Process a Text unit to find all named entities on it annotated segment.
	 * Matching is based on the document legend.
	 * 
	 * @param out
	 * @param unit
	 * @param legend
	 * @param listEN
	 */
	@SuppressWarnings("unchecked")
	private static void matchEntitiesInUnit(MediaUnit unit,
			MediaUnitLegend legend, List<NamedEntity> listEN,
			int totalCharAdded, Statements mediaUnitWTmap, int startOffset, int endOffset) {
		
		
		
		Set<String> legendPredicates = legend.getPredicats();
		HashMap<String, List<String>> eqClasses = new HashMap<String, List<String>>();
		String[]predicatList = new String[legendPredicates.size()];
		
		int i=0;
		for (String p : legendPredicates) {
			predicatList[i]= p;
			i++;
		}
		
		for (LegendField f : legend.getFields()) {
			eqClasses.put(f.getEntityType(), f.getEqClasses());
		}
			
		//RDFSelector RDFSel;
		List<Segment> listSeg = unit.getSegment();
		/*
		 * sort segments
		 */
		Collections.sort(listSeg, new SegmentComparator());

		/*
		 * keep segments about offset text
		 */
		listSeg = keepSegmentInRange(listSeg, startOffset, endOffset);

		/*
		 * Processing all linear segments
		 */
		for (int j = 0; j < listSeg.size(); j++) {
			LinearSegment linSeg = (LinearSegment) listSeg.get(j);
			
			/*
			 * looking for named entities 
			 */
			//RDFSel = RDFSelectorFactory.getSelector(linSeg.getUri());
			
			//get annots for this segment
			IPredicateValuePairs map = mediaUnitWTmap.get(linSeg.getUri());
			if (map != null && (map.getValue(WebLabProcessing.REFERS_TO) != null)) {
				
				//linear segments with refersTo annot
				if (map.getValue(WebLabProcessing.REFERS_TO) instanceof List<?>) {
					//multiple references
					for (String instance : (List<String>)map.getValue(WebLabProcessing.REFERS_TO)) {
						for (String pred : predicatList) {
							if (mediaUnitWTmap.get(instance).getValue(RDF.TYPE).equals(pred) || ((eqClasses.get(pred)!=null) && (eqClasses.get(pred).contains(mediaUnitWTmap.get(instance).getValue(RDF.TYPE))))) {
								/*
								 * creating a new named entity
								 */
								boolean iscandidate = true;
								String label="";
								//is canditate
								if (mediaUnitWTmap.get(instance).getValue(WebLabProcessing.IS_CANDIDATE)!=null) {
									iscandidate = new Boolean(mediaUnitWTmap.get(instance).getValue(WebLabProcessing.IS_CANDIDATE).toString());
								}
							
								//label
								if (mediaUnitWTmap.get(instance).getValue(RDFS.LABEL)!=null) {
									label = mediaUnitWTmap.get(instance).getValue(RDFS.LABEL).toString();
								}
								
								/*
								 * adding entity to list
								 */
								listEN.add(new NamedEntity(linSeg.getStart()
										+ UNIT_START_TAG_SEPARATOR.length()
										+ totalCharAdded - startOffset, linSeg
										.getEnd()
										+ UNIT_START_TAG_SEPARATOR.length()
										+ totalCharAdded - startOffset, label,
										instance, pred, iscandidate));
							}
						}
					}
				} else {
					String instance = (String) map.getValue(WebLabProcessing.REFERS_TO);
					//simple reference
					for (String pred : predicatList) {
						
						
						if (mediaUnitWTmap.get(instance).getValue(RDF.TYPE).equals(pred) || ((eqClasses.get(pred)!=null) && (eqClasses.get(pred).contains(mediaUnitWTmap.get(instance).getValue(RDF.TYPE))))) {
						/*
						 * adding a new named entity
						 */
							boolean iscandidate = true;
							String label="";
							
							//is candidate
							if (mediaUnitWTmap.get(instance).getValue(WebLabProcessing.IS_CANDIDATE)!=null) {
								iscandidate = new Boolean(mediaUnitWTmap.get(instance).getValue(WebLabProcessing.IS_CANDIDATE).toString());
							}
						
							//label
							if (mediaUnitWTmap.get(instance).getValue(RDFS.LABEL)!=null) {
								label = mediaUnitWTmap.get(instance).getValue(RDFS.LABEL).toString();
							}
							
							/*
							 * adding entity to list	
							 */
							listEN.add(new NamedEntity(linSeg.getStart()
									+ UNIT_START_TAG_SEPARATOR.length()
									+ totalCharAdded - startOffset, linSeg
									.getEnd()
									+ UNIT_START_TAG_SEPARATOR.length()
									+ totalCharAdded - startOffset, label,
									instance, pred, iscandidate));
						}
					}
				}
			}
		}
	}

	/**
	 * Filter segments to keep segments in a offset range.
	 * 
	 * @param segList
	 *            the list of segment to filter
	 * @param startOffset
	 *            start offset of text in media unit
	 * @param endOffset
	 *            end offset of text in media unit
	 * @return the list of segments inclued in the offset range.
	 */
	private static List<Segment> keepSegmentInRange(List<Segment> segList,
			int startOffset, int endOffset) {
		LinearSegment range = new LinearSegment();
		range.setStart(startOffset);
		range.setEnd(endOffset);

		
		LinkedList<Segment> ret = new LinkedList<Segment>();
		Collections.sort(segList, new SegmentComparator());
		
		if (segList.size() > 0) {
			int max = segList.size();
			int cur = 0;
			boolean stop = false;
			while (cur < max && !stop) {
				if (((LinearSegment)segList.get(cur)).getStart()<((LinearSegment)segList.get(cur)).getEnd()) {
					if (SegmentUtil.isIncluded(segList.get(cur), range)) {
						ret.add(segList.get(cur));
					}
					if (((LinearSegment)segList.get(cur)).getStart()>=endOffset) {
						stop=true;
					}
				}
				cur++;
			}
		}
		return ret;
	}
}