/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 *
 * Copyright (C) 2004 - 2009 EADS DEFENCE AND SECURITY SYSTEMS
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of the
 * GNU Lesser General Public License as published by the Free Software Foundation; either version
 * 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See
 * the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.portlet.tool;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.ResourceBundle;
import java.util.Set;
import java.util.TreeSet;

import org.ow2.weblab.portlet.bean.LegendField;
import org.ow2.weblab.portlet.bean.NamedEntity;

/**
 * Class representing an editable Media Unit legend and get it HTML
 * representation. The legend contains a set of fields. For each field,
 * corresponding a color and a name. Legend can be modified with HTML browser
 * using a colorpicker. modification on legend can be retrieved, using a HTML
 * form.
 * 
 * @author emilien
 * 
 */
public class MediaUnitLegend {

	public static String SAVE_LEGEND_SUBMIT = "save_legend";
	public static String LEGEND_COLOR_INPUT = "hidden_color_";
	public static String LEGEND_CHECKBOX_INPUT = "_showCheckBox";
	public static String LEGEND_CHECKBOX_COLOR = "checkbox_color_";
	private String appContext;
	private String styleContext;
	private String actionURL;
	private List<LegendField> fields;
	private Map<String, String> colorMap;
	private Map<String, String> styleMap;
	private Set<String> predicats;
	private Map<String, TreeSet<String>> predicatValuesMap;
	private Map<String, Boolean> checkedMap;

	/**
	 * Public constructor without parameters.
	 */
	public MediaUnitLegend() {
		this.colorMap = new HashMap<String, String>();
		this.styleMap = new HashMap<String, String>();
		this.checkedMap = new HashMap<String, Boolean>();
		this.predicats = new TreeSet<String>();
		this.predicatValuesMap = new HashMap<String, TreeSet<String>>();
	}

	/**
	 * Public constructor taking in parameters
	 * 
	 * @param fields
	 *            the list of fields of the legend
	 * @param appContext
	 *            the application context used
	 * @param styleContext
	 *            the HTML element class root name
	 */
	public MediaUnitLegend(List<LegendField> fields, String appContext,
			String styleContext) {
		this.fields = fields;
		this.appContext = appContext;
		this.styleContext = styleContext;
	}

	/**
	 * Method used to get a HTML representation of the legend, where colors are
	 * editable or not throw a JavaScript color picker .
	 * 
	 * @param editable
	 *            boolean allowing color edition
	 * @return
	 */
	public String getHTMLLegend(boolean editable, ResourceBundle resourceBundle) {
		/*
		 * start legend div
		 */
		StringBuilder strBuf = new StringBuilder("<div id='");
		strBuf.append(this.styleContext);
		strBuf.append("_div_legend'>");
		//strBuf.append("<script type='text/javascript'> jQuery( function() { var colorPicker = new Liferay.ColorPicker( { hasImage: true, onClose: function(){console.log(this); }} ); } ); </script>");
		strBuf.append("<fieldset title='Legend'");
		strBuf.append(" id='");
		strBuf.append(this.styleContext);
		strBuf.append("_div_legend_fieldset'");
		strBuf.append(" class='");
		strBuf.append(this.styleContext);
		strBuf.append("_div_legend_fieldset'>");
		strBuf.append("<legend>"+resourceBundle.getString("portlet.legend.title")+"</legend>");
		strBuf.append("<form id='");
		strBuf.append(this.styleContext);
		strBuf.append("_div_legend_fieldset_form'");
		strBuf.append("method='post' action='");
		strBuf.append(this.actionURL);
		strBuf.append("'>");
		strBuf.append("<table class='");
		strBuf.append(this.styleContext);
		strBuf.append("_div_legend_fieldset_form_table'>");

		/*
		 * additionnal field for instance canditate 
		 */
		strBuf.append("<tr><td><div class='instance_candidate_legend_box'><div> </div></div></td><td>&nbsp;"+resourceBundle.getString("portlet.legend.instance_candidate")+"</tr><br>");
		
		/*
		 * processing fields
		 */
		int i = 0;
		for (LegendField f : this.fields) {
			if (i%3 == 0) {
				strBuf.append("<tr>");
			}
			
			strBuf.append("<td style='width=20'>");
			strBuf.append("<div class='");
			strBuf.append(this.styleContext);
			strBuf.append("_div_legend_box'>");
			strBuf.append("<div class='");
			strBuf.append(this.styleContext);
			strBuf.append(f.getStyleClassName());
			strBuf.append("_legend'");
			strBuf.append(" style='background-color:rgb(");
			strBuf.append(f.getStyleColor());
			strBuf.append(")' >&nbsp;");
			strBuf.append("</div>");
			strBuf.append("</div>");
			strBuf.append("</td>");
			strBuf.append("<td class='");
			strBuf.append(this.styleContext);
			strBuf.append("_entityType'>&nbsp;");
			if (resourceBundle.containsKey(f.getDisplayName())) {
				strBuf.append(resourceBundle.getString(f.getDisplayName()));
			} else {
				strBuf.append(f.getDisplayName());
			}
			strBuf.append("</td>");

			if (editable) {
				/*
				 * edit mode, adding color picker & checkbox
				 */
				strBuf.append("<td>");
				strBuf.append("<img src='");
				strBuf.append(this.appContext);
				strBuf.append("/js/images/color.png' ");
				strBuf.append("width='16' alt='Change color' height='16'");
				strBuf
						.append(" onclick=\"jQuery.ColorPickerManager.OpenColorPicker2(");
				strBuf.append("'");
				strBuf.append(this.styleContext);
				strBuf.append(f.getStyleClassName());
				strBuf.append("',event)\" ");
				strBuf.append(" id='");
				strBuf.append(this.styleContext);
				strBuf.append(f.getStyleClassName());
				strBuf.append("picker'");
				strBuf.append(" class='use-colorpicker'/>");
				strBuf.append("</td>");
				strBuf.append("<td>");
				strBuf.append("&nbsp;&nbsp;");
				strBuf.append("<input type='checkbox' ");
				strBuf.append("id='");
				strBuf.append(this.styleContext);
				strBuf.append(f.getStyleClassName());
				strBuf.append(LEGEND_CHECKBOX_INPUT);
				strBuf
						.append("' onclick=\"jQuery.ColorPickerManager.ShowEntity('");
				strBuf.append(this.styleContext);
				strBuf.append(f.getStyleClassName());
				strBuf.append("')\" ");
				strBuf.append("name='");
				strBuf.append(LEGEND_CHECKBOX_COLOR);
				strBuf.append(this.styleContext);
				strBuf.append(f.getStyleClassName());
				strBuf.append("'");
				strBuf.append("value='yes' ");
				if (f.isShow()) {
					strBuf.append("checked='true'");
				}
				strBuf.append(" >");
				strBuf.append("Show");
				strBuf.append("</input>");
				strBuf.append("<input type='hidden'");
				strBuf.append("id='");
				strBuf.append(LEGEND_COLOR_INPUT);
				strBuf.append(this.styleContext);
				strBuf.append(f.getStyleClassName());
				strBuf.append("' ");
				strBuf.append("value='");
				strBuf.append(f.getStyleColor());
				strBuf.append("'");
				strBuf.append("name='");
				strBuf.append(LEGEND_COLOR_INPUT);
				strBuf.append(this.styleContext);
				strBuf.append(f.getStyleClassName());
				strBuf.append("' >");
				strBuf.append("</td>");
			}
			if (i%3 == 2) {
				strBuf.append("</tr>");
			}
			i++;
		}
		strBuf.append("</table>");
		if (editable) {
			/*
			 * adding form button
			 */
			strBuf.append("<input name='");
			strBuf.append(SAVE_LEGEND_SUBMIT);
			strBuf.append("'");
			strBuf.append("type='submit'");
			strBuf.append("value='Save'>");
		}
		/*
		 * end legend div
		 */
		
		strBuf.append("</form>");
		strBuf.append("</fieldset>");
		strBuf.append("</div>");
		return strBuf.toString();
	}

	/**
	 * Method to get the color of a typed valued entity in the legend. Value may
	 * be null.
	 * 
	 * @param entityType
	 *            entity type (For example
	 *            "http://weblab.eads.com/property/gate/annotation")
	 * @param matchedValue
	 *            (For example "Person")
	 * @return the color of the typed valued entity in the legend.
	 */
	public String getColor(NamedEntity ne) {
		StringBuilder color = new StringBuilder();
//		if (ne.getEntityLabel() == null) {
			color.append(this.colorMap.get(ne.getEntityClassURI()));
//		} else {
//			color.append(this.colorMap.get(ne.getEntityURI() + "#"
//					+ ne.getEntityLabel()));
//		}
		return color.toString();
	}

	/**
	 * Method to get the css style name of an named entity.
	 * 
	 * @param ne
	 *            the named entity
	 * @return the css style name of named entity for this legend.
	 */
	public String getStyle(NamedEntity ne) {
//		if (ne.getEntityLabel() != null) {
//			return this.styleMap.get(ne.getEntityClassURI() + "#"
//					+ ne.getEntityLabel());
//		}
		return this.styleMap.get(ne.getEntityClassURI());
	}

	/**
	 * Method to get visibility of the named entity. Corresponding field is
	 * always visible in legend.
	 * 
	 * @param ne
	 *            the named entity
	 * @return true if named entity checked, false otherwise.
	 */
	public boolean show(NamedEntity ne) {
		
		//System.out.println("ne :"+ne.getEntityClassURI());
		
		//System.out.println("legend map"+checkedMap);
//		if (ne.getEntityLabel() != null) {
//			return this.checkedMap.get(ne.getEntityURI() + "#"
//					+ ne.getEntityLabel());
//		}
		return this.checkedMap.get(ne.getEntityClassURI());
	}

	/**
	 * 
	 */
	public Set<String> getPredicats() {
		return this.predicats;
	}

	/**
	 * Get values corresponding a predicate in the legend.
	 * 
	 * @param predicat
	 *            the predicate.
	 * @return a set of values for predicat.
	 */
	public Set<String> getMachedValuesOnPredicat(String predicat) {
		return this.predicatValuesMap.get(predicat);
	}

	/**
	 * Private method to put in maps : color of entity, distinct predicates and
	 * it value.
	 */
	private void fieldsToMap() {
		for (LegendField field : this.fields) {
			String key = field.getEntityType();
			String matchedValue = field.getMatchedValue();
			String color = field.getStyleColor();
			String style = field.getStyleClassName();
			boolean check = field.isShow();
			if (this.predicatValuesMap.get(key) == null) {
				this.predicatValuesMap.put(key, new TreeSet<String>());
			}

			if (matchedValue != null) {
				this.colorMap.put(key + "#" + matchedValue, color);
				this.styleMap.put(key + "#" + matchedValue, style);
				this.checkedMap.put(key + "#" + matchedValue, check);
				this.predicatValuesMap.get(key).add(matchedValue);
			} else {
				this.colorMap.put(key, color);
				this.styleMap.put(key, style);
				this.checkedMap.put(key, check);
			}
			this.predicats.add(key);
		}
	}

	/**
	 * To get the color of the field in the HTML form.
	 * 
	 * @param field
	 * @return the color input name of the field in form
	 */
	public String getColorInputName(LegendField field) {
		return LEGEND_COLOR_INPUT + this.styleContext
				+ field.getStyleClassName();
	}

	/**
	 * To get the checkbox value of the field in the HTML form.
	 * 
	 * @param field
	 * @return the checkbox input name of the field in form
	 */
	public String getCheckBoxInputName(LegendField field) {
		return LEGEND_CHECKBOX_COLOR + this.styleContext
				+ field.getStyleClassName();
	}
	
	
	/**
	 * to get the display class name for the entity
	 */
	public String getDisplayName(NamedEntity ne, ResourceBundle resourceBundle) {
		String classURI = ne.getEntityClassURI();
		for (LegendField field : fields){
			if (field.getEntityType().equals(classURI)) {
				return resourceBundle.getString(field.getDisplayName());
			}
		}
		return null;
	}

	/**
	 * Getters and setters
	 */
	public String getAppContext() {
		return this.appContext;
	}

	public void setAppContext(String appContext) {
		this.appContext = appContext;
	}

	public String getStyleContext() {
		return this.styleContext;
	}

	public void setStyleContext(String styleContext) {
		this.styleContext = styleContext;
	}

	public List<LegendField> getFields() {
		return this.fields;
	}

	public void setFields(List<LegendField> fields) {
		this.fields = fields;
		/*
		 * Update color map
		 */
		fieldsToMap();
	}

	public String getActionURL() {
		return this.actionURL;
	}

	public void setActionURL(String actionURL) {
		this.actionURL = actionURL;
	}

	public void updateShemColor() {
		fieldsToMap();
	}

	public Map<String, String> getColorMap() {
		return this.colorMap;
	}

	public void setColorMap(Map<String, String> colorMap) {
		this.colorMap = colorMap;
	}

	public Map<String, String> getStyleMap() {
		return this.styleMap;
	}

	public void setStyleMap(Map<String, String> styleMap) {
		this.styleMap = styleMap;
	}

	public Map<String, Boolean> getCheckedMap() {
		return this.checkedMap;
	}

	public void setCheckedMap(Map<String, Boolean> checkedMap) {
		this.checkedMap = checkedMap;
	}
}