/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 *
 * Copyright (C) 2004 - 2009 EADS DEFENCE AND SECURITY SYSTEMS
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of the
 * GNU Lesser General Public License as published by the Free Software Foundation; either version
 * 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See
 * the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.portlet;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Date;
import java.util.HashMap;
import java.util.ResourceBundle;

import javax.portlet.ActionRequest;
import javax.portlet.ActionResponse;
import javax.portlet.EventRequest;
import javax.portlet.EventResponse;
import javax.portlet.PortletException;
import javax.portlet.PortletMode;
import javax.portlet.PortletRequestDispatcher;
import javax.portlet.PortletSession;
import javax.portlet.PortletURL;
import javax.portlet.RenderRequest;
import javax.portlet.RenderResponse;
import javax.portlet.ResourceRequest;
import javax.portlet.ResourceResponse;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.ow2.weblab.core.extended.exception.WebLabCheckedException;
import org.ow2.weblab.core.extended.factory.ResourceFactory;
import org.ow2.weblab.core.extended.ontologies.DublinCore;
import org.ow2.weblab.core.extended.ontologies.RDF;
import org.ow2.weblab.core.extended.ontologies.RDFS;
import org.ow2.weblab.core.extended.ontologies.WebLabProcessing;
import org.ow2.weblab.core.extended.ontologies.WebLabRetrieval;
import org.ow2.weblab.core.extended.util.ResourceUtil;
import org.ow2.weblab.core.helper.PoKHelper;
import org.ow2.weblab.core.helper.impl.AdvancedSelector;
import org.ow2.weblab.core.helper.impl.JenaPoKHelper;
import org.ow2.weblab.core.helper.impl.RDFSelectorFactory;
import org.ow2.weblab.core.helper.impl.Statements;
import org.ow2.weblab.core.model.Document;
import org.ow2.weblab.core.model.MediaUnit;
import org.ow2.weblab.core.model.PieceOfKnowledge;
import org.ow2.weblab.core.model.Resource;
import org.ow2.weblab.core.model.Text;
import org.ow2.weblab.core.services.ResourceContainer;
import org.ow2.weblab.core.services.resourcecontainer.LoadResourceArgs;
import org.ow2.weblab.portlet.bean.LegendField;
import org.ow2.weblab.portlet.tool.LegendFactory;
import org.ow2.weblab.portlet.tool.MediaUnitHighLighter;
import org.ow2.weblab.portlet.tool.MediaUnitLegend;
import org.ow2.weblab.portlet.tool.MediaUnitSplitter;
import org.ow2.weblab.portlet.tool.RepoServiceConfigBean;
import org.ow2.weblab.portlet.tool.ResourceContainerUtil;
import org.springframework.beans.factory.BeanFactory;
import org.springframework.beans.factory.xml.XmlBeanFactory;
import org.springframework.core.io.ClassPathResource;


/**
 * Portlet class to view annotations of a Weblab Document. This portlet process
 * events containing a PieceOfKnowledge about the document to show. It
 * differentiate candidate and validated instances and send events on instances
 * selection (containing PieceOfKnowledge about the selected instance).
 * 
 * @author Emilien Bondu
 * 
 */
public class DocViewPortlet extends WebLabPortlet {

	// Configurable fields
	/**
	 * Default number of characters displayed per page
	 */
	private static final int DEFAULT_TEXT_SIZE = 10000;

	/**
	 * Event send when adding document to a basket (see portlet.xml for mapping)
	 */
	private static final String ADD_TO_BASKET_ACTION = "addToBasketSelection";

	/**
	 * Use the default default if not set in portlet.xml
	 */
	private static boolean USE_DEFAULT_LEGEND = true;

	/**
	 * Use the redirection after instance selection if not set in portlet.xml
	 */
	private static boolean USE_REDIRECTION = false;
	
	/**
	 * Namespace property used in PieceOkKnowledge to get the Document to
	 * display URI
	 */
	private static final String POK_PROPERTY_DOC_URI = WebLabRetrieval.IS_LINKED_TO;

	// Private fields used for session
	/**
	 * User's media unit identifier in session
	 */
	private final static String USER_MEDIA_UNIT = "user_media_unit";

	private static final String USER_MEDIA_UNIT_WTMAP = "user_mediaunit_triplemap";
	/**
	 * User's legend identifier in session
	 */
	private final static String USER_LEGEND = "user_legend";

	/**
	 * Buttons for external events
	 */
	private static boolean SHOW_ADVANCED_FUNCTIONS = true;
	
	/**
	 * Link on candidate instances
	 */
	private static boolean SHOW_INSTANCES_LINK = true;
	
	/**
	 * Link on candidate instances
	 */
	private static boolean SHOW_TOOLTIPS = true;

	/**
	 * User's ResultSetSplitter identifier in session
	 */
	private final static String USER_SPLITTER = "user_spliter";

	/**
	 * User's Repository configuration identifier in session
	 */
	private final static String USER_REPO_CONF = "user_repo_service_conf";

	/**
	 * Variable identifier in session to know if the user's current document is
	 * loaded or not
	 */
	private final static String CURRENT_RESSOURCE_IS_LOADED = "user_resource_is_loaded";

	// Private fields for request parameters identifier
	/**
	 * Error request parameter name
	 */
	private final static String ERROR = "message_error";

	// Private class objects

	/**
	 * Context used for css. Must match with css file.
	 */
	private final static String CSS_STYLE_CONTEXT = "doc_view_portlet";

	private static final String USER_RES_ANNOTS_LOADED = "user_wtriple_map_isloaded";

	private static final String SIMILARITY_SEARCH_ACTION = "searchSimilarResources";

	

	/**
	 * Map of ResourceContainer client
	 */
	private HashMap<URL, ResourceContainer> repoMap;

	/**
	 * URL of the repository WSDL
	 */
	private static URL WSDL_REPO_URL;

	/**
	 * Default legend
	 */
	private static MediaUnitLegend default_legend;

	/**
	 * Default Respository service configuration bean
	 */
	private static RepoServiceConfigBean default_serv_conf;

	/**
	 * RDFSelector used by the portlet
	 */
	private AdvancedSelector rdfSelector;

	/**
	 * Logger
	 */
	private Log logger;

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.portlet.GenericPortlet#init()
	 */
	@Override
	public void init() throws PortletException {
		/*
		 * init WebLab portlet for event mapping
		 */
		super.init();
		default_serv_conf = new RepoServiceConfigBean();
		default_serv_conf.setServiceURL(getInitParameter("repo_service_url"));
		this.repoMap = new HashMap<URL, ResourceContainer>();

		/*
		 * initializing logger
		 */
		logger = LogFactory.getLog(this.getClass());

		/*
		 * getting repository / search WSDL URL
		 */
		try {
			WSDL_REPO_URL = new File(getPortletContext().getRealPath(
					"WEB-INF/classes/services/WebLab.wsdl")).toURI()
					.toURL();
		} catch (MalformedURLException e) {
			logger.error(e);
		}

		/*
		 * setting the redirect parameter
		 */
		if (getInitParameter("use_redirection") != null) {
			USE_REDIRECTION = Boolean
					.parseBoolean(getInitParameter("use_redirection"));
		}

		/*
		 * setting the default legend
		 */
		if (getInitParameter("use_default_legend") != null) {
			USE_DEFAULT_LEGEND = Boolean
					.parseBoolean(getInitParameter("use_default_legend"));
		}
		if (USE_DEFAULT_LEGEND) {
			ClassPathResource resource = new ClassPathResource(getInitParameter("default_legend_bean"));
			BeanFactory factory = new XmlBeanFactory(resource);

			default_legend = factory.getBean(
					getInitParameter("default_legend_bean_id"),
					MediaUnitLegend.class);
		}

		if (getInitParameter("show_instances_link") != null) {
			SHOW_INSTANCES_LINK = Boolean
					.parseBoolean(getInitParameter("show_instances_link"));
		}
		
		if (getInitParameter("show_tooltips") != null) {
			SHOW_TOOLTIPS = Boolean
					.parseBoolean(getInitParameter("show_tooltips"));
		}
		
		if (getInitParameter("show_advanced_functions") != null) {
			SHOW_ADVANCED_FUNCTIONS = Boolean
					.parseBoolean(getInitParameter("show_advanced_functions"));
		}
		
		/*
		 * initializing the rdfselector
		 */
		this.rdfSelector = RDFSelectorFactory.getSelector(true);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.portlet.GenericPortlet#destroy()
	 */
	@Override
	public void destroy() {
		default_legend = null;
		default_serv_conf = null;
		this.repoMap = null;
		super.destroy();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.portlet.GenericPortlet#doView()
	 */
	@Override
	public void doView(RenderRequest req, RenderResponse res)
			throws IOException, PortletException {

		long start = System.currentTimeMillis();
		/*
		 * changing portlet title
		 */
		res.setTitle(ResourceBundle.getBundle("docview_portlet",
				req.getLocale()).getString("portlet.title"));

		try {
			if (req.getPortletSession().getAttribute(USER_MEDIA_UNIT,
					PortletSession.APPLICATION_SCOPE) == null
					|| req.getPortletSession().getAttribute(
							CURRENT_RESSOURCE_IS_LOADED,
							PortletSession.APPLICATION_SCOPE) == null) {
				req.setAttribute("no_doc", true);
			}
		} catch (IllegalStateException e) {
			/*
			 * Dispatch view to JSP page
			 */
			req.setAttribute("error", "session expired");
			PortletRequestDispatcher rd = getPortletContext()
					.getRequestDispatcher(getInitParameter("erro_page_url"));
			rd.include(req, res);
			logger.debug("doView reponse time :"
					+ Long.toString(System.currentTimeMillis() - start));
		}

		/*
		 * Dispatch view to JSP page
		 */
		PortletRequestDispatcher rd = getPortletContext().getRequestDispatcher(
				getInitParameter("view_page_url"));
		rd.include(req, res);
		logger.debug("doView reponse time :"
				+ Long.toString(System.currentTimeMillis() - start));
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.portlet.GenericPortlet#doEdit()
	 */
	@Override
	public void doEdit(RenderRequest req, RenderResponse res)
			throws IOException, PortletException {

		/*
		 * long to get duration
		 */
		long start = System.currentTimeMillis();

		/*
		 * user's media unit
		 */
		MediaUnit unit = null;

		/*
		 * user's legend
		 */
		MediaUnitLegend legend = null;

		/*
		 * user's splitter
		 */
		MediaUnitSplitter split = null;

		/*
		 * changing portlet title
		 */
		res.setTitle(ResourceBundle.getBundle("docview_portlet",
				req.getLocale()).getString("portlet.title.edit"));


		if (req.getParameter(ERROR) != null) {
			/*
			 * no errors occurs
			 */
			req.setAttribute(ERROR, req.getParameter(ERROR));
			/*
			 * Dispatch view to JSP page
			 */
			PortletRequestDispatcher rd = getPortletContext()
					.getRequestDispatcher(getInitParameter("error_page_url"));
			rd.include(req, res);
		}

		/*
		 * getting user's media unit, legend and media unit splitter
		 */
		if (req.getPortletSession().getAttribute(USER_MEDIA_UNIT,
				PortletSession.APPLICATION_SCOPE) != null) {
			unit = ((MediaUnit) req.getPortletSession().getAttribute(
					USER_MEDIA_UNIT, PortletSession.APPLICATION_SCOPE));
		}
		if (req.getPortletSession().getAttribute(USER_LEGEND,
				PortletSession.APPLICATION_SCOPE) != null) {
			legend = ((MediaUnitLegend) req.getPortletSession().getAttribute(
					USER_LEGEND, PortletSession.APPLICATION_SCOPE));
			legend.setActionURL(res.createActionURL().toString());
		} else {
			if (unit != null) {
				legend = LegendFactory.getLegend(unit);
				legend.setActionURL(res.createActionURL().toString());
			}
		}
		if (req.getPortletSession().getAttribute(USER_SPLITTER,
				PortletSession.APPLICATION_SCOPE) != null) {
			split = (MediaUnitSplitter) req.getPortletSession().getAttribute(
					USER_SPLITTER, PortletSession.APPLICATION_SCOPE);
		}

		if (unit != null && split != null && legend != null
				&& unit.getUri() != null) {

			/*
			 * injecting document title using RDF selector
			 */
			String title = getTitleFromUnit(unit);
			if (title != null) {
				/*
				 * Injecting title attribute
				 */
				req.setAttribute("DOCUMENT_TITLE", title);
			} else {
				/*
				 * Injecting title attribute using lang porperties file
				 */
				req.setAttribute("DOCUMENT_TITLE", ResourceBundle.getBundle(
						"docview_portlet", req.getLocale()).getString(
						"portlet.no_doc_title"));
			}

			/*
			 * Injecting advanced functions
			 */
			req.setAttribute("show_advanced_functions", SHOW_ADVANCED_FUNCTIONS);
			
			/*
			 * Injecting current splitter position
			 */
			req.setAttribute("current_split_pos", split.getCurrentPageIndex());

			/*
			 * Injecting total splitter position
			 */
			req.setAttribute("max_positions", split.getAvaiblePageCount());

			/*
			 * Injecting legend in request
			 */
			req.setAttribute("HTML_LEGEND", legend.getHTMLLegend(true,
					ResourceBundle
							.getBundle("docview_portlet", req.getLocale())));
		} else {
			/*
			 * no document or unavailable document in session
			 */
			req.setAttribute("no_doc", true);
		}

		/*
		 * Dispatch view to JSP page
		 */
		PortletRequestDispatcher rd = getPortletContext().getRequestDispatcher(
				getInitParameter("edit_page_url"));
		rd.include(req, res);

		logger.debug("doEdit reponse time :"
				+ Long.toString(System.currentTimeMillis() - start));
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.portlet.GenericPortlet#processAction()
	 */
	@Override
	public void processAction(ActionRequest req, ActionResponse res)
			throws IOException, PortletException {

		/*
		 * long to get duration
		 */
		long start = System.currentTimeMillis();

		/*
		 * getting user's legend
		 */
		MediaUnitLegend legend = null;
		if (req.getPortletSession().getAttribute(USER_LEGEND,
				PortletSession.APPLICATION_SCOPE) != null) {
			legend = ((MediaUnitLegend) req.getPortletSession().getAttribute(
					USER_LEGEND, PortletSession.APPLICATION_SCOPE));
		}

		/*
		 * checking for update legend action
		 */
		if ((req.getParameter(MediaUnitLegend.SAVE_LEGEND_SUBMIT) != null)
				&& (legend != null)) {
			logger.debug("update user's legend");
			for (LegendField field : legend.getFields()) {

				/*
				 * getting form color
				 */
				String selectedColor = req.getParameter(legend
						.getColorInputName(field));
				/*
				 * getting check box
				 */
				String checkBox = req.getParameter(legend
						.getCheckBoxInputName(field));
				logger.debug("selected color :" + selectedColor);

				/*
				 * convert to rgb
				 */
				String[] tabColor = selectedColor.split(",");
				/*
				 * updating the field color
				 */
				if (tabColor.length == 3) {
					field.setStyleColor(tabColor[0] + "," + tabColor[1] + ","
							+ tabColor[2]);
				} else {
					field.setStyleColor(req.getParameter(legend
							.getColorInputName(field)));
				}

				/*
				 * updating field show value
				 */
				if (checkBox != null) {
					field.setShow(true);
				} else {
					field.setShow(false);
				}
			}
			/*
			 * changing portlet to view mode
			 */
			res.setPortletMode(PortletMode.VIEW);

			/*
			 * commit legend modification
			 */
			legend.updateShemColor();

			/*
			 * set modified legend in session
			 */
			req.getPortletSession().setAttribute(USER_LEGEND, legend,
					PortletSession.APPLICATION_SCOPE);
		}

		/*
		 * Processing selected instances
		 */
		if (req.getParameter("instance_uri") != null) {
			// instance is validate
			/*
			 * selected instance URI
			 */
			String instanceURI = req.getParameter("instance_uri");
			/*
			 * Selected instance class URI
			 */
			String classURI = req.getParameter("class_uri");

			/*
			 * PieceOfKnowledge to send
			 */
			PieceOfKnowledge pok = ResourceFactory.createResource("docview", ""
					+ new Date().getTime(), PieceOfKnowledge.class);

			/*
			 * RDF helper to set annotation on pok
			 */
			PoKHelper h = new JenaPoKHelper(pok);
			h.setAutoCommitMode(false);

			/*
			 * setting rdf properties
			 */
			h.createLitStat(instanceURI, WebLabProcessing.IS_CANDIDATE, "false");
			h.createResStat(instanceURI, RDF.TYPE, classURI);
			h.commit();

			logger.debug("Sending event for the action instanceSelection");
			/*
			 * sending a event a maybe redirect page
			 */
			if (USE_REDIRECTION) {
				sendEventForActionAndRedirect("instanceSelection", pok, res);
			} else {
				sendEventForAction("instanceSelection", pok, res);
			}
		}

		if (req.getParameter("candidate_instance_uri") != null) {
			// instance is candidate

			/*
			 * selected instance URI
			 */
			String instanceURI = req.getParameter("candidate_instance_uri");
			/*
			 * Selected instance class URI
			 */
			String classURI = req.getParameter("class_uri");
			/*
			 * Document uri
			 */
			// String documentURI = req.getParameter("document_uri");
			String documentURI = ((Resource) req.getPortletSession()
					.getAttribute(USER_MEDIA_UNIT,
							PortletSession.APPLICATION_SCOPE)).getUri();
			/*
			 * Candidate instance selected
			 */

			PieceOfKnowledge pok = ResourceFactory.createResource("docview", ""
					+ new Date().getTime(), PieceOfKnowledge.class);

			PoKHelper h = new JenaPoKHelper(pok);
			h.setAutoCommitMode(false);
			h.createLitStat(instanceURI, WebLabProcessing.IS_CANDIDATE, "true");
			h.createResStat(instanceURI, RDF.TYPE, classURI);
//			h.createResStat(documentURI, RDF.TYPE, WebLab
//					.getUriFromClass(Document.class));
			h.createResStat(documentURI, WebLabProcessing.REFERS_TO, instanceURI);
			h.commit();
			/*
			 * sending a event a maybe redirect page
			 */
			if (USE_REDIRECTION) {
				sendEventForActionAndRedirect("candidateInstanceSelection",
						pok, res);
			} else {
				sendEventForAction("candidateInstanceSelection", pok, res);
			}
		}

		/*
		 * checking for add to basket action
		 */
		if (req.getParameter("add_doc_to_basket_action") != null
				&& req.getParameter("add_doc_to_basket_action").equals("add")) {

			sendEventForAction(ADD_TO_BASKET_ACTION, (Resource) req
					.getPortletSession().getAttribute(USER_MEDIA_UNIT,
							PortletSession.APPLICATION_SCOPE), res);
		}
		
		if (req.getParameter("similarity_search_action") != null
				&& req.getParameter("similarity_search_action").equals("search")) {
				sendEventForActionAndRedirect(SIMILARITY_SEARCH_ACTION, (Resource) req
					.getPortletSession().getAttribute(USER_MEDIA_UNIT,
							PortletSession.APPLICATION_SCOPE), res);
		}

		logger.debug("process action reponse time :"
				+ Long.toString(System.currentTimeMillis() - start));
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.portlet.GenericPortlet#processEvent()
	 */
	@Override
	public void processEvent(EventRequest req, EventResponse resp)
			throws PortletException, IOException {
		/*
		 * long to get duration
		 */
		long start = System.currentTimeMillis();
		logger.debug(req.getEvent().getQName()+ " event received in document viewer portlet.");
		/*
		 * document received
		 */
		Document doc_evt = null;

		/*
		 * checking action according portlet.xml
		 */
		if (getReaction(req.getEvent().getQName()).getLocalPart().equals(
				"displayDocument")) {
			// receive a event containing a document
			logger.debug("Received a event containing a complete Document in document viewer portlet.");
			/*
			 * Setting IS_NOT_LOADED to true because the document is already
			 * loaded
			 */
			req.getPortletSession().setAttribute(CURRENT_RESSOURCE_IS_LOADED,
					true, PortletSession.APPLICATION_SCOPE);

			doc_evt = (Document) req.getEvent().getValue();

			logger.info("receive a loaded document");
		} else if (getReaction(req.getEvent().getQName()).getLocalPart()
				.equals("loadAndDisplayDocument")) {
			// receive a event containing a PieceOfKnowledge about a document
			logger.debug("Received a event containing a PieceOfKnowledge about a document in document viewer portlet.");
			/*
			 * Setting IS_NOT_LOADED to false because the document is not loaded
			 * yet
			 */
			req.getPortletSession().setAttribute(CURRENT_RESSOURCE_IS_LOADED,
					false, PortletSession.APPLICATION_SCOPE);

			PieceOfKnowledge pok = (PieceOfKnowledge) req.getEvent().getValue();
			/*
			 * getting the document URI using RDFSelector
			 */
			String docURI = getURIFromPoK(pok);

			if (docURI != null) {
				/*
				 * creating the document using the retrieve URI
				 */
				doc_evt = new Document();
				doc_evt.setUri(docURI);
				logger.info("receive an unloaded document with URI : "
						+ doc_evt.getUri());
			} else {
				// no URI found in pok
				try {
					logger
							.info("receive an unloaded document with unavailable URI (null), corresponding PoK :"
									+ ResourceUtil.saveToXMLString(pok));
				} catch (WebLabCheckedException e) {
					logger.info("Unable to get PoK XML");
				}
			}
		}

		if (doc_evt != null) {
			/*
			 * Setting media unit in PortletSession
			 */
			req.getPortletSession().setAttribute(USER_MEDIA_UNIT, doc_evt,
					PortletSession.APPLICATION_SCOPE);
			
			req.getPortletSession().setAttribute(USER_RES_ANNOTS_LOADED, false, PortletSession.APPLICATION_SCOPE);
		}
		logger.debug("process event reponse time :"
				+ Long.toString(System.currentTimeMillis() - start));
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.portlet.GenericPortlet#serveResource()
	 */
	@Override
	public void serveResource(ResourceRequest request, ResourceResponse response)
			throws PortletException, IOException {

		/*
		 * long to get duration
		 */
		long start = System.currentTimeMillis();

		/*
		 * user current document
		 */
		MediaUnit unit = null;

		if (request.getPortletSession().getAttribute(USER_MEDIA_UNIT,
				PortletSession.APPLICATION_SCOPE) != null
				&& request.getPortletSession().getAttribute(
						CURRENT_RESSOURCE_IS_LOADED,
						PortletSession.APPLICATION_SCOPE) != null) {
			/*
			 * get document from session
			 */
			unit = (MediaUnit) request.getPortletSession().getAttribute(
					USER_MEDIA_UNIT, PortletSession.APPLICATION_SCOPE);

			/*
			 * checking the document is load
			 */
			if ((Boolean) request.getPortletSession().getAttribute(
					CURRENT_RESSOURCE_IS_LOADED,
					PortletSession.APPLICATION_SCOPE) == false) {
				/*
				 * Document need to be loaded from repository
				 */
				String uri_to_load = unit.getUri();

				if (uri_to_load != null) {
					/*
					 * load a resource
					 */
					boolean loaded = false;
					logger.info("new resource to load : " + uri_to_load);

					/*
					 * getting Document from repository
					 */
					LoadResourceArgs args = new LoadResourceArgs();
					args.setResourceId(uri_to_load);
					Resource loadedResource = null;

					/*
					 * setting user repo url
					 */
					if (request.getPortletSession().getAttribute(
							USER_REPO_CONF, PortletSession.APPLICATION_SCOPE) == null) {
						RepoServiceConfigBean repo_conf = new RepoServiceConfigBean();
						repo_conf
								.setServiceURL(getInitParameter("repo_service_url"));
						request.getPortletSession().setAttribute(
								USER_REPO_CONF, repo_conf,
								PortletSession.APPLICATION_SCOPE);
					}
					URL repoURL = new URL(((RepoServiceConfigBean) request
							.getPortletSession().getAttribute(USER_REPO_CONF,
									PortletSession.APPLICATION_SCOPE))
							.getServiceURL());
					try {
						/*
						 * call web service
						 */
						ResourceContainer repo;
						repo = getRepoService(repoURL);
						loadedResource = repo.loadResource(args).getResource();
						loaded = true;
					} catch (Exception e) {
						
						logger.error(e);
						request.setAttribute(ERROR,
								"Unable to get resource on repository service");
						loaded = false;
					}

					if (loaded && loadedResource instanceof Document) {
						unit = (Document) loadedResource;
						boolean containsTextUnits = false;
						
						containsTextUnits = !ResourceUtil.getSelectedSubResources(unit, Text.class).isEmpty();
						

						if (containsTextUnits) {
							/*
							 * Setting media unit in PortletSession and
							 * CURRENT_RESOURCE_IS_LOADED
							 */
							request.getPortletSession().setAttribute(
									USER_MEDIA_UNIT, unit,
									PortletSession.APPLICATION_SCOPE);
							request.getPortletSession().setAttribute(
									CURRENT_RESSOURCE_IS_LOADED, true,
									PortletSession.APPLICATION_SCOPE);
	
							/*
							 * setting new document splitter for the loaded resource
							 */
							MediaUnitSplitter split = new MediaUnitSplitter(unit,
									DEFAULT_TEXT_SIZE);
							request.getPortletSession().setAttribute(USER_SPLITTER,
									split, PortletSession.APPLICATION_SCOPE);
	
						} else {
							request.setAttribute(ERROR, true);
							request.setAttribute("no_doc", true);
							request.getPortletSession().removeAttribute(USER_SPLITTER, PortletSession.APPLICATION_SCOPE);
							request.getPortletSession().removeAttribute(USER_MEDIA_UNIT, PortletSession.APPLICATION_SCOPE);
							request.getPortletSession().removeAttribute(CURRENT_RESSOURCE_IS_LOADED, PortletSession.APPLICATION_SCOPE);
						}
					} 
				}
			}

			/*
			 * resource loaded, checking for problem when loading
			 */

			if (request.getAttribute(ERROR) != null) {
				/*
				 * Dispatch view to JSP page
				 */
				response.setContentType("text/html");
				PortletRequestDispatcher rd = getPortletContext()
						.getRequestDispatcher(
								getInitParameter("error_page_url"));
				rd.include(request, response);
			} else {
				/*
				 * no error during loading resource : continuing serving
				 * resource
				 */

				/*
				 * checking splitter
				 */
				if (request.getPortletSession().getAttribute(USER_SPLITTER,
						PortletSession.APPLICATION_SCOPE) == null) {
					/*
					 * no splitter define yet, creating it and put it in session
					 */

					request.getPortletSession().setAttribute(USER_SPLITTER,
							new MediaUnitSplitter(unit, DEFAULT_TEXT_SIZE),
							PortletSession.APPLICATION_SCOPE);
				} else {
					if (!((MediaUnitSplitter) request.getPortletSession()
							.getAttribute(USER_SPLITTER,
									PortletSession.APPLICATION_SCOPE))
							.getMediaUnit().getUri().equals(unit.getUri())) {
						/*
						 * setting in session a new splitter
						 */
						request.getPortletSession().setAttribute(USER_SPLITTER,
								new MediaUnitSplitter(unit, DEFAULT_TEXT_SIZE),
								PortletSession.APPLICATION_SCOPE);
					}
				}

				/*
				 * checking for existing legend
				 */
				if (request.getPortletSession().getAttribute(USER_LEGEND,
						PortletSession.APPLICATION_SCOPE) == null) {
					/*
					 * Legend is not set, creating it from user document or
					 * default legend
					 */

					MediaUnitLegend legend;
					if (USE_DEFAULT_LEGEND) {
						legend = default_legend;
					} else {
						legend = LegendFactory.getLegend((MediaUnit) request
								.getPortletSession().getAttribute(
										USER_MEDIA_UNIT,
										PortletSession.APPLICATION_SCOPE));
					}
					legend.setStyleContext(CSS_STYLE_CONTEXT);
					legend.setAppContext(request.getContextPath());
					legend.setActionURL(response.createActionURL().toString());

					/*
					 * Adding legend in session
					 */
					request.getPortletSession().setAttribute(USER_LEGEND,
							legend, PortletSession.APPLICATION_SCOPE);
				}

				if (request.getParameter("page_part").equals("legend")) {
					/*
					 * serving legend
					 */
					MediaUnitLegend legend = (MediaUnitLegend) (request
							.getPortletSession().getAttribute(USER_LEGEND,
							PortletSession.APPLICATION_SCOPE));

					/*
					 * setting request attribute legend
					 */
					request.setAttribute("HTML_LEGEND", legend.getHTMLLegend(
							false, ResourceBundle.getBundle("docview_portlet",
									request.getLocale())));

					/*
					 * dispatch to jsp
					 */
					response.setContentType("text/html");
					PortletRequestDispatcher rd = getPortletContext()
							.getRequestDispatcher(
									getInitParameter("legend_document_page_url"));
					rd.include(request, response);

				} else if (request.getParameter("page_part").equals("text")) {
					/*
					 * Serving text
					 */

					/*
					 * getting legend and splitter from session
					 */
					MediaUnitLegend legend = (MediaUnitLegend) request
							.getPortletSession().getAttribute(USER_LEGEND,
									PortletSession.APPLICATION_SCOPE);

					MediaUnitSplitter split = (MediaUnitSplitter) request
							.getPortletSession().getAttribute(USER_SPLITTER,
									PortletSession.APPLICATION_SCOPE);

					/*
					 * injecting document title using RDF selector
					 */
					String title = getTitleFromUnit(unit);
					if (title != null) {
						/*
						 * Injecting title attribute
						 */
						request.setAttribute("DOCUMENT_TITLE", title);
					} else {
						/*
						 * Injecting title attribute using lang porperties file
						 */
						request.setAttribute("DOCUMENT_TITLE", ResourceBundle
								.getBundle("docview_portlet",
										request.getLocale()).getString(
										"portlet.no_doc_title"));
					}

					/*
					 * Injecting advanced functions
					 */
					request.setAttribute("show_advanced_functions", SHOW_ADVANCED_FUNCTIONS);
					
					/*
					 * injecting document URI attribute
					 */
					request.setAttribute("doc_uri", unit.getUri());

					if (request.getParameter("splitter_position") != null) {
						/*
						 * updating user splitter
						 */
						split.setCurrentPageIndex(Integer.parseInt(request
								.getParameter("splitter_position")));
					}

					/*
					 * adding the document content highlighted as HTML
					 */
					Statements mediaUnitWTMap = null;
					
					if (request.getPortletSession().getAttribute(USER_RES_ANNOTS_LOADED, PortletSession.APPLICATION_SCOPE)!=null) {
						if (!(Boolean)request.getPortletSession().getAttribute(USER_RES_ANNOTS_LOADED, PortletSession.APPLICATION_SCOPE)) {
							AdvancedSelector RDFSel = RDFSelectorFactory.getSelector(true, unit.getUri());
							mediaUnitWTMap = RDFSel.searchFor(unit, RDF.TYPE, WebLabProcessing.IS_CANDIDATE, RDFS.LABEL, WebLabProcessing.REFERS_TO);
							request.getPortletSession().setAttribute(USER_MEDIA_UNIT_WTMAP, mediaUnitWTMap, PortletSession.APPLICATION_SCOPE);
							request.getPortletSession().setAttribute(USER_RES_ANNOTS_LOADED, true, PortletSession.APPLICATION_SCOPE);
						}
					}
					
					if (request.getPortletSession().getAttribute(USER_MEDIA_UNIT_WTMAP, PortletSession.APPLICATION_SCOPE) != null ) {
						mediaUnitWTMap = (Statements) request.getPortletSession().getAttribute(USER_MEDIA_UNIT_WTMAP, PortletSession.APPLICATION_SCOPE);
					}
					
					PortletURL actionURL = response.createActionURL();
					/*
					 * Injecting Highlighted HTML
					 */
					request.setAttribute("HTML_DOCUMENT", MediaUnitHighLighter
							.getHtmlRepresentation(unit, "docview_portlet",
									legend, split.getCurrentPageStartOffset(),
									split.getCurrentPageEndOffset(), actionURL,
									ResourceBundle.getBundle("docview_portlet",
											request.getLocale()),
									SHOW_INSTANCES_LINK, SHOW_TOOLTIPS, SHOW_ADVANCED_FUNCTIONS, mediaUnitWTMap));
					/*
					 * Injecting current splitter position
					 */
					request.setAttribute("current_split_pos", split
							.getCurrentPageIndex());

					/*
					 * Injecting total splitter position
					 */
					request.setAttribute("max_positions", split
							.getAvaiblePageCount());

					/*
					 * dispatch to jsp
					 */
					response.setContentType("text/html");
					PortletRequestDispatcher rd = getPortletContext()
							.getRequestDispatcher(
									getInitParameter("hightlighted_document_page_url"));
					rd.include(request, response);

				}
			}
		}
		logger.debug("serveresource reponse time :"
				+ Long.toString(System.currentTimeMillis() - start));
	}

	/**
	 * Method to get Respository service from it URL
	 * 
	 * @param serviceURL
	 *            url of the service
	 * @return corresponding service
	 */

	private ResourceContainer getRepoService(URL serviceURL) {

		if (!this.repoMap.containsKey(serviceURL)) {
			/*
			 * web service doesn't exist, create it and add it to service map
			 */
			logger.info("Building repo at [" + serviceURL + "]");
			

			this.repoMap.put(serviceURL, ResourceContainerUtil
					.getResourceContainerService(WSDL_REPO_URL, serviceURL));
		}
		return this.repoMap.get(serviceURL);
	}

	/**
	 * Method to get URI from a receive PieceOfKnowledge using
	 * POK_PROPERTY_DOC_URI
	 * 
	 * @param pok
	 * @return URI if exist null otherwise
	 */
	private String getURIFromPoK(PieceOfKnowledge pok) {
		if (pok != null) {
			Statements map = this.rdfSelector.searchFor(pok,
					POK_PROPERTY_DOC_URI);

			if (map != null && !map.entrySet().isEmpty()) {
				return (String) map.entrySet().iterator().next().getValue()
						.getValue(POK_PROPERTY_DOC_URI);
			}
		} else {
			logger.error("Received Pok is null");
		}
		return null;
	}

	/**
	 * Method to get title from media unit using DublinCore title property
	 * 
	 * @param unit
	 *            the media unit
	 * @return title of the media unit if exist null otherwise
	 */

	private String getTitleFromUnit(MediaUnit unit) {
		String title = null;
		Statements map = null;
		if (unit != null) {
			map = this.rdfSelector.searchFor(unit,
					DublinCore.TITLE_PROPERTY_NAME);
			if (map != null) {
				if (map.get(unit.getUri()) != null
						&& ((map.get(unit.getUri()))
								.getValue(DublinCore.TITLE_PROPERTY_NAME)) != null) {
					title = ((map.get(unit.getUri()))
							.getValue(DublinCore.TITLE_PROPERTY_NAME)).toString();
				}
			}
			if (title == null) {
				map = this.rdfSelector.searchFor(unit,
						WebLabProcessing.HAS_ORIGINAL_FILE_NAME);
				if (map != null) {
					if (map.get(unit.getUri()) != null
							&& ((map.get(unit.getUri()))
									.getValue(WebLabProcessing.HAS_ORIGINAL_FILE_NAME)) != null) {
						title = ((map.get(unit.getUri()))
								.getValue(WebLabProcessing.HAS_ORIGINAL_FILE_NAME))
								.toString();
					}
				}
			}
		}
		return title;
	}
}
