/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2009 EADS DEFENCE AND SECURITY SYSTEMS
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.portlet;

import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.util.Map;
import java.util.ResourceBundle;

import javax.portlet.GenericPortlet;
import javax.portlet.PortletException;
import javax.portlet.PortletRequestDispatcher;
import javax.portlet.RenderRequest;
import javax.portlet.RenderResponse;
import javax.portlet.ResourceRequest;
import javax.portlet.ResourceResponse;
import javax.xml.namespace.QName;
import javax.xml.ws.BindingProvider;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.ow2.weblab.chain.ChainPortType;
import org.ow2.weblab.chain.ChainRequest;
import org.ow2.weblab.chain.ChainResponse;
import org.ow2.weblab.chain.ChainService;

/**
 * Class that call the processing chain in the bus with the associated context:
 * <ul>
 * <li>files context: ask for files processing</li>
 * <li>warcs context: ask for warc archives processing</li>
 * </ul>
 *  
 * @author asaval
 *
 */
public class LaunchChainPortlet extends GenericPortlet {

	private ChainPortType chainPortType;

	/**
	 * The logger
	 */
	private Log logger = LogFactory.getLog(LaunchChainPortlet.class);
	
	/**
	 * Name identifier of the <select> jsp tag for context selection. 
	 */
	private static final String CONTEXT = "context";
	
	/**
	 * Context to process files (default context) 
	 */
	private static final String FILES_CONTEXT = "files";
	
	/**
	 * Context to process warcs  
	 */
	private static final String WARCS_CONTEXT = "warcs";
	
	@Override
	public void init() throws PortletException {
		super.init();
		try {
			URL wsdlUrl = new File(getPortletContext().getRealPath(
					"WEB-INF/classes/ChainDefinition.wsdl")).toURI().toURL();
			this.chainPortType = new ChainService(wsdlUrl,
					new QName("http://weblab.ow2.org/Chain/",
							"ChainService")).getChainEndpoint();
			Map<String, Object> context = ((BindingProvider) this.chainPortType)
					.getRequestContext();
			context.put(BindingProvider.ENDPOINT_ADDRESS_PROPERTY,
					getInitParameter("chain_url"));

		} catch (final Exception e) {
			throw new PortletException(e);
		}
	}

	@Override
	protected void doView(RenderRequest request, RenderResponse response)
			throws PortletException, IOException {
		/*
		 * changing portlet title
		 */
		response.setTitle(ResourceBundle.getBundle("chain_launcher_portlet",
				request.getLocale()).getString("portlet.title"));

		PortletRequestDispatcher prdis = getPortletContext().getRequestDispatcher(
				getInitParameter("launch_page_url"));
		prdis.include(request, response);
	}

	@Override
	public void serveResource(ResourceRequest request, ResourceResponse response)
			throws PortletException, IOException {
		try {
			String context = readContext(request);
			ChainRequest chainRequest = new ChainRequest();
			chainRequest.setContext(context);
			ChainResponse ret = this.chainPortType
					.process(chainRequest);
			request.setAttribute("res", ret);

			PortletRequestDispatcher prdis = getPortletContext()
					.getRequestDispatcher(getInitParameter("res_page_url"));
			prdis.include(request, response);
		} catch (final Exception e) {
			PortletRequestDispatcher prdis = getPortletContext()
					.getRequestDispatcher(getInitParameter("error_page_url"));
			request.setAttribute("message", e.getMessage());
			prdis.include(request, response);
		}

	}

	/**
	 * Retrieve the context from user selection.
	 * By default, it select the FILES_CONTEXT, the WARCS_CONTEXT is selected only on explicit selection from the user.
	 * @param request the request
	 * @return the context to send with the chain call, FILES_CONTEXT by default
	 */
	private String readContext(ResourceRequest request) {
		String value = request.getParameter(CONTEXT);
		logger.debug("Received context is : "+value);
		String context = FILES_CONTEXT;
		if (value != null && value.equals(WARCS_CONTEXT)){
				context = WARCS_CONTEXT;
		}
		logger.debug("Context uses : "+context);
		return context;
	}

}
