/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 *
 * Copyright (C) 2004 - 2009 EADS DEFENCE AND SECURITY SYSTEMS
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of the
 * GNU Lesser General Public License as published by the Free Software Foundation; either version
 * 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See
 * the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.portlet.tool;

import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map.Entry;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.ow2.weblab.core.extended.factory.ResourceFactory;
import org.ow2.weblab.core.extended.ontologies.RDF;
import org.ow2.weblab.core.extended.ontologies.WebLabRetrieval;
import org.ow2.weblab.core.helper.PoKHelper;
import org.ow2.weblab.core.helper.impl.AdvancedSelector;
import org.ow2.weblab.core.helper.impl.IPredicateValuePairs;
import org.ow2.weblab.core.helper.impl.JenaPoKHelper;
import org.ow2.weblab.core.helper.impl.RDFSelectorFactory;
import org.ow2.weblab.core.helper.impl.Statements;
import org.ow2.weblab.core.helper.impl.WTriple;
import org.ow2.weblab.core.model.PieceOfKnowledge;
import org.ow2.weblab.core.model.ResultSet;
import org.ow2.weblab.portlet.bean.AnnotationsDescBean;
import org.ow2.weblab.portlet.bean.HitDescriptionBean;

/**
 * A navigation manager for ResultSet.
 * 
 * @author emilien
 * 
 */
public class ResultSetSplitter {

	private String SORT_METHOD;
	private static final String OFFSET_PROPERTY = WebLabRetrieval.HAS_EXPECTED_OFFSET;
	private static final String LIMIT_PROPERTY = WebLabRetrieval.HAS_EXPECTED_LIMIT;
	private int splitSize;
	private int totalNumberOfResults;
	private int totalSplitterPartNumber;
	private String queryURI;

	private int currentPosition;
	private AnnotationsDescBean descBean;
	private HitDescriptionBean allRSMap;
	private AdvancedSelector rdfSelector;

	private Log logger = LogFactory.getLog(this.getClass());;

	public ResultSetSplitter() {
		// nothing to do
	}

	/**
	 * Constructor.
	 * 
	 * @param args
	 *            the searchArgs corresponding the search.
	 * @param numberOfresults
	 *            the number of result fot this searchArgs.
	 */
	public ResultSetSplitter(ResultSet rs, int splitSize,
			AnnotationsDescBean annotDesc, String sortMethod) {
		this.SORT_METHOD = sortMethod;
		this.descBean = annotDesc;
		/*
		 * setting size of each split
		 */
		this.splitSize = splitSize;
		this.rdfSelector = RDFSelectorFactory.getSelector(true);
		this.allRSMap = new HitDescriptionBean();
		/*
		 * getting rdf map from rs
		 */
		Statements fullResutSetDecMap = this.rdfSelector.searchFor(rs,
				annotDesc.getDisctinctAnnotationsNS());
		
		/*
		 * getting the total number of results for this resultset
		 */		
		this.totalNumberOfResults =this.rdfSelector
			.searchFor(rs, WebLabRetrieval.HAS_NUMBER_OF_RESULTS)
			.getTypedValue(rs.getUri(), WebLabRetrieval.HAS_NUMBER_OF_RESULTS, Integer.class);
			
		logger.info("totalNumberOfResults: "+this.totalNumberOfResults);
		
		
		/*
		 * getting page count
		 */
		this.totalSplitterPartNumber = (int) Math
				.ceil((float) this.totalNumberOfResults / this.splitSize);

		/*
		 * setting current position to 1
		 */
		this.currentPosition = 1;

		/*
		 * getting Query URI
		 */
		this.queryURI = (String) this.rdfSelector.searchFor(rs,
				WebLabRetrieval.IS_RESULT_OF).get(rs.getUri()).getValue(
				WebLabRetrieval.IS_RESULT_OF);

		/*
		 * sorting map
		 */
		if (this.SORT_METHOD.equals("DEFAULT")) {
			fillMapDefault(fullResutSetDecMap);
		} else {
			fillMapByRank(fullResutSetDecMap);
		}
	}

	/**
	 * Get the searchArgs of the current page of the search.
	 * 
	 * @return the SearchArgs of the current page.
	 */
	public HitDescriptionBean getWTripleMap(int splitNumber) {

		HitDescriptionBean res = new HitDescriptionBean();
		logger.debug("Asking for index:" + splitNumber);

		if (isInMemory(splitNumber)) {
			int startOffset;
			int endOffset;
			if (splitNumber == 1) {
				startOffset = 1;
			} else {
				startOffset = (splitNumber - 1) * this.splitSize + 1;
			}
			if (splitNumber == this.totalSplitterPartNumber) {

				endOffset = this.totalNumberOfResults;
			} else {
				endOffset = startOffset + this.splitSize - 1;
			}

			for (int i = startOffset; i <= endOffset; i++) {
				res.getDescriptionMap().put(Integer.toString(i), this.allRSMap.getDescriptionMap().get(Integer
						.toString(i)));
			}
			return res;
		}
		return null;
	}

	/**
	 * Get the searchArgs of the current page of the search.
	 * 
	 * @return the SearchArgs of the current page.
	 */
	public HitDescriptionBean getCurrentWTripleMap() {
		return getWTripleMap(this.currentPosition);
	}

	/**
	 * Change the current page to the next page.
	 * 
	 * @return the new current page.
	 */
	public int nextPosition() {
		if (this.currentPosition < this.totalSplitterPartNumber) {
			this.currentPosition++;
		}
		return this.currentPosition;

	}

	/**
	 * Change the current page to the previous page.
	 * 
	 * @return the new current page.
	 */
	public int prevPostion() {
		if (this.currentPosition > 1) {
			this.currentPosition--;
		}
		return this.currentPosition;
	}

	/**
	 * Get the total result page number for the search according the total
	 * number of result.
	 * 
	 * @return the number of page avaible for this search.
	 */
	public int getAvaibleSubResourcesNumber() {
		return this.totalSplitterPartNumber;
	}

	/**
	 * Get the current page index for this search
	 * 
	 * @return the page number for this search
	 */
	public int getCurrentPosition() {
		return this.currentPosition;
	}

	/**
	 * Set the current page index to a new index.
	 * 
	 * @param currentPage
	 * @return the new current page index.
	 */
	public int setCurrentPageIndex(int currentPosition) {
		if (currentPosition > 0
				&& currentPosition <= this.totalSplitterPartNumber) {
			this.currentPosition = currentPosition;
		}
		return this.currentPosition;
	}

	/**
	 * return the number of result of the search
	 * 
	 * @return the number of result for the search
	 */
	public int getTotalNumberOfResources() {
		return this.totalSplitterPartNumber;
	}

	public boolean isInMemory(int splitNumber) {

		int startOffset;
		int endOffset;
		if (splitNumber == 1) {
			startOffset = 1;
		} else {
			startOffset = (splitNumber - 1) * this.splitSize + 1;
		}
		if (splitNumber == this.totalSplitterPartNumber) {
			endOffset = this.totalNumberOfResults;
		} else {
			endOffset = startOffset + this.splitSize - 1;
		}

		this.logger.debug(startOffset + " - " + endOffset + this.allRSMap);
		if (startOffset == endOffset) {
			return this.allRSMap.getDescriptionMap().containsKey(Integer.toString(startOffset));
		}
		
		for (int i = startOffset; i < endOffset; i++) {
			if (!this.allRSMap.getDescriptionMap().containsKey(Integer.toString(i))) {
				return false;
			}
		}
		return true;
	}

	public boolean currentSplitIsInMemory() {
		return isInMemory(this.currentPosition);
	}

	/**
	 * Update this splitter with the new resultset without changing the current
	 * position.
	 * 
	 * @param rs
	 */

	public void updateResultSet(ResultSet rs) {
		/*
		 * getting rdf map from rs
		 */
		Statements fullResutSetDecMap = this.rdfSelector.searchFor(rs);
		
		/*
		 * sorting map
		 */
		if (this.SORT_METHOD.equals("RANK")) {
			fillMapByRank(fullResutSetDecMap);
		} else if (this.SORT_METHOD.equals("DEFAULT")) {
			fillMapDefault(fullResutSetDecMap);
		}

		/*
		 * getting the total number of results for this resultset
		 */
		
//		int nbRes = Integer.parseInt(((this.rdfSelector.searchFor(rs,
//				WebLab.HAS_NUMBER_OF_RESULTS).get(rs.getUri()))
//				.getValue(WebLab.HAS_NUMBER_OF_RESULTS)).toString());
		int nbRes = this.rdfSelector
		.searchFor(rs, WebLabRetrieval.HAS_NUMBER_OF_RESULTS)
		.getTypedValue(null, WebLabRetrieval.HAS_NUMBER_OF_RESULTS, Integer.class);
		
		/*
		 * updating number of splits
		 */
		this.totalSplitterPartNumber = (int) Math.ceil((float) nbRes
				/ this.splitSize);
	}

	public PieceOfKnowledge getPOK(int splitNumber) {
		if (isInMemory(splitNumber)) {
			return null;
		}

		int startOffset;
		if (splitNumber == 1) {
			startOffset = 0;
		} else {
			startOffset = (splitNumber - 1) * this.splitSize;
		}

		PieceOfKnowledge pok = ResourceFactory.createResource("resultporlet",
				"" + new Date().getTime(), PieceOfKnowledge.class);

		PoKHelper h =new JenaPoKHelper(pok); 
		h.setAutoCommitMode(false);
		h.createLitStat(pok.getUri(), RDF.TYPE, WebLabRetrieval.SEARCH_ORDER);
		h.createLitStat(pok.getUri(), WebLabRetrieval.HAS_ORDERED_QUERY, this.queryURI);
		h.createLitStat(pok.getUri(), OFFSET_PROPERTY, Integer
				.toString(startOffset));
		h.createLitStat(pok.getUri(), LIMIT_PROPERTY, Integer
				.toString(this.splitSize));
		h.commit();
		return pok;
	}

	public PieceOfKnowledge getCurrentSplitPOK() {
		return getPOK(this.currentPosition);
	}
	
	/**
	 * Sort method accessor
	 */
	public String getSortMethod() {
		return this.SORT_METHOD;
	}

	/**
	 * private void to fill the sorted map, ordering by rank
	 * 
	 * @param fullResutSetDecMap
	 */
	private void fillMapByRank(Statements fullResutSetDecMap) {
		/*
		 * filling allRSMap using Rank value as key
		 */
		Iterator<Entry<String, IPredicateValuePairs>> hitIt = fullResutSetDecMap.entrySet().iterator();
		
		while (hitIt.hasNext()) {
			String resURI = null;
			Entry<String, IPredicateValuePairs> entry = hitIt.next();
			
			// checking if entry corresponds to hit description
			if (entry.getValue().getValue(WebLabRetrieval.HAS_RANK) != null) {
				String rank = (String) entry.getValue().getValue(WebLabRetrieval.HAS_RANK);
				//entry is a hit description with a rank value
				resURI = fullResutSetDecMap.getFirstValue(entry.getKey(), WebLabRetrieval.IS_LINKED_TO, null);
				
				HashMap<String, List<String>> hitDescription = new HashMap<String, List<String>>();
				//filling hit description
				
				
				for (WTriple triple: entry.getValue().getStatements()) {
					
					if (triple.getObject() != null || !triple.getObject().equals("")) {
						if (hitDescription.containsKey(triple.getPredicate())) {
							hitDescription.get(triple.getPredicate()).add(triple.getObject());
						} else {
							LinkedList<String> values = new LinkedList<String>();
							values.add(triple.getObject());
							hitDescription.put(triple.getPredicate(), values);
						}
					}
				}

				// filling description about resource linked to this hit
				if (fullResutSetDecMap.containsKey(resURI)) {
					IPredicateValuePairs resDesc = fullResutSetDecMap.get(resURI);
					//filling hit description
					for (WTriple triple: resDesc.getStatements()) {
						if (triple.getObject() != null || !triple.getObject().equals("")) {
							if (hitDescription.containsKey(triple.getPredicate())) {
								hitDescription.get(triple.getPredicate()).add(triple.getObject());
							} else {
								LinkedList<String> values = new LinkedList<String>();
								values.add(triple.getObject());
								hitDescription.put(triple.getPredicate(), values);
							}
						}
					}
				}
				this.allRSMap.getDescriptionMap().put(rank, hitDescription);
			}
		}
	}
	
	/**
	 * private void to fill the sorted map, ordering by rank
	 * 
	 * @param fullResutSetDecMap
	 */
	private void fillMapDefault(Statements fullResutSetDecMap) {
		/*
		 * filling allRSMap using Rank value as key
		 */
		int rank;
		if (this.currentPosition == 1) {
			rank = 1;
		} else {
			rank = (this.currentPosition - 1) * this.splitSize + 1;
		}
	
		Iterator<Entry<String, IPredicateValuePairs>> hitIt = fullResutSetDecMap.entrySet().iterator();
		
		while (hitIt.hasNext()) {
			String resURI = null;
			Entry<String, IPredicateValuePairs> entry = hitIt.next();
			
			// checking if entry corresponds to hit description
			if (entry.getValue().getValue(WebLabRetrieval.HAS_RANK) != null) {
				
				//entry is a hit description with a rank value
				resURI = fullResutSetDecMap.getFirstValue(entry.getKey(), WebLabRetrieval.IS_LINKED_TO, null);
				
				HashMap<String, List<String>> hitDescription = new LinkedHashMap<String, List<String>>();
				//filling hit description
				for (WTriple triple: entry.getValue().getStatements()) {
					if (triple.getObject() != null || !triple.getObject().equals("")) {
					
						if (hitDescription.containsKey(triple.getPredicate())) {
							hitDescription.get(triple.getPredicate()).add(triple.getObject());
						} else {
							LinkedList<String> values = new LinkedList<String>();
							values.add(triple.getObject());
							hitDescription.put(triple.getPredicate(), values);
						}
					}
				}

				// filling description about resource linked to this hit
				if (fullResutSetDecMap.containsKey(resURI)) {
					IPredicateValuePairs resDesc = fullResutSetDecMap.get(resURI);
					//filling hit description
					for (WTriple triple: resDesc.getStatements()) {
						if (triple.getObject() != null || !triple.getObject().equals("")) {
							if (hitDescription.containsKey(triple.getPredicate())) {
								hitDescription.get(triple.getPredicate()).add(triple.getObject());
							} else {
								LinkedList<String> values = new LinkedList<String>();
								values.add(triple.getObject());
								hitDescription.put(triple.getPredicate(), values);
							}
						}
					}
				}			
				this.allRSMap.getDescriptionMap().put(Integer.toString(rank), hitDescription);
				rank++;
			}
		}
	}
}
