/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 *
 * Copyright (C) 2004 - 2009 EADS DEFENCE AND SECURITY SYSTEMS
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of the
 * GNU Lesser General Public License as published by the Free Software Foundation; either version
 * 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See
 * the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.portlet;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLDecoder;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.ResourceBundle;

import javax.portlet.ActionRequest;
import javax.portlet.ActionResponse;
import javax.portlet.EventRequest;
import javax.portlet.EventResponse;
import javax.portlet.PortletException;
import javax.portlet.PortletMode;
import javax.portlet.PortletRequestDispatcher;
import javax.portlet.PortletSession;
import javax.portlet.RenderRequest;
import javax.portlet.RenderResponse;
import javax.portlet.ResourceRequest;
import javax.portlet.ResourceResponse;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.ow2.weblab.core.extended.factory.ResourceFactory;
import org.ow2.weblab.core.extended.ontologies.DublinCore;
import org.ow2.weblab.core.extended.ontologies.RDF;
import org.ow2.weblab.core.extended.ontologies.RDFS;
import org.ow2.weblab.core.extended.ontologies.WebLabProcessing;
import org.ow2.weblab.core.extended.ontologies.WebLabRetrieval;
import org.ow2.weblab.core.helper.PoKHelper;
import org.ow2.weblab.core.helper.impl.AdvancedSelector;
import org.ow2.weblab.core.helper.impl.JenaPoKHelper;
import org.ow2.weblab.core.helper.impl.RDFSelectorFactory;
import org.ow2.weblab.core.helper.impl.Statements;
import org.ow2.weblab.core.helper.impl.WTriple;
import org.ow2.weblab.core.model.ComposedResource;
import org.ow2.weblab.core.model.Document;
import org.ow2.weblab.core.model.PieceOfKnowledge;
import org.ow2.weblab.core.model.Resource;
import org.ow2.weblab.core.model.ResultSet;
import org.ow2.weblab.core.services.ResourceContainer;
import org.ow2.weblab.core.services.resourcecontainer.LoadResourceArgs;
import org.ow2.weblab.portlet.bean.AnnotationsDescBean;
import org.ow2.weblab.portlet.bean.HitDescriptionBean;
import org.ow2.weblab.portlet.bean.RepoServiceConfigBean;
import org.ow2.weblab.portlet.bean.ResultConfBean;
import org.ow2.weblab.portlet.tool.ResourceContainerUtil;
import org.ow2.weblab.portlet.tool.ResourceRankFactory;
import org.ow2.weblab.portlet.tool.ResultSetSplitter;
import org.springframework.beans.factory.BeanFactory;
import org.springframework.beans.factory.xml.XmlBeanFactory;
import org.springframework.core.io.ClassPathResource;

/**
 * A portlet to display WebLab ResultSet, navigate in it and select some Hits.
 * @author emilien
 *
 */
public class ResultPortlet extends WebLabPortlet {
	/**
	 * Fields for events mapping according portlet.xml
	 */
	private static String LOADED_DOCUMENT_ACTION = "documentSelection";
	private static String NOT_LOADED_IMAGE_ACTION = "notLoadedImageSelection";
	private static String NOT_LOADED_DOCUMENT_ACTION = "notLoadedDocumentSelection";
	private static String LOADED_DOCUMENTS_ACTION = "documentsSelection";
	private static String NOT_LOADED_DOCUMENTS_ACTION = "notLoadedDocumentsSelection";
	private static String NEXT_RESULT_ACTION = "nextDocumentsSelection";
	private static final String ALL_NOT_LOADED_DOCUMENTS_GEO_ACTION = "viewNotLoadedDocumentsGeo";
	private static final String ALL_NOT_LOADED_DOCUMENTS_TIME_ACTION = "viewNotLoadedDocumentsTime";
	
	private static String DISPLAY_RESULTS = "displayResults";
	/**
	 * Default portlet behavior for redirection when use_rediction not set in
	 * portlet.xml
	 */
	private static boolean USE_REDIRECTION = false;

	/**
	 * The number of stars displayed in hit (when score value > 0 >= 1)
	 */
	private static int RANK_DIGIT_NUMBER = 5;

	/**
	 * The number of hit per result page
	 */
	private static int MAX_DISPLAYED_RESOURCES = 5;

	/**
	 * The maximum number of page displayed beside current position
	 */
	private static int MAX_DISPLAYED_PAGES = 8;

	/**
	 * Load each resources before (display or) send its. When true, sending events
	 * are Document or ResourceCollection of Documents, PieceOfKnowledge
	 * otherwise.
	 */
	private static boolean loadResource;
	
	/**
	 * Sort method used to sort hits when possible
	 */
	private static final String SORT_METHOD = "RANK";
	
	/**
	 * Display type
	 */
	private static final String TITLE_RANK_MODE = "title - rank";
	
	/**
	 * Sort method used to sort hits when possible
	 */
	private static final String DEFAULT_SORT_METHOD = "DEFAULT";

	/**
	 * RDF property used to get Hit on processed ResultSet
	 */
	private static final String HIT_PROPERTY = WebLabRetrieval.HIT;
	
	/**
	 * Attribute name of the user's current ResultSet in session
	 */
	private static final String USER_RESULT_SET = "resultSet";

	/**
	 * 
	 */
	private static final String DISPLAY_MODE = "display_mode";
	
	/**
	 * Attribute name of the user's current ResultSetSplitter in session
	 */
	private static final String USER_RESULTSET_SPLITTER = "user_rs_splitter";

	/**
	 * Attribute name of the user's current display configuration bean in
	 * session
	 */
	private static final String META_CONF_BEAN_ID = "meta_conf_bean";

	/**
	 * Attribute name of the user's current repository service configuration
	 * bean in session
	 */
	private static final String DEFAULT_REPO_BEAN = "repo_conf_bean";

	/**
	 * Form parameter name to get URI of selected resource
	 */
	public static final String LOAD_RESOURCE_ACTION = "load_resource_uri";

	/**
	 * Form parameter name to get URIs of selected resources
	 */
	public static final String SELECTED_RESOURCES = "selected_res";

	/**
	 * Form parameter name to get the selected splitter position
	 */
	private static final String UPDATE_SPLITTER_ACTION = "next_splitter_position";

	/**
	 * Attribute name used to send hit description to jsp
	 */
	public static final String RESULTS_MAP_DESC = "resultSet_desc";
	
	/**
	 * Default meta configuration bean
	 */
	public static ResultConfBean DEFAULT_META_CONF_BEAN;

	/**
	 * Attribute name used to send display configuration bean to jsp
	 */
	public static final String USER_META_CONF = "meta_conf_bean";

	/**
	 * Attribute name used to send repository service configuration bean to jsp
	 */
	public static final String USER_REPO_CONF = "user_repo_service_conf";

	/**
	 * Attribute name used to send a error message to jsp
	 */
	public static final String ERROR = "message_error";

	/**
	 * RDFSelector used by the portlet
	 */
	//private RDFSelector rdfSelector;

	/**
	 * Map of resource container services
	 */
	private Map<URL, ResourceContainer> repoMap;

	/**
	 * Default AnnotationsDescBean
	 */
	private AnnotationsDescBean rdfProperties;
	
	/**
	 * Logger
	 */
	private Log logger = LogFactory.getLog(this.getClass());
	
	/**
	 * URL of WSDL of repository service
	 */
	private static URL WSDL_REPO_URL;

	private static final String EXPOSED_PROPERTY =WebLabProcessing.NAMESPACE+ "isExposedAsThumbnail";
	private static final String  LINKED_TO_RELEVANT_MU = WebLabRetrieval.NAMESPACE+"isLinkedToRelevantMediaUnit";


	

	@Override
	public void init() throws PortletException {
		super.init();
		/*
		 * creating map of repo service
		 */
		this.repoMap = new HashMap<URL, ResourceContainer>();

		/*
		 * getting repository WSDL URL
		 */
		try {

			WSDL_REPO_URL = new File(getPortletContext().getRealPath(
					"WEB-INF/classes/services/WebLab.wsdl")).toURI()
					.toURL();

		} catch (MalformedURLException e) {
			logger.error(e);
		}

		/*
		 * getting meta-data description list from a XML bean, using Spring
		 */
		ClassPathResource resource = new ClassPathResource(AnnotationsDescBean.DEFAULT_CONF_FILE);
		BeanFactory factory = new XmlBeanFactory(resource);
		this.rdfProperties = factory.getBean(AnnotationsDescBean.DEFAULT_BEAN_NAME, AnnotationsDescBean.class);

		/*
		 * set loading parameter, if set to true, received documents by eventing
		 * are loaded from the repository
		 */
		if (getInitParameter("load_resources_on_repository") != null) {
			loadResource = Boolean
					.parseBoolean(getInitParameter("load_resources_on_repository"));
		} else {
			loadResource = false;
		}
		
		/*
		 * set the max displayed
		 */
		if (getInitParameter("rank_digit_number") != null) {
			RANK_DIGIT_NUMBER = Integer.parseInt(getInitParameter("rank_digit_number"));
		}
		/*
		 * Set the number of hit per result page
		 */
		if (getInitParameter("max_displayed_resources") != null) {
			MAX_DISPLAYED_RESOURCES = Integer.parseInt(getInitParameter("max_displayed_resources"));
		}
		
		/*
		 * Set the maximum number of page displayed beside current position
		 */
		if (getInitParameter("max_displayed_pages") != null) {
			MAX_DISPLAYED_PAGES = Integer.parseInt(getInitParameter("max_displayed_pages"));
		}
		
		/*
		 * set the use_redirection parameter, if set to true, request are
		 * redirected after actions
		 */
		if (getInitParameter("use_redirection") != null) {
			USE_REDIRECTION = Boolean
					.parseBoolean(getInitParameter("use_redirection"));
		} else {
			USE_REDIRECTION = false;
		}
		
		/*
		 * initializing meta conf factory
		 */
		DEFAULT_META_CONF_BEAN = factory.getBean(META_CONF_BEAN_ID, ResultConfBean.class);
	
	}

	@Override
	public void destroy() {
		this.repoMap = null;
		this.rdfProperties = null;
		//this.rdfSelector=null;
		super.destroy();
	}

	/**
	 * Render HTML corresponding to the view mode
	 */
	@Override
	public void doView(RenderRequest req, RenderResponse res)
			throws IOException, PortletException {

		long start = System.currentTimeMillis();
		boolean sessionExpired = false;
		/*
		 * changing portlet title
		 */
		res.setTitle(ResourceBundle
				.getBundle("result_portlet", req.getLocale()).getString(
						"portlet.title"));

		/*
		 * checking for session expiration
		 */
		if ((req.getPortletSession().getLastAccessedTime() + (req
				.getPortletSession().getMaxInactiveInterval() * 1000)) < System
				.currentTimeMillis()) {
			sessionExpired = true;
			req.setAttribute("message_warning", ResourceBundle.getBundle(
					"result_portlet", req.getLocale()).getString(
					"portlet.warning.session_expired"));
		}

		if (!sessionExpired) {
			Statements  map;

			if (req.getParameter(ERROR) != null) {
				/*
				 * dispatch to error jsp
				 */
				req.setAttribute(ERROR, req.getParameter(ERROR));
				PortletRequestDispatcher rd = getPortletContext()
						.getRequestDispatcher(
								getInitParameter("error_page_url"));
				rd.include(req, res);
			}

			ResultSet resultset = (ResultSet) req.getPortletSession()
					.getAttribute(USER_RESULT_SET,
							PortletSession.APPLICATION_SCOPE);

			ResultSetSplitter splitter = (ResultSetSplitter) req
					.getPortletSession().getAttribute(USER_RESULTSET_SPLITTER,
							PortletSession.APPLICATION_SCOPE);

			
			if (resultset != null) {
				String[] properties = {WebLabRetrieval.HAS_NUMBER_OF_RESULTS, RDFS.LABEL};
				AdvancedSelector rdfSelector = RDFSelectorFactory.getSelector(true);
				/*
				 * getting WTripleMap
				 */
				map = rdfSelector.searchFor(resultset,properties);
				
				/*
				 * injecting number of result
				 */
				req.setAttribute("hasNumberOfResults", map.getTypedValue(resultset.getUri(), WebLabRetrieval.HAS_NUMBER_OF_RESULTS, Integer.class));
				
//				/*
//				 * Injecting ResultSet label
//				 */
//				req.setAttribute("resultSet_label",map.getFirstValue(null, RDFS.LABEL, req.getLocale()));
	
				if (splitter != null) {

					/*
					 * Injecting current splitter position
					 */
					req.setAttribute("current_split_pos", splitter
							.getCurrentPosition());

					/*
					 * Injecting total splitter position
					 */
					req.setAttribute("max_positions", splitter
							.getAvaibleSubResourcesNumber());

					if(splitter.getSortMethod().equals(DEFAULT_SORT_METHOD)) {
						/*
						 * resultSet is not sortable
						 */
						req.setAttribute("not_sortable", true);
					}
				}
			}
			/*
			 * dispatch to jsp
			 */
			PortletRequestDispatcher rd = getPortletContext()
					.getRequestDispatcher(getInitParameter("result_page_url"));
			rd.include(req, res);
		} else {
			/*
			 * Dispatch warning to JSP page
			 */
			PortletRequestDispatcher rd = getPortletContext()
					.getRequestDispatcher(getInitParameter("warning_page_url"));
			rd.include(req, res);
		}
		logger.info(
				"doView reponse time :"
						+ Long.toString(System.currentTimeMillis() - start));
	}

	@Override
	public void doEdit(RenderRequest req, RenderResponse res)
			throws PortletException, IOException {

		long start = System.currentTimeMillis();
		ResultConfBean metaConfbean;
		RepoServiceConfigBean repoConfBean;
		
		/*
		 * result conf bean
		 */
		
		
		if (req.getPortletSession().getAttribute(USER_META_CONF,
				PortletSession.APPLICATION_SCOPE) == null) {
			metaConfbean = DEFAULT_META_CONF_BEAN;
		} else {
			metaConfbean = (ResultConfBean) req.getPortletSession().getAttribute(USER_META_CONF,
					PortletSession.APPLICATION_SCOPE);
		}
		/*
		 * injecting configuration bean
		 */
		req.setAttribute(USER_META_CONF, metaConfbean);
		

		/*
		 * repository conf bean
		 */
		if (req.getPortletSession().getAttribute(USER_REPO_CONF,
				PortletSession.APPLICATION_SCOPE) == null) {
			repoConfBean = new RepoServiceConfigBean();
		} else {	
			repoConfBean = (RepoServiceConfigBean) req.getPortletSession().getAttribute(USER_REPO_CONF,
					PortletSession.APPLICATION_SCOPE);
		}
		/*
		 * injecting configuration bean
		 */
		req.setAttribute(USER_REPO_CONF, repoConfBean);
		

		/*
		 * dispatch to jsp
		 */
		PortletRequestDispatcher rd = getPortletContext().getRequestDispatcher(
				getInitParameter("edit_page_url"));

		rd.include(req, res);
		logger.info(
				"doEdit reponse time :"
						+ Long.toString(System.currentTimeMillis() - start));
	}

	/**
	 * handle the incoming event
	 */
	@Override
	public void processEvent(EventRequest req, EventResponse resp)
			throws PortletException, IOException {
		long start = System.currentTimeMillis();
		/*
		 * reset error render parameter
		 */
		//resp.removePublicRenderParameter(ERROR);
		req.getPortletSession().removeAttribute(ERROR);
		
		logger.debug("Event received : " + req.getEvent());
		
		/*
		 * checking event
		 */
		if (req.getEvent() != null && req.getEvent().getValue() instanceof ResultSet) {

			if (getReaction(req.getEvent().getQName()).getLocalPart().equals(DISPLAY_RESULTS)) {

				ResultSet receveidResultSet = ((ResultSet) req.getEvent()
						.getValue());

				logger.info( "Event received, a new query : " + receveidResultSet);
				
				/*
				 * checking if resultSet is sortable
				 */
				
				AdvancedSelector rdfSelector = RDFSelectorFactory.getSelector(true);
				boolean isSortable = !rdfSelector.searchFor(receveidResultSet, WebLabRetrieval.HAS_RANK).isEmpty();
				
				
				
				ResultSetSplitter splitter=null;
				if (isSortable) {
					/*
					 * creating new splitter
					 */
					logger.info("processEvent : creating splitter with rank sort method");
					
					splitter = new ResultSetSplitter(
							receveidResultSet, MAX_DISPLAYED_RESOURCES,
							this.rdfProperties, SORT_METHOD);
					logger.info(
							"processEvent reponse time (creating splitter with rank sort method):"
									+ Long.toString(System.currentTimeMillis()
											- start));
				} else {
					logger.info("processEvent : creating splitter with default sort method");
					/*
					 * creating new splitter
					 */
					try {
					splitter = new ResultSetSplitter(
							receveidResultSet, MAX_DISPLAYED_RESOURCES,
							this.rdfProperties, DEFAULT_SORT_METHOD);
					} catch (Exception e) {
						req.getPortletSession().setAttribute(ERROR, "Unable to process received result", PortletSession.APPLICATION_SCOPE);
						req.getPortletSession().removeAttribute(USER_RESULTSET_SPLITTER, PortletSession.APPLICATION_SCOPE);
						req.getPortletSession().removeAttribute(USER_RESULT_SET, PortletSession.APPLICATION_SCOPE);
						
						logger.error("unable to create splitter from received resultSet"+e.getMessage());
					}
					logger.info(
							"processEvent reponse time (creating splitter with default sort):"
									+ Long.toString(System.currentTimeMillis()
											- start));
				}
				
				
				
				if (splitter !=null) {
					req.getPortletSession().setAttribute(USER_RESULTSET_SPLITTER,
							splitter, PortletSession.APPLICATION_SCOPE);
					
					/*
					 * checking if the number of hit received is enough
					 */
	
					if (!splitter.currentSplitIsInMemory()
							&& splitter.getAvaibleSubResourcesNumber() > 0) {
						/*
						 * resultSet is too small ! sending event to get a biggest
						 */
	
						logger.info(
								"resultset is too small : " + receveidResultSet);
	
						sendEventForAction(NEXT_RESULT_ACTION, splitter
								.getCurrentSplitPOK(), resp);
					}
					/*
					 * adding resource collection in user session.
					 */
					req.getPortletSession().setAttribute(USER_RESULT_SET,
							receveidResultSet, PortletSession.APPLICATION_SCOPE);
				}
			}

			if (getReaction(req.getEvent().getQName()).getLocalPart().equals(
					"updateResults")) {

				ResultSet receveidResultSet = ((ResultSet) req.getEvent()
						.getValue());

				logger.info(
						"Event received, updating Resultset : "
								+ receveidResultSet);
				/*
				 * updating splitter
				 */
				((ResultSetSplitter) req.getPortletSession().getAttribute(
						USER_RESULTSET_SPLITTER,
						PortletSession.APPLICATION_SCOPE))
						.updateResultSet(receveidResultSet);
				logger.info(
						"processEvent reponse time (updating splitter):"
								+ Long.toString(System.currentTimeMillis()
										- start));
			}
		}
		logger.info(
				"processEvent reponse time :"
						+ Long.toString(System.currentTimeMillis() - start));
	}

	@Override
	public void processAction(ActionRequest request, ActionResponse response)
			throws PortletException, IOException {
		long start = System.currentTimeMillis();
		logger.info("Process action....");
		
		if (request.getParameter("visualize_all") != null) {
			String action =request.getParameter("visualize_all");
		//String action = request.getParameter("action");

		if (action!=null && (action.equals("viewTime")||action.equals("viewGeo"))) {
			logger.info("Process action : "+action);
			final ResultSet rs=(ResultSet)request.getPortletSession().getAttribute(USER_RESULT_SET,PortletSession.APPLICATION_SCOPE);


			/*
			 * creating a pok with new hits corresponding to the query result
			 * 
			 */

			PieceOfKnowledge pok = ResourceFactory.createResource(
					"resultporlet", "" + System.nanoTime(),
					PieceOfKnowledge.class);

			PoKHelper h = new JenaPoKHelper(pok);
			h.setAutoCommitMode(false);

			AdvancedSelector rdfSelector = RDFSelectorFactory.getSelector(true);
			List<WTriple> list = rdfSelector.findInResource(rs, null, WebLabRetrieval.IS_LINKED_TO, null);
			String hitURI;
			for (WTriple wtriple:list)
			{
				hitURI="http://weblab.ow2.org/portlet/resultportlet/"
					+ System.nanoTime();
				h.createResStat(hitURI, WebLabRetrieval.IS_LINKED_TO, wtriple.getObject());
				h.createResStat(hitURI, RDF.TYPE, HIT_PROPERTY);
			}
			h.commit();
			
			
			
			
			/*
			 * send a event and maybe redirect request
			 */
			
//				if(action.equals("viewTime"))
//					this.modifyRedirection(new QName("http://weblab.ow2.org/events/result","addNotLoadedDocuments"), "/web/guest/timeline");
//				else //if(action.equals("viewGeo"))
//					this.modifyRedirection(new QName("http://weblab.ow2.org/events/result","addNotLoadedDocuments"), "/web/guest/geo");
				String redirection_uri=ALL_NOT_LOADED_DOCUMENTS_GEO_ACTION;
				if(action.equals("viewTime"))
					redirection_uri=ALL_NOT_LOADED_DOCUMENTS_TIME_ACTION;
			if (USE_REDIRECTION) {	
				sendEventForActionAndRedirect(redirection_uri,
						pok, response);
			} else {

				sendEventForAction(redirection_uri, pok,
						response);
			}		
		}
		}
		

		
		else{
		
		
		
		/*
		 * get the form name
		 */
		String form_name = request.getParameter("form_name");
		/*
		 * mapping
		 */
		if (form_name != null && form_name.equals("meta_conf_form")) {
			logger.info("meta config update");
			/*
			 * update user service config
			 */
			ResultConfBean newConf = getMetaBean(request);

			request.getPortletSession().setAttribute(USER_META_CONF,
						newConf, PortletSession.APPLICATION_SCOPE);
				response.setPortletMode(PortletMode.VIEW);
			
		}

		/*
		 * updating repo config
		 */
		if (form_name != null && form_name.equals(DEFAULT_REPO_BEAN)) {

			/*
			 * reset error render parameter
			 */
			request.getPortletSession().removeAttribute(ERROR);
//			response.removePublicRenderParameter(ERROR);
//			response.setRenderParameter(ERROR,"");
			
			logger.info("repo config update");
			/*
			 * update user service config
			 */
			RepoServiceConfigBean rc = (RepoServiceConfigBean) request
					.getPortletSession().getAttribute(USER_REPO_CONF,
							PortletSession.APPLICATION_SCOPE);
			RepoServiceConfigBean newConf = null;
			if (rc == null) {	
				/*
				 * initialize bean from request
				 */
				RepoServiceConfigBean repo_conf = new RepoServiceConfigBean();
				repo_conf.setServiceURL(getInitParameter("repo_service_url"));
					
			}
			
			request.getPortletSession().setAttribute(USER_REPO_CONF,
					newConf, PortletSession.APPLICATION_SCOPE);
			response.setPortletMode(PortletMode.VIEW);
		}

		/*
		 * One resource to load
		 */
		if (request.getParameter(LOAD_RESOURCE_ACTION) != null) {
			if (loadResource) {
				/*
				 * load a resource
				 */
				boolean loaded = false;
				String uri = URLDecoder.decode(request
						.getParameter(LOAD_RESOURCE_ACTION), "UTF-8");
				logger.info(
						"new resource to load : " + uri);

				/*
				 * getting Document from repository
				 */
				LoadResourceArgs args = new LoadResourceArgs();
				args.setResourceId(uri);
				Resource loadedResource = null;

				/*
				 * setting user repo url
				 */
				if (request.getPortletSession().getAttribute(USER_REPO_CONF,
						PortletSession.APPLICATION_SCOPE) == null) {
					RepoServiceConfigBean repo_conf = new RepoServiceConfigBean();
					repo_conf
							.setServiceURL(getInitParameter("repo_service_url"));
					request.getPortletSession().setAttribute(USER_REPO_CONF,
							repo_conf, PortletSession.APPLICATION_SCOPE);
				}

				/*
				 * Getting repo URL from session
				 */
				URL repoURL = new URL(((RepoServiceConfigBean) request
						.getPortletSession().getAttribute(USER_REPO_CONF,
								PortletSession.APPLICATION_SCOPE))
						.getServiceURL());
				try {
					/*
					 * call web service
					 */
					ResourceContainer repo;
					repo = getRepoService(repoURL);
					loadedResource = repo.loadResource(args).getResource();
					loaded = true;
				} catch (Exception e) {
					
					loaded=false;
					logger.error(e);
					
					/*
					 * reset error render parameter
					 */
					request.getPortletSession().setAttribute(ERROR,
					"Unable to get resource on repository service");
//					response.removePublicRenderParameter(ERROR);
//					response.setRenderParameter(ERROR,
//							"Unable to get resource on repository service");
				}
				if (loaded) {
					if (loadedResource == null) {

						loadedResource = ResourceFactory.createResource(
								"temporary", "" + System.nanoTime(),
								Document.class);
					}

					if (loadedResource instanceof Document) {
						/*
						 * sending event and maybe redirect request
						 */
						if (USE_REDIRECTION) {
							sendEventForActionAndRedirect(
									LOADED_DOCUMENT_ACTION, loadedResource,
									response);
						} else {

							sendEventForAction(LOADED_DOCUMENT_ACTION,
									loadedResource, response);
						}
					}
				}
			} else {

				/*
				 * no need to load the resource
				 */
				String isLinkedTo = URLDecoder.decode(request
						.getParameter(LOAD_RESOURCE_ACTION), "UTF-8");
				
				String[] selectedResourceURISplitRelevant = isLinkedTo.split("@relevantMu:");
				String linkedRelevantMU=null;
				if (selectedResourceURISplitRelevant.length>1) {
					isLinkedTo = selectedResourceURISplitRelevant[0];
					linkedRelevantMU = selectedResourceURISplitRelevant[1];
				}
				
				logger.info(
						"unloaded selection , pok about :" + isLinkedTo);
				/*
				 * creating a pok with a new hit corresponding the user
				 * selection
				 */

				PieceOfKnowledge pok = ResourceFactory.createResource(
						"resultporlet", "" + System.nanoTime(),
						PieceOfKnowledge.class);

				PoKHelper h = new JenaPoKHelper(pok);
				h.setAutoCommitMode(false);

				String hitURI = "http://weblab.ow2.org/portlet/resultportlet/"
						+ System.nanoTime();
				h.createResStat(hitURI, WebLabRetrieval.IS_LINKED_TO, isLinkedTo);
				if (linkedRelevantMU != null && !linkedRelevantMU.equals("")) {
					h.createResStat(hitURI, LINKED_TO_RELEVANT_MU, linkedRelevantMU);
				}
				h.createResStat(hitURI, RDF.TYPE, HIT_PROPERTY);
				h.commit();
				/*
				 * sending event and maybe redirect the request
				 */
				
				String format = "";
				if (request.getPortletSession().getAttribute(USER_RESULTSET_SPLITTER, PortletSession.APPLICATION_SCOPE)!= null) {
					ResultSetSplitter rsp = (ResultSetSplitter) request.getPortletSession().getAttribute(USER_RESULTSET_SPLITTER, PortletSession.APPLICATION_SCOPE);
					
					//System.out.println("rs map :"+rsp.getCurrentWTripleMap().getDescriptionMap());
					for (Entry<String, HashMap<String, List<String>>> hit:rsp.getCurrentWTripleMap().getDescriptionMap().entrySet()) {
						if (hit.getValue().get(WebLabRetrieval.IS_LINKED_TO).contains(isLinkedTo)) {
							//corresponding hit
							if (hit.getValue().containsKey(DublinCore.FORMAT_PROPERTY_NAME)) {
								format = hit.getValue().get(DublinCore.FORMAT_PROPERTY_NAME).get(0);
							}
							break;
						}
					}
				}
				logger.debug("selected resource type is:"+format);
				if (USE_REDIRECTION) {
					if (format.contains("jpeg") || format.contains("png") || format.contains("bmp") || format.contains("gif")) {
						sendEventForActionAndRedirect(NOT_LOADED_IMAGE_ACTION,
										pok, response);
					} else {
						sendEventForActionAndRedirect(NOT_LOADED_DOCUMENT_ACTION,
										pok, response);
					}
				} else {

					if (format.contains("jpeg") || format.contains("png") || format.contains("bmp") || format.contains("gif")) {
						sendEventForAction(NOT_LOADED_IMAGE_ACTION,
								pok, response);
					} else {
						sendEventForAction(NOT_LOADED_DOCUMENT_ACTION, pok,
								response);
					}
					
				}
			}
		}

		/*
		 * collection of resources selected
		 */
		if ((request.getParameter(SELECTED_RESOURCES) != null)
				&& (request.getParameter(SELECTED_RESOURCES) != "")) {

			/*
			 * creating CollectionSelection event
			 */
			ComposedResource collect = ResourceFactory.createResource(
					getDefaultNamespace() + getPortletName(),
					"CollectionSelection", ComposedResource.class);
			List<Resource> l = collect.getResource();

			if (loadResource) {

				/*
				 * setting user repo url
				 */
				if (request.getPortletSession().getAttribute(USER_REPO_CONF,
						PortletSession.APPLICATION_SCOPE) == null) {
					RepoServiceConfigBean repo_conf = new RepoServiceConfigBean();
					repo_conf
							.setServiceURL(getInitParameter("repo_service_url"));
					request.getPortletSession().setAttribute(USER_REPO_CONF,
							repo_conf, PortletSession.APPLICATION_SCOPE);
				}

				URL repoURL = new URL(((RepoServiceConfigBean) request
						.getPortletSession().getAttribute(USER_REPO_CONF,
								PortletSession.APPLICATION_SCOPE))
						.getServiceURL());

				/*
				 * call web service and add resource in event
				 */
				ResourceContainer repo;
				repo = getRepoService(repoURL);
				for (String selectedResourceURI : request
						.getParameterValues(SELECTED_RESOURCES)) {
					String uri = URLDecoder
							.decode(selectedResourceURI, "UTF-8");
					logger.info(
							"new resource to load : " + uri);
					/*
					 * getting resource from repository
					 */
					LoadResourceArgs args = new LoadResourceArgs();
					args.setResourceId(uri);

					try {
						l.add(repo.loadResource(args).getResource());
					} catch (Exception e) {
						logger.error(e);
						response.setRenderParameter(ERROR,
								"Unable to get resource on repository service");
					}
				}

				/*
				 * send a event and maybe redirect request
				 */
				if (USE_REDIRECTION) {
					sendEventForActionAndRedirect(LOADED_DOCUMENTS_ACTION,
							collect, response);
				} else {

					sendEventForAction(LOADED_DOCUMENTS_ACTION, collect,
							response);
				}
			} else {

				String[] parameters = request
						.getParameterValues(SELECTED_RESOURCES)[0].split(",");

				PieceOfKnowledge pok = ResourceFactory.createResource(
						"resultporlet", "" + System.nanoTime(),
						PieceOfKnowledge.class);

				PoKHelper h = new JenaPoKHelper(pok);
				h.setAutoCommitMode(false);
				
				//splitter
				ResultSetSplitter splitter = (ResultSetSplitter) request
				.getPortletSession().getAttribute(USER_RESULTSET_SPLITTER,
						PortletSession.APPLICATION_SCOPE);

				HitDescriptionBean hitDesc = splitter.getCurrentWTripleMap();
				
				//System.out.println("map from spliter : "+map);
				
				for (String selectedResourceURI : parameters) {
					/*
					 * no need to load the resource
					 */
					selectedResourceURI = URLDecoder.decode(
							selectedResourceURI, "UTF-8");

					String[] selectedResourceURISplitRelevant = selectedResourceURI.split("@relevantMu:");
					String linkedRelevantMU=null;
					if (selectedResourceURISplitRelevant.length>1) {
						selectedResourceURI = selectedResourceURISplitRelevant[0];
						linkedRelevantMU = selectedResourceURISplitRelevant[1];
					}
					
					logger.debug(selectedResourceURI);
					/*
					 * creating a pok with a new hit corresponding the user
					 * selection
					 */

					String hitURI = "http://weblab.ow2.org/portlet/resultportlet/selectedHit"
							+ System.nanoTime();
					h.createResStat(hitURI, WebLabRetrieval.IS_LINKED_TO,
									selectedResourceURI);
					h.createResStat(hitURI, RDF.TYPE, WebLabRetrieval.HIT);
					
					if (linkedRelevantMU != null && !linkedRelevantMU.equals("")) {
						h.createResStat(hitURI, LINKED_TO_RELEVANT_MU, linkedRelevantMU);
					}
					
					
					//trying to add meta on selected hit
					for (Entry<String, HashMap<String, List<String>>> entry : hitDesc.getDescriptionMap().entrySet()) {
					
						if (entry.getValue().get(WebLabRetrieval.IS_LINKED_TO).get(0).equals(selectedResourceURI)) {
							String title = null;
							if (entry.getValue().get(DublinCore.TITLE_PROPERTY_NAME)!=null && !entry.getValue().get(DublinCore.TITLE_PROPERTY_NAME).isEmpty()) {
								title = ((List<String>) entry.getValue().get(DublinCore.TITLE_PROPERTY_NAME)).get(0);
							}
							
							String fileName = null;
							if (entry.getValue().get(WebLabProcessing.HAS_ORIGINAL_FILE_NAME)!=null && !entry.getValue().get(WebLabProcessing.HAS_ORIGINAL_FILE_NAME).isEmpty()) {
								fileName = ((List<String>) entry.getValue().get(WebLabProcessing.HAS_ORIGINAL_FILE_NAME)).get(0);
							}
						
							String format = null;
							if (entry.getValue().get(DublinCore.FORMAT_PROPERTY_NAME)!=null && !entry.getValue().get(DublinCore.FORMAT_PROPERTY_NAME).isEmpty()) {
								format = ((List<String>) entry.getValue().get(DublinCore.FORMAT_PROPERTY_NAME)).get(0);
							}
							
							String thumbnail = null;
							if (entry.getValue().get(EXPOSED_PROPERTY)!=null && !entry.getValue().get(EXPOSED_PROPERTY).isEmpty()) {
								thumbnail = ((List<String>) entry.getValue().get(EXPOSED_PROPERTY)).get(0);
							}
							
							
//							logger.info("sending pok with title property :"+title);
//							logger.info("sending pok with format property :"+format);
//							logger.info("sending pok with thumb property :"+thumbnail);
							if (title !=null && !title.equals("")) 
								h.createLitStat(hitURI, DublinCore.TITLE_PROPERTY_NAME, title);
							if (fileName !=null && !fileName.equals("")) 
								h.createLitStat(hitURI, WebLabProcessing.HAS_ORIGINAL_FILE_NAME, fileName);
							if (format != null  && !format.equals(""))
								h.createLitStat(hitURI, DublinCore.FORMAT_PROPERTY_NAME, format);
							if (thumbnail != null && !thumbnail.equals("")) {
								h.createLitStat(hitURI, EXPOSED_PROPERTY, thumbnail);
							}
							
						}
					}
				}
				h.commit();

				/*
				 * send a event and maybe redirect request
				 */
				if (USE_REDIRECTION) {
					sendEventForActionAndRedirect(NOT_LOADED_DOCUMENTS_ACTION,
							pok, response);
				} else {

					sendEventForAction(NOT_LOADED_DOCUMENTS_ACTION, pok,
							response);
				}
			}
		}

		if (request.getParameter(UPDATE_SPLITTER_ACTION) != null) {

			/*
			 * changing the splitter position and request for next resultset
			 */
			int splitterPosition = Integer.parseInt(request
					.getParameter(UPDATE_SPLITTER_ACTION));
			this.logger.debug("Processing Action update splitter: " + splitterPosition);
			ResultSetSplitter splitter = (ResultSetSplitter) request
					.getPortletSession().getAttribute(USER_RESULTSET_SPLITTER,
							PortletSession.APPLICATION_SCOPE);
			if (splitter != null) {
				splitter.setCurrentPageIndex(splitterPosition);

				if (!splitter.isInMemory(splitterPosition)) {
					sendEventForAction(NEXT_RESULT_ACTION, splitter
							.getCurrentSplitPOK(), response);
				}
			} else {
				this.logger.debug("Splitter is NULL.");
			}
			this.logger.debug("Update splitter done.");
		}
		logger.info(
				"processAction reponse time :"
						+ Long.toString(System.currentTimeMillis() - start));
	}
	}
	@Override
	public void serveResource(ResourceRequest request, ResourceResponse response)
			throws PortletException, IOException {
		long start = System.currentTimeMillis();
		response.setContentType("text/html");
		HitDescriptionBean hitsDescription;
		boolean error = false;
		String messageError = null;
		ResultSetSplitter splitter = (ResultSetSplitter) request
				.getPortletSession().getAttribute(USER_RESULTSET_SPLITTER,
						PortletSession.APPLICATION_SCOPE);

		if (splitter != null && splitter.getAvaibleSubResourcesNumber() > 0) {
			if (request.getParameter("splitter_position") != null) {
				/*
				 * changing the splitter position, updating user splitter
				 */
				splitter.setCurrentPageIndex(Integer.parseInt(request
						.getParameter("splitter_position")));
			}
			/*
			 * getting description map from splitter
			 */
			hitsDescription = splitter.getCurrentWTripleMap();
			logger.debug("result map : " + hitsDescription);
			
			if (hitsDescription == null || hitsDescription.getDescriptionMap() == null) {
				error=true;
				messageError = ResourceBundle.getBundle("result_portlet", request.getLocale()).getString("resultset.unvalid");
				logger.error("unable to get hits description Map null or empty :"+hitsDescription);
			} else {
				/*
				 * injecting meta-data
				 */
				request.setAttribute(RESULTS_MAP_DESC, hitsDescription.getDescriptionMap());
	
				/*
				 * getting result configuration from session
				 */
				ResultConfBean res_conf = (ResultConfBean) request
						.getPortletSession().getAttribute(USER_META_CONF,
								PortletSession.APPLICATION_SCOPE);
				if (res_conf == null) {
					/*
					 * injecting result config
					 */
					request.setAttribute(USER_META_CONF, DEFAULT_META_CONF_BEAN);
				} else {
					/*
					 * injecting result config
					 */
					request.setAttribute(USER_META_CONF, res_conf);
				}
	
				/*
				 * Injecting current splitter position
				 */
				request.setAttribute("current_split_pos", splitter
						.getCurrentPosition());
	
				/*
				 * creating map of boolean to know if splitter's pages are in
				 * memory
				 */
				Map<Integer, Boolean> isInMemory = new HashMap<Integer, Boolean>();
				
				/*
				 * set min and max position
				 */
				int min_position = splitter.getCurrentPosition()
						- MAX_DISPLAYED_PAGES / 2;
				int max_position = splitter.getCurrentPosition()
						+ MAX_DISPLAYED_PAGES / 2;
	
				if (min_position < 1) {
					min_position = 1;
				}
	
				if (max_position > splitter.getTotalNumberOfResources()) {
					max_position = splitter.getTotalNumberOfResources();
				}
	
				/*
				 * filling map
				 */
				for (int i = min_position; i <= max_position; i++) {
					isInMemory.put(i, splitter.isInMemory(i));
				}
	
				/*
				 * injecting inMemoryPagesMap
				 */
				request.setAttribute("isInMemory", isInMemory);
	
				/*
				 * Injecting current min splitter position
				 */
				request.setAttribute("min_position", min_position);
	
				/*
				 * Injecting current max splitter position
				 */
				request.setAttribute("max_position", max_position);
	
				/*
				 * Injecting max position
				 */
				request.setAttribute("last_splitter_pos", splitter
						.getAvaibleSubResourcesNumber());
	
				if (RANK_DIGIT_NUMBER > 0) { 
					/*
					 * injecting rank bean
					 */
					request.setAttribute("rankBean", ResourceRankFactory
							.getResourceRankBean(hitsDescription.getDescriptionMap(), RANK_DIGIT_NUMBER));
				}
				/*
				 * injecting rdf bean
				 */
				request.setAttribute("rdfProperties", this.rdfProperties.getProperties());
				
				/*
				 * injecting display mode
				 */
				request.setAttribute(DISPLAY_MODE, TITLE_RANK_MODE);
			}
	
			
			
		}
		
		/*
		 * dispatch to jsp
		 */
		if (error) {
			request.setAttribute(ERROR, messageError);
			PortletRequestDispatcher rd = getPortletContext()
					.getRequestDispatcher(
							getInitParameter("error_page_url"));
			rd.include(request, response);
		} else {
			PortletRequestDispatcher rd = getPortletContext().getRequestDispatcher(
					getInitParameter("resources_page_url"));
			rd.include(request, response);
			logger.info(
					"serveResource reponse time :"
							+ Long.toString(System.currentTimeMillis() - start));
		}
	}

	private ResourceContainer getRepoService(URL serviceURL) {

		if (!this.repoMap.containsKey(serviceURL)) {
			/*
			 * web service doesn't exist, create it and add it to service map
			 */
			logger.info(
					"Building repo at [" + serviceURL + "]");
			
			this.repoMap.put(serviceURL, ResourceContainerUtil
					.getResourceContainerService(WSDL_REPO_URL, serviceURL));
		}
		return this.repoMap.get(serviceURL);
	}
	
	private ResultConfBean getMetaBean(ActionRequest request) {
		ResultConfBean ret = new ResultConfBean();
		for (String property : DEFAULT_META_CONF_BEAN.getProperties().keySet()) {
			if (request.getParameter(property)!=null) {
				ret.getProperties().put(property, true);
			} else {
				ret.getProperties().put(property, false);
			}
		}
		return ret;
	}
}
