/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2012 Cassidian, an EADS company
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.portlet;

import java.io.IOException;
import java.util.ResourceBundle;

import javax.portlet.ActionRequest;
import javax.portlet.ActionResponse;
import javax.portlet.EventRequest;
import javax.portlet.EventResponse;
import javax.portlet.PortletConfig;
import javax.portlet.PortletException;
import javax.portlet.PortletMode;
import javax.portlet.PortletRequestDispatcher;
import javax.portlet.PortletSession;
import javax.portlet.ProcessAction;
import javax.portlet.RenderRequest;
import javax.portlet.RenderResponse;
import javax.portlet.WindowState;
import javax.xml.namespace.QName;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.ow2.weblab.components.client.WebLabClient;
import org.ow2.weblab.core.extended.exception.WebLabCheckedException;
import org.ow2.weblab.core.extended.factory.ResourceFactory;
import org.ow2.weblab.core.extended.ontologies.WebLabRetrieval;
import org.ow2.weblab.core.helper.impl.AdvancedSelector;
import org.ow2.weblab.core.helper.impl.JenaResourceHelper;
import org.ow2.weblab.core.helper.impl.RDFSelectorFactory;
import org.ow2.weblab.core.helper.impl.Statements;
import org.ow2.weblab.core.model.ComposedQuery;
import org.ow2.weblab.core.model.Operator;
import org.ow2.weblab.core.model.PieceOfKnowledge;
import org.ow2.weblab.core.model.Query;
import org.ow2.weblab.core.model.StringQuery;
import org.ow2.weblab.core.services.AccessDeniedException;
import org.ow2.weblab.core.services.ContentNotAvailableException;
import org.ow2.weblab.core.services.InsufficientResourcesException;
import org.ow2.weblab.core.services.InvalidParameterException;
import org.ow2.weblab.core.services.Searcher;
import org.ow2.weblab.core.services.ServiceNotConfiguredException;
import org.ow2.weblab.core.services.UnexpectedException;
import org.ow2.weblab.core.services.UnsupportedRequestException;
import org.ow2.weblab.core.services.searcher.SearchArgs;
import org.ow2.weblab.core.services.searcher.SearchReturn;
import org.ow2.weblab.util.PortletQueryParser;
import org.ow2.weblab.util.WebLabQueryParser;

/**
 * Portlet used to generate text requests. When request is submitted, a
 * RequestEvent (request and service definition) is sent. Portlet allow
 * modification by user of the searcher service URL to call.
 * 
 * @author emilien, gdupont
 */
public class SearchPortlet extends WebLabPortlet {


	private static Log logger = LogFactory.getLog(SearchPortlet.class);


	private static String defaultSearcherURI;


	private static int distinct_submit_query;


	// private static boolean useRedirection;

	protected PortletRequestDispatcher normalView;


	protected PortletRequestDispatcher maximizedView;


	protected PortletRequestDispatcher helpView;


	protected PortletRequestDispatcher errorView;


	protected WebLabQueryParser parser;


	@Override
	public void init(final PortletConfig config) throws PortletException {
		super.init(config);
		super.init();

		this.parser = new PortletQueryParser();

		this.normalView = config.getPortletContext().getRequestDispatcher(this.getInitParameter(SearchPortletConstants.NORMAL_PAGE));
		this.maximizedView = config.getPortletContext().getRequestDispatcher(this.getInitParameter(SearchPortletConstants.MAXIMIZED_PAGE));
		this.helpView = config.getPortletContext().getRequestDispatcher(this.getInitParameter(SearchPortletConstants.EDIT_PAGE));
		this.errorView = config.getPortletContext().getRequestDispatcher(this.getInitParameter(SearchPortletConstants.ERROR_PAGE));

		// getting searcher uri
		SearchPortlet.defaultSearcherURI = this.getInitParameter(SearchPortletConstants.SERVICE_URI);

		// Checking the webLabClient library is ready.
		try {
			WebLabClient.getSearcher("default", "init", SearchPortlet.defaultSearcherURI);
			SearchPortlet.logger.info(SearchPortlet.class.getSimpleName() + " is ready to serve request my Lord.");
		} catch (final WebLabCheckedException e) {
			throw new PortletException("Cannot access the searcher from WebLab client.", e);
		}

		SearchPortlet.logger.info("Default searcher service uri [" + SearchPortlet.defaultSearcherURI + "].");

		// setting the redirect parameter
		// if (getInitParameter("use_redirection") != null) {
		// useRedirection =
		// Boolean.parseBoolean(getInitParameter("use_redirection"));
		// } else {
		// useRedirection = false;
		// }
		SearchPortlet.distinct_submit_query = 0;
	}


	@Override
	public void destroy() {
		super.destroy();
	}


	/**
	 * Render HTML corresponding to the view mode
	 */
	@Override
	public void doView(final RenderRequest req, final RenderResponse res) throws IOException, PortletException {
		final long start = System.currentTimeMillis();

		// changing portlet title
		res.setTitle(ResourceBundle.getBundle(SearchPortletConstants.BUNDLE_NAME, req.getLocale()).getString("portlet.title"));

		if (req.getPortletSession().getAttribute(SearchPortletConstants.USER_SERVICE_CONF) == null) {
			SearchPortlet.logger.info("Setting default searcher uri [" + SearchPortlet.defaultSearcherURI + "] in session");
			req.getPortletSession().setAttribute(SearchPortletConstants.USER_SERVICE_CONF, SearchPortlet.defaultSearcherURI);
		}

		if (req.getParameter(SearchPortletConstants.SEARCH_ERROR) != null) {
			// dispatch to jsp
			req.setAttribute(SearchPortletConstants.SEARCH_ERROR, req.getParameter(SearchPortletConstants.SEARCH_ERROR));
			SearchPortlet.logger.info("error in searcher");
			this.errorView.include(req, res);
		}

		if (req.getPortletSession().getAttribute(SearchPortletConstants.USER_LAST_SEARCH, PortletSession.APPLICATION_SCOPE) == null) {
			req.setAttribute(SearchPortletConstants.USER_LAST_SEARCH,
					ResourceBundle.getBundle(SearchPortletConstants.BUNDLE_NAME, req.getLocale()).getString("search.defaultText"));
			req.setAttribute(SearchPortletConstants.DEFAULT_SEARCH, Boolean.TRUE);
		} else {
			final Query query = ((SearchArgs) req.getPortletSession().getAttribute(SearchPortletConstants.USER_LAST_SEARCH, PortletSession.APPLICATION_SCOPE))
					.getQuery();
			try {
				req.setAttribute(SearchPortletConstants.USER_LAST_SEARCH, this.parser.getRequest(query, new JenaResourceHelper(query)));
			} catch (final WebLabCheckedException e) {
				SearchPortlet.logger.info("Cannot convert query to String: " + e.getMessage(), e);
				this.errorView.include(req, res);
			}
			req.setAttribute(SearchPortletConstants.DEFAULT_SEARCH, Boolean.FALSE);
		}
		// dispatch to jsp
		if (req.getWindowState() == WindowState.NORMAL) {
			this.normalView.include(req, res);
		} else if (req.getWindowState() == WindowState.MAXIMIZED) {
			this.maximizedView.include(req, res);
		}
		SearchPortlet.logger.info("doView reponse time :" + Long.toString(System.currentTimeMillis() - start));
	}


	@Override
	public void doEdit(final RenderRequest req, final RenderResponse res) throws PortletException, IOException {
		final long start = System.currentTimeMillis();

		res.setContentType(SearchPortletConstants.CONTENT_TYPE);
		req.setAttribute(SearchPortletConstants.SERVICE_URI, req.getPortletSession().getAttribute(SearchPortletConstants.USER_SERVICE_CONF));

		// dispatch to jsp
		this.helpView.include(req, res);
		SearchPortlet.logger.info("doEdit reponse time :" + Long.toString(System.currentTimeMillis() - start));
	}


	@ProcessAction(name = SearchPortletConstants.ACTION_DO_SEARCH)
	public void doSearch(final ActionRequest req, final ActionResponse res) throws PortletException, IOException {
		SearchPortlet.logger.debug("An action has been received [" + SearchPortletConstants.ACTION_DO_SEARCH + "].");

		if ((req.getParameter(SearchPortletConstants.QUERY_INPUT) == null) || (req.getParameter(SearchPortletConstants.QUERY_INPUT) == "")) {
			throw new PortletException("An error occured while launching query : cannot get query inputs.");
		}

		// create query
		final StringQuery q = ResourceFactory.createResource(this.getDefaultNamespace() + this.getPortletName(), "queryID_"
				+ SearchPortlet.distinct_submit_query++, StringQuery.class);
		q.setRequest(req.getParameter(SearchPortletConstants.QUERY_INPUT));

		// TODO add query language in annotation ?
		final SearchArgs args = this.createSearchArgs(q);

		// launch event to broadcast the searchArgs to all portlets
		// (searchportlet should catch the event itself)
		final QName qnOfEvent = QName.valueOf(this.getPortletConfig().getInitParameter(
				WebLabPortlet.PORTLET_ACTION_NAMESPACE + SearchPortletConstants.ACTION_PUBLISH_QUERY));
		SearchPortlet.logger.info("publishing searchArgs in event : [" + qnOfEvent + "]");
		res.setEvent(qnOfEvent, args);
	}


	@ProcessAction(name = SearchPortletConstants.ACTION_DO_ADV_SEARCH)
	public void doAdvSearch(final ActionRequest req, final ActionResponse res) throws PortletException, IOException {
		SearchPortlet.logger.debug("An expected action has been received [" + SearchPortletConstants.ACTION_DO_ADV_SEARCH + "].");
		if ((req.getParameter(SearchPortletConstants.SEARCH_AND_KEYWORDS) == null) || (req.getParameter(SearchPortletConstants.SEARCH_PHRASE_KEYWORDS) == null)
				|| (req.getParameter(SearchPortletConstants.SEARCH_OR_KEYWORDS) == null)
				|| (req.getParameter(SearchPortletConstants.SEARCH_NOT_KEYWORDS) == null)) {
			SearchPortlet.logger.error("An error occured while launching query : cannot get query inputs.");
		} else {
			// create query
			final ComposedQuery metaQuery = this.createAdvQuery(req);

			// TODO add query language in annotation ?
			final SearchArgs args = this.createSearchArgs(metaQuery);

			// launch event to broadcast the searchArgs to all portlets
			// (searchportlet should catch the event itself)
			final QName qnOfEvent = QName.valueOf(this.getPortletConfig().getInitParameter(
					WebLabPortlet.PORTLET_ACTION_NAMESPACE + SearchPortletConstants.ACTION_PUBLISH_QUERY));
			res.setEvent(qnOfEvent, args);
		}
		res.setWindowState(WindowState.NORMAL);
	}


	@ProcessAction(name = SearchPortletConstants.ACTION_GET_ADV_SEARCH)
	public void getAdvSearch(final ActionRequest req, final ActionResponse res) throws PortletException, IOException {
		SearchPortlet.logger.debug("An action has been received [" + SearchPortletConstants.ACTION_GET_ADV_SEARCH + "].");
		res.setWindowState(WindowState.MAXIMIZED);
	}


	@ProcessAction(name = SearchPortletConstants.ACTION_CHANGE_SERVICE_CONFIG)
	public void changeService(final ActionRequest req, final ActionResponse res) throws PortletException, IOException {
		SearchPortlet.logger.debug("An expected action has been received [" + SearchPortletConstants.ACTION_CHANGE_SERVICE_CONFIG + "].");
		final String searcherURI = req.getParameter(SearchPortletConstants.SERVICE_URI_INPUT);

		// Checking the uri validity and init client if necessary.
		try {
			WebLabClient.getSearcher("default", "init", searcherURI);
			SearchPortlet.logger.info(SearchPortlet.class.getSimpleName() + " is ready to serve request my Lord.");
		} catch (final WebLabCheckedException e) {
			SearchPortlet.logger.error("Searcher URI is invalid.", e);
			req.setAttribute(SearchPortletConstants.SEARCH_ERROR, "Searcher URI is invalid.");
			this.errorView.include(req, res);
		}

		// TODO enable a list of services ?

		SearchPortlet.logger.info("New searcher service uri :" + searcherURI);
		// configuration is valid, setting it in user session
		req.getPortletSession().setAttribute(SearchPortletConstants.USER_SERVICE_CONF, searcherURI);
		res.setPortletMode(PortletMode.VIEW);
	}


	@Override
	public void processAction(final ActionRequest req, final ActionResponse res) throws IOException, PortletException {
		SearchPortlet.logger.info("Action receveid : [" + req.getParameter(ActionRequest.ACTION_NAME) + "].");

		super.processAction(req, res);
	}


	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * javax.portlet.GenericPortlet#processEvent(javax.portlet.EventRequest,
	 * javax.portlet.EventResponse)
	 */
	@Override
	public void processEvent(final EventRequest req, final EventResponse res) throws PortletException, IOException {
		final long start = System.currentTimeMillis();
		final String reaction = this.getReaction(req.getEvent().getQName()).getLocalPart();

		SearchPortlet.logger.info("Event received " + req.getEvent().getQName() + " - launching reaction : " + reaction);

		if (SearchPortletConstants.REACTION_QUERY.equals(reaction)) {
			if ((req.getEvent().getValue() == null) || !(req.getEvent().getValue() instanceof SearchArgs)) {
				SearchPortlet.logger.error("The Event passed has no (or incorrect) value: [" + req.getEvent().getValue() + "]");
				res.setRenderParameter(SearchPortletConstants.SEARCH_ERROR, "The Event passed has no (or incorrect) value: [" + req.getEvent().getValue() + "]");
			}
			final SearchArgs args = (SearchArgs) req.getEvent().getValue();

			// TODO fix the selection of user URI format and context
			final String userURI = req.getRemoteUser();
			final String context = "defaultContext";
			String searcherURI = null;
			if (req.getPortletSession().getAttribute(SearchPortletConstants.USER_SERVICE_CONF) != null) {
				searcherURI = req.getPortletSession().getAttribute(SearchPortletConstants.USER_SERVICE_CONF).toString();
			}

			try {
				final SearchReturn searchResult = this.doSearch(userURI, context, searcherURI, args);
				// setting Query in user session
				req.getPortletSession().setAttribute(SearchPortletConstants.USER_LAST_SEARCH, args, PortletSession.APPLICATION_SCOPE);
				// dispatching results
				this.sendEventForAction(SearchPortletConstants.SEARCH_ACTION, searchResult.getResultSet(), res);
			} catch (final WebLabCheckedException e) {
				final String error = "Search service is not accessible.";
				SearchPortlet.logger.error(error, e);
				res.setRenderParameter(SearchPortletConstants.SEARCH_ERROR, error);
			}

		} else if (SearchPortletConstants.REACTION_NEXT_DOCUMENTS.equals(reaction)) {
			// checking if the portlet is the query producer
			if ((req.getEvent().getValue() != null) && (req.getEvent().getValue() instanceof PieceOfKnowledge)) {
				final PieceOfKnowledge pok = (PieceOfKnowledge) req.getEvent().getValue();
				final AdvancedSelector rdfSelector = RDFSelectorFactory.getSelector(true);
				final Statements map = rdfSelector.searchFor(pok);
				final String queryURI = map.getFirstValue(null, WebLabRetrieval.HAS_ORDERED_QUERY, null);

				SearchPortlet.logger.info("query URI :" + queryURI);

				final SearchArgs userSearchArg = (SearchArgs) req.getPortletSession().getAttribute(SearchPortletConstants.USER_LAST_SEARCH,
						PortletSession.APPLICATION_SCOPE);

				if ((userSearchArg != null) && userSearchArg.getQuery().getUri().equals(queryURI)) {
					// this portlet has generated the query, redo search new
					// offset and limit
					userSearchArg.setOffset(map.getTypedValue(userSearchArg.getQuery().getUri(), WebLabRetrieval.HAS_EXPECTED_OFFSET, Integer.class));
					userSearchArg.setLimit(map.getTypedValue(userSearchArg.getQuery().getUri(), WebLabRetrieval.HAS_EXPECTED_LIMIT, Integer.class));

					// TODO fix the selection of user URI format and context
					final String userURI = req.getRemoteUser();
					final String context = "defaultContext";
					String searcherURI = null;
					if (req.getPortletSession().getAttribute(SearchPortletConstants.USER_SERVICE_CONF) != null) {
						searcherURI = req.getPortletSession().getAttribute(SearchPortletConstants.USER_SERVICE_CONF).toString();
					}

					SearchPortlet.logger.info("searcher : updating a resultSet, offset:" + userSearchArg.getOffset() + " limit:" + userSearchArg.getLimit());
					// process query to the searcher
					try {
						final SearchReturn searchResult = this.doSearch(userURI, context, searcherURI, userSearchArg);
						this.sendEventForAction(SearchPortletConstants.SEARCH_ACTION, searchResult.getResultSet(), res);
					} catch (final WebLabCheckedException e) {
						SearchPortlet.logger.error("An error occured while querying the search service.", e);
						res.setRenderParameter(SearchPortletConstants.SEARCH_ERROR, "An error occured while querying the search service.");
					}
				}
			}
		} else {
			SearchPortlet.logger.error("An unknown event has been receveid [" + req.getEvent().getQName() + "].");
			res.setRenderParameter(SearchPortletConstants.SEARCH_ERROR, "An unknown event has been receveid [" + req.getEvent().getQName() + "].");
		}
		SearchPortlet.logger.info("processEvent reponse time :" + Long.toString(System.currentTimeMillis() - start));
	}


	protected ComposedQuery createAdvQuery(final ActionRequest req) {
		final ComposedQuery metaQuery = ResourceFactory.createResource(this.getDefaultNamespace() + this.getPortletName(), "queryID_"
				+ SearchPortlet.distinct_submit_query++, ComposedQuery.class);
		metaQuery.setOperator(Operator.AND);

		final String and_keywords = req.getParameter(SearchPortletConstants.SEARCH_AND_KEYWORDS).toString().trim();
		if (and_keywords != "") {
			final StringQuery q = ResourceFactory.createResource(this.getDefaultNamespace() + this.getPortletName(), "queryID_"
					+ SearchPortlet.distinct_submit_query++, StringQuery.class);
			q.setRequest(and_keywords.replaceAll(" ", " " + Operator.AND + " "));
			metaQuery.getQuery().add(q);
		}
		final String phrase_keywords = req.getParameter(SearchPortletConstants.SEARCH_PHRASE_KEYWORDS).toString().trim();
		if (phrase_keywords != "") {
			final StringQuery q = ResourceFactory.createResource(this.getDefaultNamespace() + this.getPortletName(), "queryID_"
					+ SearchPortlet.distinct_submit_query++, StringQuery.class);
			q.setRequest('"' + phrase_keywords + '"');
			metaQuery.getQuery().add(q);
		}
		final String or_keywords = req.getParameter(SearchPortletConstants.SEARCH_OR_KEYWORDS).toString().trim();
		if (or_keywords != "") {
			final StringQuery q = ResourceFactory.createResource(this.getDefaultNamespace() + this.getPortletName(), "queryID_"
					+ SearchPortlet.distinct_submit_query++, StringQuery.class);
			q.setRequest('(' + or_keywords.replaceAll(" ", " " + Operator.OR + " ") + ')');
			metaQuery.getQuery().add(q);
		}
		final String not_keywords = req.getParameter(SearchPortletConstants.SEARCH_NOT_KEYWORDS).toString().trim();
		if (not_keywords != "") {
			final StringQuery q = ResourceFactory.createResource(this.getDefaultNamespace() + this.getPortletName(), "queryID_"
					+ SearchPortlet.distinct_submit_query++, StringQuery.class);
			q.setRequest("" + Operator.NOT + ' ' + "(" + not_keywords.replaceAll(" ", " " + Operator.OR + " ") + ')');
			metaQuery.getQuery().add(q);
		}
		return metaQuery;
	}


	protected SearchArgs createSearchArgs(final Query q) {

		try {
			SearchPortlet.logger.info("Creating search args for [" + this.parser.getRequest(q, new JenaResourceHelper(q)) + "].");
		} catch (final WebLabCheckedException e) {
			SearchPortlet.logger.warn("Cannot convert query " + q + " to String. May not be a real issue.");
		}

		// create args for search service
		final SearchArgs args = new SearchArgs();
		args.setLimit(Integer.valueOf(10));
		args.setOffset(Integer.valueOf(0));
		args.setQuery(q);

		return args;
	}


	protected SearchReturn doSearch(final String userURI, final String context, final String searcherURI, final SearchArgs args) throws WebLabCheckedException {
		SearchPortlet.logger.debug("Launching search query...");
		// TODO design correct message handler
		final String error_message = "An error occured during search.";

		SearchPortlet.logger.info("Searching with user [" + userURI + "] and context [" + context + "] on service [" + searcherURI + "].");

		try {
			final Searcher client = WebLabClient.getSearcher(userURI, context, searcherURI);
			if (client != null) {
				final SearchReturn searchResult = client.search(args);
				if (searchResult == null) {
					throw new WebLabCheckedException(error_message + " Search results are null.");
				}
				SearchPortlet.logger.debug(searchResult.getResultSet());
				SearchPortlet.logger.info("Results received for search [" + args + "].");
				return searchResult;
			}
			throw new WebLabCheckedException(error_message + " Search client is invalid.");
		} catch (final UnsupportedRequestException e) {
			throw new WebLabCheckedException(error_message + " [" + e.getClass().getSimpleName() + "].", e);
		} catch (final ServiceNotConfiguredException e) {
			throw new WebLabCheckedException(error_message + " [" + e.getClass().getSimpleName() + "].", e);
		} catch (final AccessDeniedException e) {
			throw new WebLabCheckedException(error_message + " [" + e.getClass().getSimpleName() + "].", e);
		} catch (final ContentNotAvailableException e) {
			throw new WebLabCheckedException(error_message + " [" + e.getClass().getSimpleName() + "].", e);
		} catch (final InvalidParameterException e) {
			throw new WebLabCheckedException(error_message + " [" + e.getClass().getSimpleName() + "].", e);
		} catch (final InsufficientResourcesException e) {
			throw new WebLabCheckedException(error_message + " [" + e.getClass().getSimpleName() + "].", e);
		} catch (final UnexpectedException e) {
			throw new WebLabCheckedException(error_message + " [" + e.getClass().getSimpleName() + "].", e);
		} catch (final Exception e) {
			e.printStackTrace();
			throw new WebLabCheckedException(error_message + " [" + e.getClass().getSimpleName() + "]", e);
		}
	}
}
