/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2012 Cassidian, an EADS company
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */
package org.ow2.weblab.util;

import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.ow2.weblab.core.extended.exception.WebLabCheckedException;
import org.ow2.weblab.core.helper.ResourceHelper;
import org.ow2.weblab.core.model.ComposedQuery;
import org.ow2.weblab.core.model.Query;
import org.ow2.weblab.core.model.Resource;
import org.ow2.weblab.core.model.SimilarityQuery;
import org.ow2.weblab.core.model.StringQuery;

public class PortletQueryParser extends WebLabQueryParser {


	private final Log logger = LogFactory.getLog(PortletQueryParser.class);


	@Override
	public String getRequest(final Query q, final ResourceHelper hlpr) throws WebLabCheckedException {
		if (q instanceof StringQuery) {
			return ((StringQuery) q).getRequest();
		} else if (q instanceof ComposedQuery) {
			return this.getComposedRequest((ComposedQuery) q, hlpr);
		} else if (q instanceof SimilarityQuery) {
			final SimilarityQuery query = (SimilarityQuery) q;
			final StringBuffer queryString = new StringBuffer();
			final List<Resource> resources = query.getResource();

			if (resources.size() == 1) {
				// single sample => can use more like this function
				queryString.append("id");
				queryString.append(':');
				queryString.append('"');
				queryString.append(resources.get(0).getUri());
				queryString.append('"');
			} else {
				// multiple samples, thus try simple ID based query
				// TODO check this is the expected behavior
				for (int i = 0; i < resources.size(); i++) {
					final Resource r = resources.get(i);
					if (i > 0) {
						queryString.append(" OR ");
					}
					queryString.append("id");
					queryString.append(':');
					queryString.append('"');
					queryString.append(r.getUri());
					queryString.append('"');
				}
			}
			return queryString.toString();
		}
		throw new WebLabCheckedException("Cannot parse query type : " + q.getClass() + ".");
	}


	@Override
	public String getComposedRequest(final ComposedQuery q, final ResourceHelper hlpr) throws WebLabCheckedException {
		// Watch out : this supposes that WebLab Operator are following Lucene
		// operator syntax...
		final String operator = q.getOperator().value();

		final StringBuffer queryString = new StringBuffer();
		queryString.append('(');
		for (final Query subQ : q.getQuery()) {
			if (queryString.length() > 1) {
				queryString.append(' ');
				queryString.append(operator);
				queryString.append(' ');
			}
			if (subQ instanceof StringQuery) {
				queryString.append(this.getRequestWithScope(subQ, hlpr));
			} else if (subQ instanceof ComposedQuery) {
				queryString.append(this.getComposedRequest((ComposedQuery) subQ, hlpr));
			} else {
				throw new WebLabCheckedException("Solr engine service can only process " + ComposedQuery.class.getSimpleName() + " or "
						+ StringQuery.class.getSimpleName() + ".");
			}
		}
		queryString.append(')');
		return queryString.toString();
	}


	@Override
	public String getRequestWithScope(final Query q, final ResourceHelper hlpr) throws WebLabCheckedException {
		final List<String> values = hlpr.getRessOnPredSubj(q.getUri(), WebLabQueryParser.HAS_SCOPE);
		if (values.size() == 0) {
			this.logger.debug("No scope.");
		} else if (values.size() > 1) {
			this.logger.info("Query is holding multiple values for [" + WebLabQueryParser.HAS_SCOPE + "]. That's not cool, so we ignore all of them.");
		} else {
			String scope = values.get(0);
			if (scope.contains("#")) {
				scope = scope.substring(scope.indexOf('#') + 1).trim().toLowerCase();
			} else {
				scope = scope.substring(scope.lastIndexOf('/') + 1).trim().toLowerCase();
			}
			return scope + ":(" + this.getRequest(q, hlpr) + ")";
		}
		return this.getRequest(q, hlpr);
	}


	@Override
	public String getOrderBy(final Query q, final ResourceHelper hlpr) throws WebLabCheckedException {
		// List<String> values = hlpr.getRessOnPredSubj(q.getUri(),
		// TO_BE_ORDERED_BY);
		// if (values.size() != 1) {
		// logger.info("Query is holding multiple or no values for [" +
		// TO_BE_ORDERED_BY + "]. That's not funky, so we ignore all of them.");
		// } else {
		// if (config != null) {
		// return config.getPropertyToFieldMap().get(values.get(0)).getName();
		// }
		// }
		// return null;
		throw new WebLabCheckedException("Not yet implemented...");

	}


	@Override
	public boolean getOrder(final Query q, final ResourceHelper hlpr) throws WebLabCheckedException {
		final List<String> values = hlpr.getLitsOnPredSubj(q.getUri(), WebLabQueryParser.ASCENDENT_ORDERING_MODE_EXPECTED);
		if (values.size() != 1) {
			this.logger.info("Query is holding multiple values for [" + WebLabQueryParser.ASCENDENT_ORDERING_MODE_EXPECTED
					+ "]. That's not well perceived, so we ignore all of them.");
		} else {
			return Boolean.parseBoolean(values.get(0));
		}
		return false;
	}

}
